/* glpmip/branch_last.c */

/*----------------------------------------------------------------------
-- This file is a part of the GLPK package.
--
-- Copyright (C) 2000, 2001 Andrew Makhorin <mao@mai2.rcnet.ru>,
--                          Department for Applied Informatics,
--                          Moscow Aviation Institute, Moscow, Russia.
--                          All rights reserved.
--
-- This code is free software; you can redistribute it and/or modify
-- it under the terms of the GNU General Public License as published by
-- the Free Software Foundation; either version 2 of the License, or
-- (at your option) any later version.
--
-- This software is distributed "as is" in the hope that it will be
-- useful, but WITHOUT ANY WARRANTY; without even the implied warranty
-- of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
-- General Public License for more details.
--
-- You should have received a copy of the GNU General Public License
-- along with this program; if not, write to the Free Software
-- Foundation, 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA.
----------------------------------------------------------------------*/

#include <math.h>
#include "glpmip.h"

/*----------------------------------------------------------------------
-- branch_last - choose the last appropriate branching variable.
--
-- *Synopsis*
--
-- #include "glpmip.h"
-- int branch_last(BBDATA *bb, int *what);
--
-- *Description*
--
-- The branch_last routine chooses a branching variable which is an
-- integer structural variable xS[j] (1 <= j <= n).
--
-- The routine also sets the parameter what which tells the solver what
-- of two subproblems should be solved after the current problem:
--
-- 'L' - that one where xS[j] >= new lower bound = ceil(beta)  > beta
-- 'U' - that one where xS[j] <= new upper bound = floor(beta) < beta
--
-- where beta is the (fractional) value of the variable xS[j] in basis
-- solution of the current problem.
--
-- *Returns*
--
-- The branch_last routine returns a number of the variable xS[j].
--
-- *Heuristic*
--
-- The branch_last routine looks up the list of structural variables in
-- reverse order and the chooses the last variable (if to see from the
-- beginning of the list) which is of integer kind and (being basic) has
-- fractional (non-integer) value in the current basis solution. */

int branch_last(BBDATA *bb, int *what)
{     int m = bb->m, n = bb->n, i, j, k, this = 0;
      double beta;
      for (j = n; j >= 1; j--)
      {  /* skip continuous variable */
         if (!bb->mip->kind[j]) continue;
         /* x[k] = xS[j] */
         k = m + j;
         /* skip non-basic variable */
         if (bb->rsm->posx[k] < 0) continue;
         /* xB[i] = x[k] */
         i = +bb->rsm->posx[k];
         insist(1 <= i && i <= m);
         /* beta = value of xS[j] in the current basis solution */
         beta = bb->bbar[i];
         /* skip basic variable which has an integer value */
         if (beta == floor(beta)) continue;
         /* choose xS[j] */
         this = j;
         break;
      }
      insist(1 <= this && this <= n);
      /* tell the solver to get next that problem where xS[j] violates
         its new bound less than in the other */
      if (ceil(beta) - beta < beta - floor(beta))
         *what = 'L';
      else
         *what = 'U';
      return this;
}

/* eof */
