/* glprsm/rsm_set_cbar.c */

/*----------------------------------------------------------------------
-- This file is a part of the GLPK package.
--
-- Copyright (C) 2000, 2001 Andrew Makhorin <mao@mai2.rcnet.ru>,
--                          Department for Applied Informatics,
--                          Moscow Aviation Institute, Moscow, Russia.
--                          All rights reserved.
--
-- This code is free software; you can redistribute it and/or modify
-- it under the terms of the GNU General Public License as published by
-- the Free Software Foundation; either version 2 of the License, or
-- (at your option) any later version.
--
-- This software is distributed "as is" in the hope that it will be
-- useful, but WITHOUT ANY WARRANTY; without even the implied warranty
-- of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
-- General Public License for more details.
--
-- You should have received a copy of the GNU General Public License
-- along with this program; if not, write to the Free Software
-- Foundation, 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA.
----------------------------------------------------------------------*/

#include <stddef.h>
#include "glprsm.h"

/*----------------------------------------------------------------------
-- rsm_set_cbar - compute reduced costs of non-basis variables.
--
-- *Synopsis*
--
-- #include "glprsm.h"
-- void rsm_set_cbar(RSM *rsm);
--
-- *Description*
--
-- The rsm_set_cbar routine computes the reduced costs of all non-basis
-- variables and stores these costs to the array cbar. This routine
-- assumes that objective function coefficients were stored before to
-- the array cbar and simplex multipliers were computed before by the
-- rsm_set_pi routine.
--
-- The rsm_set_cbar routine uses the following formula:
--
--    cbar[j] = cN[j] - pi' * N[j], j = 1, 2, ..., n,
--
-- where cbar[j] is the reduced cost of the non-basis variable xN[j],
-- cN[j] is the objective function coefficient at the variable xN[j],
-- pi is the vector of simplex multipliers (pi' denotes transposition,
-- i.e. pi' is a row), N[j] is the column of the expanded constraint
-- matrix A~, which corresponds to the variable xN[j]. */

void rsm_set_cbar(RSM *rsm)
{     ELEM *e;
      int n = rsm->n, j, k;
      /* cbar[j] = d[j] = cN[j] - pi * N[j] */
      for (j = 1; j <= n; j++)
      {  k = rsm->indn[j]; /* x[k] = xN[j] */
         rsm->cbar[j] = rsm->cost[k];
         for (e = rsm->A->col[k]; e != NULL; e = e->col)
            rsm->cbar[j] -= rsm->pi[e->i] * e->val;
      }
      return;
}

/* eof */
