/* glpapi/glp_write_mps.c */

/*----------------------------------------------------------------------
-- This file is a part of the GLPK package.
--
-- Copyright (C) 2000, 2001 Andrew Makhorin <mao@mai2.rcnet.ru>,
--                          Department for Applied Informatics,
--                          Moscow Aviation Institute, Moscow, Russia.
--                          All rights reserved.
--
-- This code is free software; you can redistribute it and/or modify
-- it under the terms of the GNU General Public License as published by
-- the Free Software Foundation; either version 2 of the License, or
-- (at your option) any later version.
--
-- This software is distributed "as is" in the hope that it will be
-- useful, but WITHOUT ANY WARRANTY; without even the implied warranty
-- of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
-- General Public License for more details.
--
-- You should have received a copy of the GNU General Public License
-- along with this program; if not, write to the Free Software
-- Foundation, 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA.
----------------------------------------------------------------------*/

#include <errno.h>
#include <math.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include "glpk.h"
#include "glpavl.h"
#include "glpset.h"

/*----------------------------------------------------------------------
-- glp_write_mps - write problem data using MPS format.
--
-- *Synopsis*
--
-- #include "glpk.h"
-- int glp_write_mps(char *fname);
--
-- *Description*
--
-- The glp_write_routine writes LP problem data using MPS format from
-- the workspace to the output text file, whose name is the character
-- string fname.
--
-- Should note that the MPS format allows row and column names up to 8
-- characters. So, if some rows or columns presented in the workspace
-- contain more 8 characters, the routine will reduce such names.
-- However, the routine make all reduced names be unique.
--
-- Detailed description of the MPS format can be found, for example,
-- in the following book:
--
-- B.A.Murtagh. Advanced Linear Programming: Computation and Practice.
-- McGraw-Hill, 1981.
--
-- *Control parameters*
--
-- The behavior of the glp_write_mps routine depends on the following
-- control parameters:
--
-- mps_use_names (reduce long names using original names);
-- mps_one_entry (don't use fields 5 and 6);
-- mps_pedantic (always put column and vector names to the field 2);
-- mps_skip_empty (skip empty columns).
--
-- *Returns*
--
-- 0 - no errors;
-- 1 - the operation failed because of errors. All diagnostics was sent
--     to stderr. */

typedef char mps_name[8+1];
/* standard MPS names (can contain up to 8 characters) */

static void make_names(int what, mps_name alias[]);
/* generate standard MPS names using original names */

static char *mps_number(double val);
/* convert number to standard 12-character MPS format */

int glp_write_mps(char *fname)
{     FILE *fp;
      mps_name *row_name = NULL, *col_name = NULL;
      int use_names, one_entry, pedantic, skip_empty;
      int nrows, ncols, ret, flag;
      print("glp_write_mps: writing problem data to `%s'...", fname);
      glp_get_ipar("mps_use_names", &use_names);
      glp_get_ipar("mps_one_entry", &one_entry);
      glp_get_ipar("mps_pedantic", &pedantic);
      glp_get_ipar("mps_skip_empty", &skip_empty);
      /* open the output text file */
      fp = fopen(fname, "w");
      if (fp == NULL)
      {  error("glp_write_mps: can't create `%s' - %s", fname,
            strerror(errno));
         goto fail;
      }
      /* count and renumber rows and columns (their sequential numbers
         are placed in the seqn fields) */
      nrows = 0;
      for (ret = glp_first_item(GLP_ROW); ret == 0;
           ret = glp_next_item(GLP_ROW))
         glp_set_seqn(GLP_ROW, ++nrows);
      ncols = 0;
      for (ret = glp_first_item(GLP_COL); ret == 0;
           ret = glp_next_item(GLP_COL))
         glp_set_seqn(GLP_COL, ++ncols);
      /* the problem should contain at least one row and one column */
      if (nrows == 0)
      {  error("glp_write_mps: problem has no rows");
         goto fail;
      }
      if (ncols == 0)
      {  error("glp_write_mps: problem has no columns");
         goto fail;
      }
      /* allocate arrays for 8-character row and column names */
      row_name = ucalloc(1+nrows, sizeof(mps_name));
      col_name = ucalloc(1+ncols, sizeof(mps_name));
      /* generate 8-character row and column names (the 8-character
         name of the row or the column having number seqn is placed to
         row_name[seqn] or col_name[seqn] respectively) */
      if (use_names)
      {  /* use original row and column names as templates */
         make_names(GLP_ROW, row_name);
         make_names(GLP_COL, col_name);
      }
      else
      {  /* make plain names based on sequential numbers */
         int k;
         char *t;
         for (k = 1; k <= nrows; k++)
         {  sprintf(row_name[k], "R%7d", k);
            for (t = row_name[k]; *t; t++) if (*t == ' ') *t = '_';
         }
         for (k = 1; k <= ncols; k++)
         {  sprintf(col_name[k], "C%7d", k);
            for (t = col_name[k]; *t; t++) if (*t == ' ') *t = '_';
         }
      }
      /* write NAME indicator card */
      {  char name[GLP_MAX_NAME+1];
         glp_get_cpar("problem", name), name[8] = '\0';
         fprintf(fp, "NAME          %s\n", name);
      }
      /* write ROWS section */
      fprintf(fp, "ROWS\n");
      for (ret = glp_first_item(GLP_ROW); ret == 0;
           ret = glp_next_item(GLP_ROW))
      {  int seqn, type;
         glp_get_seqn(GLP_ROW, &seqn);
         glp_get_bounds(GLP_ROW, &type, NULL, NULL);
         switch (type)
         {  case GLP_FR: type = 'N'; break;
            case GLP_LO: type = 'G'; break;
            case GLP_UP: type = 'L'; break;
            case GLP_DB: type = 'E'; break;
            case GLP_FX: type = 'E'; break;
            default: insist(type != type);
         }
         fprintf(fp, " %c  %s\n", type, row_name[seqn]);
      }
      /* write COLUMNS section */
      fprintf(fp, "COLUMNS\n");
      for (ret = glp_first_item(GLP_COL); ret == 0;
           ret = glp_next_item(GLP_COL))
      {  int seqn, nl = 1, ret;
         char *name;
         glp_get_seqn(GLP_COL, &seqn);
         name = col_name[seqn];
         if (glp_first_coef(GLP_COL) != 0 && !skip_empty)
            fprintf(fp, "    %-8s  %-8s  %12s   $ empty column\n",
               name, row_name[1], mps_number(0.0));
         for (ret = glp_first_coef(GLP_COL); ret == 0;
              ret = glp_next_coef(GLP_COL))
         {  int seqn;
            double coef;
            if (nl)
               fprintf(fp, "    %-8s  ", name);
            else
               fprintf(fp, "   ");
            glp_get_seqn(GLP_ROW, &seqn);
            glp_get_coef(&coef);
            fprintf(fp, "%-8s  %12s", row_name[seqn], mps_number(coef));
            if (!one_entry) nl = 1 - nl;
            if (nl) fprintf(fp, "\n");
            if (!pedantic) name = "";
         }
         if (!nl) fprintf(fp, "\n");
      }
      /* write RHS section */
      flag = 0;
      {  int nl = 1, ret;
         char *name = (pedantic ? "RHS1" : "");
         for (ret = glp_first_item(GLP_ROW); ret == 0;
              ret = glp_next_item(GLP_ROW))
         {  int seqn, type;
            double lb, ub, rhs;
            glp_get_seqn(GLP_ROW, &seqn);
            glp_get_bounds(GLP_ROW, &type, &lb, &ub);
            switch (type)
            {  case GLP_FR: rhs = 0.0; break;
               case GLP_LO: rhs = lb;  break;
               case GLP_UP: rhs = ub;  break;
               case GLP_DB: rhs = (ub > 0.0 ? lb : ub); break;
               case GLP_FX: rhs = lb;  break;
               default: insist(type != type);
            }
            if (rhs == 0.0) continue;
            if (!flag) fprintf(fp, "RHS\n"), flag = 1;
            if (nl)
                fprintf(fp, "    %-8s  ", name);
            else
                fprintf(fp, "   ");
            fprintf(fp, "%-8s  %12s", row_name[seqn], mps_number(rhs));
            if (!one_entry) nl = 1 - nl;
            if (nl) fprintf(fp, "\n");
         }
         if (!nl) fprintf(fp, "\n");
      }
      /* write RANGES section */
      flag = 0;
      {  int nl = 1, ret;
         char *name = (pedantic ? "RNG1" : "");
         for (ret = glp_first_item(GLP_ROW); ret == 0;
              ret = glp_next_item(GLP_ROW))
         {  int seqn, type;
            double lb, ub, rng;
            glp_get_seqn(GLP_ROW, &seqn);
            glp_get_bounds(GLP_ROW, &type, &lb, &ub);
            if (type != GLP_DB) continue;
            if (!flag) fprintf(fp, "RANGES\n"), flag = 1;
            if (nl)
                fprintf(fp, "    %-8s  ", name);
            else
                fprintf(fp, "   ");
            rng = (ub > 0.0 ? ub - lb : lb - ub);
            fprintf(fp, "%-8s  %12s", row_name[seqn], mps_number(rng));
            if (!one_entry) nl = 1 - nl;
            if (nl) fprintf(fp, "\n");
         }
         if (!nl) fprintf(fp, "\n");
      }
      /* write BOUNDS section */
      flag = 0;
      {  int ret;
         char *name = (pedantic ? "BND1" : "");
         for (ret = glp_first_item(GLP_COL); ret == 0;
              ret = glp_next_item(GLP_COL))
         {  int seqn, type;
            double lb, ub;
            glp_get_seqn(GLP_COL, &seqn);
            glp_get_bounds(GLP_COL, &type, &lb, &ub);
            if (type == GLP_LO && lb == 0.0) continue;
            if (!flag) fprintf(fp, "BOUNDS\n"), flag = 1;
            switch (type)
            {  case GLP_FR:
                  fprintf(fp, " FR %-8s  %-8s\n", name, col_name[seqn]);
                  break;
               case GLP_LO:
                  fprintf(fp, " LO %-8s  %-8s  %12s\n", name,
                     col_name[seqn], mps_number(lb));
                  break;
               case GLP_UP:
                  fprintf(fp, " MI %-8s  %-8s\n", name, col_name[seqn]);
                  fprintf(fp, " UP %-8s  %-8s  %12s\n", name,
                     col_name[seqn], mps_number(ub));
                  break;
               case GLP_DB:
                  if (lb != 0.0)
                  fprintf(fp, " LO %-8s  %-8s  %12s\n", name,
                     col_name[seqn], mps_number(lb));
                  fprintf(fp, " UP %-8s  %-8s  %12s\n", name,
                     col_name[seqn], mps_number(ub));
                  break;
               case GLP_FX:
                  fprintf(fp, " FX %-8s  %-8s  %12s\n", name,
                     col_name[seqn], mps_number(lb));
                  break;
               default:
                  insist(type != type);
            }
         }
      }
      /* write ENDATA indicator card */
      fprintf(fp, "ENDATA\n");
      /* free working arrays */
      ufree(row_name), row_name = NULL;
      ufree(col_name), col_name = NULL;
      /* close the output text file */
      fflush(fp);
      if (ferror(fp))
      {  error("glp_write_mps: can't write to `%s' - %s", fname,
            strerror(errno));
         goto fail;
      }
      fclose(fp);
      /* returns to the calling program */
      return 0;
fail: /* the operation failed */
      if (row_name != NULL) ufree(row_name);
      if (col_name != NULL) ufree(col_name);
      if (fp != NULL) fclose(fp);
      return 1;
}

/*----------------------------------------------------------------------
-- make_names - generate standard MPS names using original names.
--
-- This routine tries to make names of rows (if what = GLP_ROW) or
-- columns (if what = GLP_COL), whose length doesn't exceed 8 chars,
-- using original row and column names as templates. The result names
-- are placed in alias[1], ..., alias[n], where n is the total number
-- of rows/columns. */

static void make_names(int what, mps_name alias[])
{     AVLTREE *tree;
      AVLNODE *node;
      int k = 0, ret;
      tree = create_avl((int (*)(void *, void *))strcmp);
      for (ret = glp_first_item(what); ret == 0;
           ret = glp_next_item(what))
      {  char name[GLP_MAX_NAME+1];
         int len;
         k++;
         glp_get_name(what, name);
         if (name[0] == '$') goto alas;
         len = strlen(name);
         if (len <= 8)
         {  strcpy(alias[k], name);
            if (find_by_key(tree, alias[k]) == NULL) goto fini;
            goto alas;
         }
         /* the first try: abc~wxyz */
         memcpy(alias[k]+0, name+0, 3);
         memcpy(alias[k]+3, "~", 1);
         memcpy(alias[k]+4, name+(len-4), 4);
         if (find_by_key(tree, alias[k]) == NULL) goto fini;
         /* the second try: abcd~xyz */
         memcpy(alias[k]+0, name+0, 4);
         memcpy(alias[k]+4, "~", 1);
         memcpy(alias[k]+5, name+(len-3), 3);
         if (find_by_key(tree, alias[k]) == NULL) goto fini;
         /* the third try: abcde~yz */
         memcpy(alias[k]+0, name+0, 5);
         memcpy(alias[k]+4, "~", 1);
         memcpy(alias[k]+6, name+(len-2), 2);
         if (find_by_key(tree, alias[k]) == NULL) goto fini;
         /* the fourth try: abcdef~z */
         memcpy(alias[k]+0, name+0, 6);
         memcpy(alias[k]+4, "~", 1);
         memcpy(alias[k]+7, name+(len-1), 1);
         if (find_by_key(tree, alias[k]) == NULL) goto fini;
alas:    /* nothing came of it */
         {  char *t;
            sprintf(alias[k], "%c%7d", what == GLP_ROW ? 'R' : 'C', k);
            for (t = alias[k]; *t; t++) if (*t == ' ') *t = '~';
            insist(find_by_key(tree, alias[k]) == NULL);
         }
fini:    /* enter the generated name to the symbol table */
         insist(strlen(alias[k]) <= 8);
         node = insert_by_key(tree, alias[k]);
         node->type = k;
      }
      delete_avl(tree);
      return;
}

/*----------------------------------------------------------------------
-- mps_number - convert number to standard 12-character MPS format.
--
-- This routine converts the given floating point value val to the
-- standard 12-character MPS format. It tries to provide maximal number
-- of significan digits. */

static char *mps_number(double val)
{     static char numb[255+1]; int n; char *e;
      for (n = 12; n >= 6; n--)
      {  if (val != 0.0 && fabs(val) < 1.0)
            sprintf(numb, "%.*E", n, val);
         else
            sprintf(numb, "%.*G", n, val);
         insist(strlen(numb) <= 255);
         e = strrchr(numb, 'E');
         if (e != NULL) sprintf(e+1, "%d", atoi(e+1));
         if (strlen(numb) <= 12) return numb;
      }
      fault("glp_write_mps: can't convert floating point number '%g' to"
         " character string", val);
      return NULL; /* to relax compiler */
}

/* eof */
