/* glpmat/vt_solve.c */

/*----------------------------------------------------------------------
-- This file is a part of the GNU LPK package.
--
-- Copyright (C) 2000 Andrew Makhorin <mao@mai2.rcnet.ru>,
--                    Department for Applied Informatics,
--                    Moscow Aviation Institute, Moscow, Russia.
--                    All rights reserved.
--
-- This code is free software; you can redistribute it and/or modify
-- it under the terms of the GNU General Public License as published by
-- the Free Software Foundation; either version 2 of the License, or
-- (at your option) any later version.
--
-- This software is distributed "as is" in the hope that it will be
-- useful, but WITHOUT ANY WARRANTY; without even the implied warranty
-- of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
-- General Public License for more details.
--
-- You should have received a copy of the GNU General Public License
-- along with this program; if not, write to the Free Software
-- Foundation, 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA.
----------------------------------------------------------------------*/

#include <stddef.h>
#include "glpmat.h"
#include "glpset.h"

/*----------------------------------------------------------------------
-- vt_solve - solve implicit transposed upper triangular system V'*x =b.
--
-- *Synopsis*
--
-- #include "glpmat.h"
-- double *vt_solve(PER *P, MAT *V, PER *Q, double x[], double work[]);
--
-- *Description*
--
-- The vt_solve routine solves the system V'*x = b, where V' is a
-- matrix transposed to the given sparse matrix V, U = P*V*Q is upper
-- triangular matrix, P and Q are permutation matrices, x is dense
-- vector of unknowns, b is dense vector of right-hand sides. The matrix
-- U should be non-singular, i.e. all its diagonal elements should be
-- non-zeros. On entry the array x should contain elements of the vector
-- b in locations x[1], x[2], ..., x[n]. On exit this array will contain
-- the vector x in the same locations. The auxiliary array work should
-- have at least 1+n elements, where n is order of the system.
--
-- *Returns*
--
-- The vt_solve routine returns a pointer to the array x. */

double *vt_solve(PER *P, MAT *V, PER *Q, double x[], double work[])
{     ELEM *e;
      int i, j, flag = 1;
      double uii;
      if (V->m != V->n)
         fault("vt_solve: matrix is not quadratic");
      if (!(P->n == V->m && Q->n == V->m))
         fault("vt_solve: permutation matrices have invalid order");
      /* compute the vector b~ = inv(Q)*b */
      for (i = 1; i <= V->m; i++) work[i] = x[i];
      iper_vec(x, Q, work);
      /* solve the system U'*x~ = b~, where U' is a matrix transposed
         to U = P*V*Q */
      for (i = 1; i <= V->m; i++)
      {  /* flag = 1 means that x~[1] = ... = x~[i-1] = 0; therefore if
            b~[i] = 0 then x~[i] = 0 */
         if (flag && x[i] == 0.0) continue;
         uii = 0.0;
         for (e = V->col[Q->col[i]]; e != NULL; e = e->col)
         {  j = P->col[e->i];
            if (j > i)
               fault("vt_solve: matrix P*V*Q is not upper triangular");
            if (j == i)
               uii = e->val;
            else
               x[i] -= e->val * x[j];
         }
         if (uii == 0.0)
            fault("vt_solve: matrix is singular");
         x[i] /= uii;
         if (x[i] != 0.0) flag = 0;
      }
      /* compute the vector x = inv(P)*x~ */
      for (i = 1; i <= V->m; i++) work[i] = x[i];
      iper_vec(x, P, work);
      return x;
}

/* eof */
