/* glpmat/eq_scaling.c */

/*----------------------------------------------------------------------
-- This file is a part of the GNU LPK package.
--
-- Copyright (C) 2000 Andrew Makhorin <mao@mai2.rcnet.ru>,
--                    Department for Applied Informatics,
--                    Moscow Aviation Institute, Moscow, Russia.
--                    All rights reserved.
--
-- This code is free software; you can redistribute it and/or modify
-- it under the terms of the GNU General Public License as published by
-- the Free Software Foundation; either version 2 of the License, or
-- (at your option) any later version.
--
-- This software is distributed "as is" in the hope that it will be
-- useful, but WITHOUT ANY WARRANTY; without even the implied warranty
-- of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
-- General Public License for more details.
--
-- You should have received a copy of the GNU General Public License
-- along with this program; if not, write to the Free Software
-- Foundation, 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA.
----------------------------------------------------------------------*/

#include <math.h>
#include <stddef.h>
#include "glpmat.h"

/*----------------------------------------------------------------------
-- eq_scaling - implicit equilibration scaling.
--
-- *Synopsis*
--
-- #include "glpmat.h"
-- void eq_scaling(MAT *A, double R[], double S[], int ord);
--
-- *Description*
--
-- The eq_scaling routine performs implicit equilibration scaling of
-- the matrix R*A*S, where A is the given sparse matrix (that remains
-- unchanged on exit), R and S are the given diagonal scaling matrices.
-- The result of scaling is the matrix R'*A*S', where R' and S' are new
-- scaling matrices computed by the routine, which are placed in the
-- same array. Diagonal elements of the matrix R should be placed in
-- locations R[1], R[2], ..., R[m], where m is number of rows of the
-- matrix A. Diagonal elements of the matrix S should be placed in
-- locations S[1], S[2], ..., S[n], where n is number of columns of the
-- matrix A. Diagonal elements of the matrices R' and S' will be placed
-- in the same manner.
--
-- To perform equilibration scaling the routine divide all elements of
-- each row (column) to the largest absolute value of elements in the
-- corresponding row (column).
--
-- Before a call the matrices R and S should be defined (if the matrix
-- A is unscaled, R and S should be untity matrices). As a result of
-- scaling the routine computes new matrices R' and S', that define the
-- scaled matrix R'*A*S'. (Thus scaling is implicit, because the matrix
-- A remains unchanged.)
--
-- The parameter ord defines the order of scaling:
--
-- if ord = 0, at first rows, then columns;
-- if ord = 1, at first columns, then rows. */

static void scale_rows(MAT *A, double R[], double S[]);
static void scale_cols(MAT *A, double R[], double S[]);

void eq_scaling(MAT *A, double R[], double S[], int ord)
{     if (ord == 0)
      {  scale_rows(A, R, S);
         scale_cols(A, R, S);
      }
      else
      {  scale_cols(A, R, S);
         scale_rows(A, R, S);
      }
      return;
}

static void scale_rows(MAT *A, double R[], double S[])
{     /* this routine performs equilibration scaling of rows */
      ELEM *e;
      int i;
      double big, temp;
      for (i = 1; i <= A->m; i++)
      {  big = 0.0;
         for (e = A->row[i]; e != NULL; e = e->row)
         {  temp = fabs(R[e->i] * e->val * S[e->j]);
            if (big < temp) big = temp;
         }
         if (big != 0.0) R[i] /= big;
      }
      return;
}

static void scale_cols(MAT *A, double R[], double S[])
{     /* this routine performs equilibration scaling of columns */
      ELEM *e;
      int j;
      double big, temp;
      for (j = 1; j <= A->n; j++)
      {  big = 0.0;
         for (e = A->col[j]; e != NULL; e = e->col)
         {  temp = fabs(R[e->i] * e->val * S[e->j]);
            if (big < temp) big = temp;
         }
         if (big != 0.0) S[j] /= big;
      }
      return;
}

/* eof */
