/* glpgel/gel.c */

/*----------------------------------------------------------------------
-- This file is a part of the GNU LPK package.
--
-- Copyright (C) 2000 Andrew Makhorin <mao@mai2.rcnet.ru>,
--                    Department for Applied Informatics,
--                    Moscow Aviation Institute, Moscow, Russia.
--                    All rights reserved.
--
-- This code is free software; you can redistribute it and/or modify
-- it under the terms of the GNU General Public License as published by
-- the Free Software Foundation; either version 2 of the License, or
-- (at your option) any later version.
--
-- This software is distributed "as is" in the hope that it will be
-- useful, but WITHOUT ANY WARRANTY; without even the implied warranty
-- of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
-- General Public License for more details.
--
-- You should have received a copy of the GNU General Public License
-- along with this program; if not, write to the Free Software
-- Foundation, 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA.
----------------------------------------------------------------------*/

#include <assert.h>
#include <float.h>
#include <math.h>
#include <stddef.h>
#include "glpgel.h"
#include "glpset.h"

/*----------------------------------------------------------------------
-- gel - sparse gaussian elimination.
--
-- *Synopsis*
--
-- #include "glpgel.h"
-- int gel(MAT *V, PER *P, PER *Q,
--    void (*func)(int i, int k, double f), double tol, double lim,
--    int *Unz, double *Umax, double *Ubig, DUFF *rs, DUFF *cs,
--    double rmax[], double work[]);
--
-- *Description*
--
-- The gel routine uses gaussian elimination technique to transform the
-- given sparse matrix V to the upper triangular form.
--
-- The result of the transformation is the matrix V', which are placed
-- instead the input matrix V, and permutation matrices P and Q, which
-- define the upper triangular matrix U = P*V'*Q. Note that the matrix
-- U has implicit representation, therefore all operations are actually
-- performed on the matrix V that differs from U only in the order of
-- rows and columns determined by permutation matrices P and Q.
--
-- To eliminate subdiagonal elements of the matrix U the routine applies
-- to this matrix elementary transformations of the following types: row
-- permutation, column permutation, and subtraction one row multiplied
-- by a number from other row (gaussian transformation). Each time when
-- gaussian transformation is applied to the matrix U (in order to
-- eliminate one subdiagonal element), the gel routine calls the func
-- routine in order that the calling routine would have possibility to
-- accumulate all information about these transformations (information
-- about row and column permutations are accumulated in the matrices P
-- and Q by the gel routine).
--
-- The point where the func routine is called, and the meaning of the
-- parameters passed to this routine can be explained by means of the
-- following general scheme of elimination process:
--
-- for (k = 1; k <= n; k++)
-- {  (* k-th elimination step starts here *)
--    (* choose the pivot element u[p,q] *)
--    p = ...; assert(k <= p <= n);
--    q = ...; assert(k <= q <= n);
--    (* permute rows of the matrix U with numbers k and p *)
--    ...;
--    (* permute columns of the matrix U with numbers k and q *)
--    ...;
--    (* now the pivot element is u[k,k] *)
--    assert(u[k,k] != 0.0);
--    (* eliminate non-zero subdiagonal elements of the matrix U that
--       are placed in k-th column *)
--    for (i = k+1; i <= n; i++)
--    {  (* eliminate subdiagional element u[i,k] *)
--       if (u[i,k] == 0.0) continue;
--       (* compute gaussian multiplier *)
--       f = u[i,k] / u[k,k];
--       (* (i-th row of U) := (i-th row of U) - f * (k-th row of U) *)
--       ...;
--       assert(u[i,k] == 0.0);
--       (* i-th row of the matrix U has been transformed *)
--       func(i, k, f);
--    }
-- }
--
-- Should note that row numbers passed to the func routine correspond to
-- the matrix U (not to V!). The correspondence between row and column
-- numbers of the matrices U = P*V*Q and V is determined by the row
-- permutation matrix P and the column permutation matrix Q. So, if an
-- element u[i,j] of the matrix U corresponds to an element v[i',j'] of
-- the matrix V, the following formulae take a place:
--
-- i' = P->row[i], j' = Q->col[j], i = P->col[i'], j = Q->row[j'].
--
-- The parameter tol is the tolerance used for threshold pivoting. It
-- should be in the range 0 < tol < 1. For more details see remarks for
-- the find_pivot routine.
--
-- The parameter lim specifies maximal allowable growth of elements of
-- the matrix U during elimination process. The process is considered as
-- numerically stable if on each step the following condition is true:
--
-- Ubig <= lim * Umax
--
-- (description of the parameters Umax and Ubig are given below).
--
-- Before a call the variable Unz should define the total number of
-- non-zeros in the source matrix U. After a call this variable will
-- define the total number of non-zeros in the transformed matrix U'.
-- (If before a call Unz is set to zero, after a call Unz will define
-- the difference between total numbers of non-zeros in the matrices
-- U' and U.)
--
-- The variables Umax and Ubig are ignored before a call. After a call
-- the variable Umax will contain the maximum of absolute values of
-- elements of the source matrix U, and the variable Ubig will contain
-- the maximum of absolute values of those elements, which appeared in
-- the matrix U during elimination process.
--
-- The parameters rs and cs should define Duff schemes used to keep
-- the lists of active rows and columns. Before a call these schemes
-- may contain arbitrary information that is ignored. In the case of
-- error (see below) after a call these schemes will contain active
-- rows and columns of the matrix V (not U!). These schemes should be
-- created by calls rs = create_duff(n, n) and cs = create_duff(n, n),
-- where n is the order of the matrix V.
--
-- The auxiliary arrays rmax and work should have at least 1+n elements,
-- where n is the order of the matrix V.
--
-- *Returns*
--
-- The gel routine returns one of the following error codes:
--
--  0 - no errors;
-- -k - on k-th elimination step pivot can't be chosen because all
--      elements of the active submatrix are zeros;
-- +k - on k-th elimination step numerical stability condition (see
--      above) has been violated. */

static int debug = 0;
/* debug mode flag */

static MAT *V;
/* sparse matrix V that should be transformed */

static PER *P;
/* left permutation matrix P */

static PER *Q;
/* right permutation matrix Q */

static int n;
/* order of matrices P, V, Q, and U = P*V*Q */

static int k;
/* the number of elimination step */

static void (*func)(int i, int k, double f);
/* external routine that accumulates elementary transformations applied
   to the matrix U */

static double tol;
/* tolerance used for pivot choosing */

static double lim;
/* maximal allowable relative growth of elements of the matrix U */

static int Unz;
/* count of non-zero elements of the matrix U */

static double Umax;
/* maximum of absolute values of elements of the matrix U (or of its
   initial active submatrix) */

static double Ubig;
/* maximum of absolute values of elements appeared in the matrix U (or
   in its initial active submatrix) */

static DUFF *rs, *cs;
/* lists of the numbers of rows and columns of the matrix V that belong
   to the active submatrix before k-th elimination step */

static double *rmax; /* double rmax[1+n]; */
/* if i-th row of the matrix V belongs to the active submatrix, rmax[i]
   is maximum of absolute values of elements in i-th row */

static double *work; /* double work[1+n]; */
/* auxiliary array of the eliminate routine */

#define iU(i) (P->col[i])
/* converts row number of V to row number of U */

#define iV(i) (P->row[i])
/* converts row number of U to row number of V */

#define jU(j) (Q->row[j])
/* converts column number of V to column number of U */

#define jV(j) (Q->col[j])
/* converts column number of U to column number of V */

#define include_row(i, nz) include_obj(rs, i, nz)
#define include_col(j, nz) include_obj(cs, j, nz)
#define exclude_row(i)     exclude_obj(rs, i)
#define exclude_col(j)     exclude_obj(cs, j)

static void check_data(void);
/* check main data structures for correctness */

static ELEM *find_pivot(void);
/* choose pivot element */

static void eliminate(ELEM *piv);
/* eliminate subdiagonal elements */

int gel(MAT *_V, PER *_P, PER *_Q,
      void (*_func)(int i, int k, double f), double _tol, double _lim,
      int *_Unz, double *_Umax, double *_Ubig, DUFF *_rs, DUFF *_cs,
      double _rmax[], double _work[])
{     ELEM *piv, *e;
      int i, j, p, q, nz, ret = 0;
      double big;
      /* initialization */
      V = _V; P = _P; Q = _Q; n = V->m;
      func = _func; tol = _tol; lim = _lim;
      Unz = *_Unz; Umax = 0.0; Ubig = 0.0;
      rs = _rs; cs = _cs; rmax = _rmax; work = _work;
      if (V->m != V->n)
         fault("gel: transformed matrix is not quadratic");
      if (!(P->n == n && Q->n == n))
         fault("gel: permutation matrices have invalid order");
      /* reset permutation matrices, because initially U = V */
      reset_per(P);
      reset_per(Q);
      /* build lists of (active) rows of the matrix V, compute Umax and
         Ubig, and fill elements of the array rmax */
      reset_duff(rs);
      for (i = 1; i <= n; i++)
      {  /* count non-zeros in i-th row of the matrix V and determine
            the maximum of absolute values of elements in this row */
         nz = 0;
         big = 0.0;
         for (e = V->row[i]; e != NULL; e = e->row)
         {  if (e->val == 0.0)
               fault("gel: transformed matrix has invalid pattern");
            nz++;
            if (big < fabs(e->val)) big = fabs(e->val);
         }
         /* include i-th row to the active list */
         include_row(i, nz);
         /* store the value max|v[i,*]| for this active row */
         rmax[i] = big;
         /* update the value max|u[*,*]| */
         if (Umax < big) Umax = Ubig = big;
      }
      /* build lists of (active) columns of the matrix V */
      reset_duff(cs);
      for (j = 1; j <= n; j++)
      {  /* count non-zeros in j-th column of the matrix V */
         nz = 0;
         for (e = V->col[j]; e != NULL; e = e->col) nz++;
         /* include j-th column to the active list */
         include_col(j, nz);
      }
      /* clear auxiliary array */
      for (j = 1; j <= n; j++) work[j] = 0.0;
      /* main loop of gaussian elimination */
      for (k = 1; k <= n; k++)
      {  /* the goal of k-th step is to nullify subdiagonal elements
            placed in k-th column of the matrix U */
         check_data();
         /* choose pivot element u[p,q] */
         piv = find_pivot();
         if (piv == NULL)
         {  /* all elements of the active submatrix are zero */
            ret = -k;
            goto done;
         }
         /* piv = v[i,j] = u[p,q] */
         p = iU(piv->i), q = jU(piv->j);
         assert(k <= p && p <= n && k <= q && q <= n);
         assert(piv->val != 0.0);
         /* perform implicit permutations of rows and columns of the
            matrix U in order to move the pivot element from u[p,q] to
            u[k,k] */
         {  int t1, t2;
            /* permute k-th and p-th rows of the matrix U */
            t1 = iV(k); t2 = iV(p);
            iV(k) = t2; iU(t2) = k;
            iV(p) = t1; iU(t1) = p;
            /* permute k-th and q-th columns of the matrix U */
            t1 = jV(k); t2 = jV(q);
            jV(k) = t2; jU(t2) = k;
            jV(q) = t1; jU(t1) = q;
         }
         /* eliminate subdiagonal elements in k-th column of the matrix
            U using the pivot element u[k,k] */
         eliminate(piv);
         *_Unz = Unz; *_Umax = Umax; *_Ubig = Ubig;
         if (Ubig > lim * Umax)
         {  /* elements of the matrix U are growing too intense */
            ret = +k;
            goto done;
         }
      }
      check_data();
done: /* return to the calling program */
      return ret;
}

/*----------------------------------------------------------------------
-- check_data - check main data structures for correctness.
--
-- This routine checks the correctness of the main data structures
-- before k-th elimination step. The following conditions are checked:
--
-- 1. Matrices V, P, and Q should have correct representations.
--
-- 2. Matrix U = P*V*Q should be the following:
--
--    1       k         n
-- 1  x x x x x x x x x x
--    . x x x x x x x x x
--    . . x x x x x x x x
--    . . . x x x x x x x
-- k  . . . . * * * * * *
--    . . . . * * * * * *
--    . . . . * * * * * *
--    . . . . * * * * * *
--    . . . . * * * * * *
-- n  . . . . * * * * * *
--
-- (elements of the active submatrix are marked by '*').
--
-- 3. All zeros of the matrix V should be symbolic zeros.
--
-- 4. Rows and columns of the matrix V corresponding to the rows and
--    columns 1, 2, ..., k-1 of the matrix U should be missing from the
--    active lists, and rows and columns of V corresponding to the rows
--    and columns k, k+1, ..., n should be placed in the active lists.
--    In the latter case counters of the active rows and columns should
--    correctly reflect the number of non-zeros in the active submatrix.
--
-- 5. Maximums of absolute values of elements of the active rows of the
--    matrix V should be stored in the array rmax.
--
-- This routine allows a call when k = m+1 in order to check the
-- correctness of the main data structures after the last elimination
-- step. */

static void check_data(void)
{     ELEM *e;
      int i, j, nz;
      double big;
      if (!debug) goto skip;
      assert(1 <= k && k <= n+1);
      check_mat(V);
      check_per(P);
      check_per(Q);
      /* check for rows */
      for (i = 1; i <= n; i++)
      {  if (i < k)
         {  /* i-th row of U is placed above the active submatrix */
            for (e = V->row[iV(i)]; e != NULL; e = e->row)
            {  j = jU(e->j);
               assert(j >= i);
               assert(e->val != 0.0);
            }
            assert(rs->len[iV(i)] < 0);
         }
         else
         {  /* i-th row of U belongs to the active submatrix */
            nz = 0;
            big = 0.0;
            for (e = V->row[iV(i)]; e != NULL; e = e->row)
            {  j = jU(e->j);
               assert(j >= k);
               assert(e->val != 0.0);
               nz++;
               if (big < fabs(e->val)) big = fabs(e->val);
            }
            assert(rs->len[iV(i)] == nz);
            assert(rmax[iV(i)] == big);
         }
      }
      /* check for columns */
      for (j = 1; j <= n; j++)
      {  if (j < k)
         {  /* j-th column of U is placed left the active submatrix */
            for (e = V->col[jV(j)]; e != NULL; e = e->col)
            {  i = iU(e->i);
               assert(i <= j);
               assert(e->val != 0.0);
            }
            assert(cs->len[jV(j)] < 0);
         }
         else
         {  /* j-th column of U belongs to the active submatrix */
            nz = 0;
            for (e = V->col[jV(j)]; e != NULL; e = e->col)
            {  i = iU(e->i);
               assert(e->val != 0.0);
               if (i >= k) nz++;
            }
            assert(cs->len[jV(j)] == nz);
         }
      }
skip: return;
}

/*----------------------------------------------------------------------
-- find_pivot - choose pivot element.
--
-- This routine chooses a pivot element in the active submatrix of the
-- matrix U = P*V*Q and returns a pointer to the chosen element (the
-- routine returns NULL, if appropriate element cannot be chosen; see
-- below).
--
-- It is assumed that on a call the matrix U is the following:
--
--    1       k         n
-- 1  x x x x x x x x x x
--    . x x x x x x x x x
--    . . x x x x x x x x
--    . . . x x x x x x x
-- k  . . . . * * * * * *
--    . . . . * * * * * *
--    . . . . * * * * * *
--    . . . . * * * * * *
--    . . . . * * * * * *
-- n  . . . . * * * * * *
--
-- where rows and columns with numbers k, k+1, ..., n from the active
-- submatrix (elements of this submatrix are marked by '*').
--
-- Each active row of the matrix V should be in the list R[nz], where
-- nz is the number of non-zeros in that row. Analogously, each active
-- column should be in the list C[nz], where nz is the number of
-- non-zeros in that column (except elements missing from the active
-- submatrix). If i-th row of the matrix V is active, the element
-- rmax[i] should contain the maximum of absolute values of elements in
-- this row.
--
-- In order that computations to be numerically stable the routine uses
-- so called threshold pivoting proposed by J.Reid. The element u[i,j]
-- satisfies to the stability condition if it is not very small among
-- other elements in the same row, i.e. if |u[i,j]| >= tol * max|u[i,*]|
-- where 0 < tol < 1 is the given tolerance.
--
-- In order to preserve sparsity of the matrix U the routine uses the
-- Markowitz strategy, trying to choose such element u[i,j], which
-- satisfies to the stability condition (see above) and also has the
-- smallest Markowitz cost (nr[i]-1) * (nc[j]-1), where nr[i] and nc[j]
-- are the number of non-zeros in i-th row and in j-th column resp. of
-- the active submatrix of the matrix U.
--
-- In order to reduce the search in active rows and columns (i.e. in
-- order not to look through entire active submatrix) the routine uses
-- the technique proposed by I.Duff. If there is a column or a row that
-- contains exactly one non-zero element (singlet), the routine chooses
-- such element at once. Otherwise the routine continues the search for
-- nz = 2, 3, ..., n, analyzing at each step those rows and columns that
-- contain nz non-zeros in the active submatrix. The routine stops the
-- search in the following two cases: (a) if all columns containing nz
-- non-zeros were analyzed and best <= nz * (nz-1), or (b) if all rows
-- containing nz non-zeros were analyzed and best <= (nz-1) * (nz-1),
-- where best = (nr[i]-1) * (nc[j]-1) is best (smallest) Markowitz cost,
-- that was reached for some element u[i,j], which the routine chooses
-- as the pivot element. Such partial pivoting is able to reduce the
-- search keeping good sparsity for the most practical cases.
--
-- If all elements of the active submatrix are zero and it is impossible
-- to choose the pivot element, the routine returns NULL.
--
-- The main data structures should satisfy the same conditions, which
-- are checked by the routine check_data. */

static ELEM *find_pivot(void)
{     ELEM *piv = NULL, *e;
      int i, j, nz;
      double best = DBL_MAX, cost;
      assert(1 <= k && k <= n);
      /* if there is a column containing exactly one non-zero in the
         active submatrix (all such columns are in the list C[1]), the
         corresponding element may be chosen as pivot. */
      j = cs->head[1];
      if (j != 0)
      {  /* j-th column contains column singlet, but this column may
            contain other elements that are not belong to the active
            submatrix (in the matrix U all such elements are placed
            above k-th row), so singlet should be found */
         for (piv = V->col[j]; piv != NULL; piv = piv->col)
            if (iU(piv->i) >= k) break;
         assert(piv != NULL);
         goto done;
      }
      /* if there is a row containing exactly one non-zero in the
         active submatrix (all such rows are in the list R[1]), the
         corresponding element may be chosen as pivot */
      i = rs->head[1];
      if (i != 0)
      {  /* i-th row contains row singlet (there should be no other
            elements in this row) */
          piv = V->row[i];
          assert(piv != NULL && piv->row == NULL);
          goto done;
      }
      /* main loop for searching for pivot */
      for (nz = 2; nz <= n; nz++)
      {  /* look up columns from the list C[nz] */
         for (j = cs->head[nz]; j != 0; j = cs->next[j])
         {  /* j-th column contains exactly nz non-zeros in the active
               submatrix */
            for (e = V->col[j]; e != NULL; e = e->col)
            {  i = e->i;
               /* if v[i,j] is not in the active submatrix, it should
                  be skipped */
               if (iU(i) < k) continue;
               /* if v[i,j] doesn't satisfy to the numerical stability
                  condition, it should be skipped */
               if (fabs(e->val) < tol * rmax[i]) continue;
               /* compute Markowitz cost of v[i,j] */
               cost = (double)(rs->len[i] - 1) * (double)(nz - 1);
               /* decide, whether v[i,j] fits to be pivot */
               if (piv == NULL || best > cost ||
                  best == cost && fabs(piv->val) < fabs(e->val))
                     piv = e, best = cost;
            }
            /* check Duff's criterion to terminate searching */
            if (best <= (double)(nz) * (double)(nz - 1)) goto done;
         }
         /* look up rows from the list R[nz] */
         for (i = rs->head[nz]; i != 0; i = rs->next[i])
         {  /* i-th row contains exactly nz non-zeros in the active
               submatrix */
            for (e = V->row[i]; e != NULL; e = e->row)
            {  j = e->j;
               /* v[i,j] is always in the active submatrix */
               assert(jU(j) >= k);
               /* if v[i,j] doesn't satisfy to the numerical stability
                  condition, it should be skipped */
               if (fabs(e->val) < tol * rmax[i]) continue;
               /* compute Markowitz cost of v[i,j] */
               cost = (double)(nz - 1) * (double)(cs->len[j] - 1);
               /* decide, whether v[i,j] fits to be pivot */
               if (piv == NULL || best > cost ||
                  best == cost && fabs(piv->val) < fabs(e->val))
                     piv = e, best = cost;
            }
            /* check Duff's criterion to terminate searching */
            if (best <= (double)(nz - 1) * (double)(nz - 1)) goto done;
         }
      }
done: if (piv != NULL)
      {  /* the chosen pivot should be in the active submatrix */
         assert(k <= iU(piv->i) && iU(piv->i) <= n);
         assert(k <= jU(piv->j) && jU(piv->j) <= n);
      }
      return piv;
}

/*----------------------------------------------------------------------
-- eliminate - eliminate subdiagonal elements.
--
-- This routine performs gaussian elimination to nullify subdiagonal
-- elements of the matrix U = P*V*Q that are placed in the pivot (k-th)
-- column. The piv parameter points to the pivot element u[k,k].
--
-- It is assumed that before a call the matrix U is the following:
--
--    1       k         n
-- 1  x x x x x x x x x x
--    . x x x x x x x x x
--    . . x x x x x x x x
--    . . . x x x x x x x
-- k  . . . . * * * * * *
--    . . . . # * * * * *
--    . . . . # * * * * *
--    . . . . # * * * * *
--    . . . . # * * * * *
-- n  . . . . # * * * * *
--
-- where rows and columns k, k+1, ..., n belong to the active submatrix
-- (eliminated elements are marked by '#', other elements of the active
-- submatrix are marked by '*').
--
-- (Since the matrix U has implicit representation, all transformations
-- are performed actually on the matrix V, which differs from the matrix
-- U only in the order of rows and columns.)
--
-- The auxiliary array work is used as an accumulator and before a call
-- it should contain zeros. The routine remains this array in the same
-- state after a call.
--
-- Let u[k,k] = v[p,q] is the pivot element. To eliminate subdiagonals
-- elements u[i',k] = v[i,q], i' = k+1, k+2, ..., n, the routine applies
-- the following gaussian transformation:
--
-- (i-th row of V) := (i-th row of V) - f[i] * (p-th row of V),
--
-- where f[i] = v[i,q]/v[p,q] is gaussian multiplier, and the
-- correspondence between row numbers i and i' is determined by the
-- permutation matrix P.
--
-- The main data structures should satisfy the same conditions, which
-- are checked by the routine check_data. After a call these conditions
-- will also take a place. */

static void eliminate(ELEM *piv)
{     ELEM *v, *vn, *e, *en, *ep;
      int i, j, k, p, q, count, nz;
      double f, big, drop = 1e-15;
      assert(piv != NULL && piv->val != 0.0);
      /* determine the pivot v[p,q] = u[k,k] */
      p = piv->i;
      q = piv->j;
      k = iU(p);
      assert(k == jU(q));
      /* main elimination loop */
      for (v = V->col[q]; v != NULL; v = v->col)
      {  i = v->i;
         /* if v[i,q] = u[i',k], where i' <= k, this element needn't to
            be eliminated (because it is placed on the main diagonal of
            the matrix U or above that diagonal) */
         if (iU(i) <= k) continue;
         /* compute gaussian multiplier f = v[i,q]/v[p,q] */
         f = v->val / piv->val;
         /* work := (i-th row of V) */
         for (e = V->row[i]; e != NULL; e = e->row)
            work[e->j] = e->val;
         /* work := work - f * (p-th row of V) */
         for (e = V->row[p]; e != NULL; e = e->row)
            work[e->j] -= f * e->val;
         /* the eliminated element v[i,q] should become exact zero */
         work[q] = 0.0;
         /* now new elements of i-th row of the matrix V are placed in
            the array work */
         /* replace existing elements of i-th row */
         big = 0.0; /* is a new value of max|v[i,*]| */
         for (e = V->row[i]; e != NULL; e = e->row)
         {  j = e->j;
            /* if |v[i,j]| < drop * max|v[i,*]|, where drop is relative
               drop tolerance, the element v[i,j] can be replaced by
               exact zero; this slightly decreases accuracy but allows
               to improve sparsity */
            if (fabs(work[j]) < drop * rmax[i]) work[j] = 0.0;
            /* replace element v[i,j] */
            if (big < fabs(work[j])) big = fabs(work[j]);
            e->val = work[j], work[j] = 0.0;
         }
         /* create new elements, which appeared in i-th row as a result
            of elimination */
         count = 0; /* number of new elements of i-th row */
         for (e = V->row[p]; e != NULL; e = e->row)
         {  j = e->j;
            /* (see notice above) */
            if (fabs(work[j]) < drop * rmax[i]) work[j] = 0.0;
            if (work[j] == 0.0) continue;
            /* create new (non-zero) element v[i,j] */
            if (big < fabs(work[j])) big = fabs(work[j]);
            new_elem(V, i, j, work[j]), Unz++, work[j] = 0.0;
            /* number of elements in i-th row and in j-th column now is
               increased by one */
            count++;
            nz = cs->len[j];
            exclude_col(j), include_col(j, nz+1);
         }
         if (count != 0)
         {  nz = rs->len[i];
            exclude_row(i), include_row(i, nz+count);
         }
         /* (now the array work again contains all zeros) */
         /* update the maximum max|v[i,*]| */
         rmax[i] = big;
         /* update the relative growth of elements of the matrix U
            during elimination */
         if (Ubig < big) Ubig = big;
         /* i-th row of the matrix V has been transformed */
         func(iU(i), k, f);
      }
      /* now all zeros appeared in the active submatrix as a result of
         elimination should be removed */
      /* most of zeros are usually placed in the q-th (pivot) column of
         the matrix V, since in other columns zeros can appear only as
         a result of numerical cancellation that happens relatively
         seldom (except some specific cases). To reduce number of passes
         through the pivot column this operations is performed in the
         following way. The outermost loop scans the pivot (q-th) column
         and removes all v[i,q] = 0 from the corresponding column list,
         so to remove most of zeros only one pass through the pivot
         column is needed. The middle loop scans each i-th row, where
         v[i,q] = 0 (this condition can take a place if and only if the
         i-th row was touched by elimination, hence this row can contain
         other zeros), and removes all v[i,j] = 0 from the corresponding
         row lists. The innermost loop scans each j-th column (except
         q-th column), where v[i,j] = 0, in order to remove these zeros
         from the corresponding column lists (so, to remove one zero
         from j-th column one pass of this column is needed) */
      for (v = V->col[q], V->col[q] = NULL; v != NULL; v = vn)
      {  vn = v->col;
         if (v->val != 0.0)
         {  /* element v[i,q] is non-zero, so it stays in the list of
               q-th column */
            v->col = V->col[q], V->col[q] = v;
            continue;
         }
         i = v->i;
         /* element v[i,q] is zero, hence i-th row was touched by
            elimination and can contain other zeros */
         count = 0; /* number of zeros placed in i-th row */
         for (e = V->row[i], V->row[i] = NULL; e != NULL; e = en)
         {  en = e->row;
            if (e->val != 0.0)
            {  /* element v[i,j] is non-zero, so it stays in the list
                  of i-th row */
               e->row = V->row[i], V->row[i] = e;
               continue;
            }
            j = e->j;
            /* element v[i,j] is zero */
            count++;
            /* remove this zero from the list of j-th column (but only
               if j != q, since zeros from the pivot column are removed
               in the outermost loop) */
            if (j == q) continue;
            if (V->col[j] == e)
               V->col[j] = e->col;
            else
            {  for (ep = V->col[j]; ep != NULL; ep = ep->col)
                  if (ep->col == e) break;
               assert(ep != NULL);
               ep->col = e->col;
            }
            /* return v[i,j] to the memory pool */
            free_atom(V->pool, e), Unz--;
            /* one element was removed from j-th column */
            nz = cs->len[j];
            exclude_col(j), include_col(j, nz-1);
         }
         /* number of elements in i-th row is decreased */
         if (count != 0)
         {  nz = rs->len[i];
            exclude_row(i), include_row(i, nz-count);
         }
         /* returns v[i,q] to the memory pool (in this case it is not
            needed to update number of zeros in q-th column, because
            this column will be removed from the corresponding active
            list) */
         free_atom(V->pool, v), Unz--;
      }
      /* k-th elimination step is finished, so the pivot (p-th) row and
         the pivot (q-th) column of the matrix V should be removed from
         the active lists; since the deletion of the pivot row involves
         the deletion of all its elements from the active submatrix,
         the corresponding column counters should be decreased (this is
         not needed for the corresponding row counters, because now all
         subdiagonal elements of the pivot row are zeros) */
      for (e = V->row[p]; e != NULL; e = e->row)
      {  j = e->j;
         /* element v[p,j] left the active submatrix, so the counter of
            j-th column should be decreased */
         nz = cs->len[j];
         exclude_col(j), include_col(j, nz-1);
      }
      /* remove the pivot row and the pivot column of the matrix V from
         the active lists */
      exclude_row(p), exclude_col(q);
      return;
}

/* eof */
