/* glpapi/glp_set_name.c */

/*----------------------------------------------------------------------
-- This file is a part of the GNU LPK package.
--
-- Copyright (C) 2000 Andrew Makhorin <mao@mai2.rcnet.ru>,
--                    Department for Applied Informatics,
--                    Moscow Aviation Institute, Moscow, Russia.
--                    All rights reserved.
--
-- This code is free software; you can redistribute it and/or modify
-- it under the terms of the GNU General Public License as published by
-- the Free Software Foundation; either version 2 of the License, or
-- (at your option) any later version.
--
-- This software is distributed "as is" in the hope that it will be
-- useful, but WITHOUT ANY WARRANTY; without even the implied warranty
-- of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
-- General Public License for more details.
--
-- You should have received a copy of the GNU General Public License
-- along with this program; if not, write to the Free Software
-- Foundation, 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA.
----------------------------------------------------------------------*/

#include <assert.h>
#include <stddef.h>
#include "glpapi.h"

/*----------------------------------------------------------------------
-- glp_set_name - change name of the current row or column.
--
-- *Synopsis*
--
-- #include "glpk.h"
-- int glp_set_name(int what, char *name);
--
-- *Description*
--
-- If what = GLP_ROW, the glp_set_name routine assigns the new name to
-- the current row.
--
-- If what = GLP_COL, the glp_set_name routine assigns the new name to
-- the current column.
--
-- *Complexity*
--
-- This operation has time complexity O(log n), where n is the current
-- number of rows or columns in the workspace.
--
-- *Returns*
--
-- 0 - no errors;
-- 1 - the current row/column is undefined or row/column with the given
--     name already exists;
-- 2 - the parameter what or the given name is invalid. */

int glp_set_name(int what, char *name)
{     if (what == GLP_ROW)
      {  STR *row_name;
         AVLNODE *node;
         if (glp->this_row == NULL) return 1;
         if (glp_check_name(name)) return 2;
         row_name = set_str(create_str(glp->str_pool), name);
         if (find_by_key(glp->row_tab, row_name) != NULL)
         {  /* row already exists */
            delete_str(row_name);
            return 1;
         }
         node = find_by_key(glp->row_tab, glp->this_row->name);
         assert(node != NULL);
         delete_node(glp->row_tab, node);
         delete_str(glp->this_row->name);
         glp->this_row->name = row_name;
         node = insert_by_key(glp->row_tab, glp->this_row->name);
         node->link = glp->this_row;
      }
      else if (what == GLP_COL)
      {  STR *col_name;
         AVLNODE *node;
         if (glp->this_col == NULL) return 1;
         if (glp_check_name(name)) return 2;
         col_name = set_str(create_str(glp->str_pool), name);
         if (find_by_key(glp->col_tab, col_name) != NULL)
         {  /* column already exists */
            delete_str(col_name);
            return 1;
         }
         node = find_by_key(glp->col_tab, glp->this_col->name);
         assert(node != NULL);
         delete_node(glp->col_tab, node);
         delete_str(glp->this_col->name);
         glp->this_col->name = col_name;
         node = insert_by_key(glp->col_tab, glp->this_col->name);
         node->link = glp->this_col;
      }
      else
      {  /* invalid parameter */
         return 2;
      }
      return 0;
}

/* eof */
