/* glprsm/rsm_dual_col.c */

/*----------------------------------------------------------------------
-- This file is a part of the GLPK package.
--
-- Copyright (C) 2000, 2001 Andrew Makhorin <mao@mai2.rcnet.ru>,
--                          Department for Applied Informatics,
--                          Moscow Aviation Institute, Moscow, Russia.
--                          All rights reserved.
--
-- This code is free software; you can redistribute it and/or modify
-- it under the terms of the GNU General Public License as published by
-- the Free Software Foundation; either version 2 of the License, or
-- (at your option) any later version.
--
-- This software is distributed "as is" in the hope that it will be
-- useful, but WITHOUT ANY WARRANTY; without even the implied warranty
-- of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
-- General Public License for more details.
--
-- You should have received a copy of the GNU General Public License
-- along with this program; if not, write to the Free Software
-- Foundation, 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA.
----------------------------------------------------------------------*/

#include <assert.h>
#include <float.h>
#include <math.h>
#include "glprsm.h"

/*----------------------------------------------------------------------
-- rsm_dual_col - choose non-basis variable (dual simplex).
--
-- *Synopsis*
--
-- #include "glprsm.h"
-- void rsm_dual_col(RSM *rsm, double tol);
--
-- *Description*
--
-- The rsm_dual_col performs the ratio test in order to choose the
-- non-basis variable xN[q] (i.e. pivot column of the current simplex
-- table), which will enter the basis. It is assumed that p is the
-- number of the basis variable xB[p] that was chosen to leave the
-- basis, and the array ap contains p-th row of the current simplex
-- table that shows an influence of each xN[j] to the basis variable
-- xB[p]. If there is no primal feasible solution (i.e. dual solution
-- becomes unbounded), the routine sets q to zero.
--
-- The parameter tol is a relative tolerance. If the coefficient ap[j]
-- is too small, i.e. if |ap[j]| < eps, where eps = tol*max|ap[*]|, then
-- non-basis variable xN[j] is not considered by the routine.
--
-- The rsm_dual_col routine acts like rsm_pivot_row routine, however
-- the former operates with reduced costs (dual variables) instead basis
-- (primal) variables. It tries to choose such non-basis variable xN[q],
-- the reduced cost of which reaches its bound (i.e. zero) before any
-- reduced costs of other non-basis variables.
--
-- *Returns*
--
-- The rsm_dual_col routine returns absolute value of the reduced cost
-- of the chosen basis variable xB[p], which will take place when this
-- variable will leave the basis. */

double rsm_dual_col(RSM *rsm, double tol)
{     int n = rsm->n, j;
      double big, eps, temp, teta;
      /* compute the absolute tolerance eps using the given relative
         tolerance tol */
      big = 0.0;
      for (j = 1; j <= n; j++)
         if (big < fabs(rsm->ap[j])) big = fabs(rsm->ap[j]);
      eps = tol * big;
      /* turn to the case of increasing xB[p] in order to simplify
         program logic (remember that the objective function should be
         minimized) */
      if (rsm->tagp == LP_NU)
         for (j = 1; j <= n; j++) rsm->ap[j] = - rsm->ap[j];
      /* initial settings */
      rsm->q = 0, teta = DBL_MAX, big = 0.0;
      /* look through the list of non-basis variables */
      for (j = 1; j <= n; j++)
      {  /* if the coefficient ap[j] is too small, it is assumed that
            xB[p] doesn't depend on xN[j] */
         if (rsm->ap[j] == 0.0 || fabs(rsm->ap[j]) < eps) continue;
         /* analyze main cases */
         if (rsm->tagn[j] == LP_NF)
         {  /* xN[j] is free variable */
            temp = 0.0;
         }
         else if (rsm->tagn[j] == LP_NL)
         {  /* xN[j] is on its lower bound */
            if (rsm->ap[j] < 0.0) continue;
            temp = rsm->cbar[j] / rsm->ap[j];
         }
         else if (rsm->tagn[j] == LP_NU)
         {  /* xN[j] is on its upper bound */
            if (rsm->ap[j] > 0.0) continue;
            temp = rsm->cbar[j] / rsm->ap[j];
         }
         else if (rsm->tagn[j] == LP_NS)
         {  /* xN[j] is fixed variable */
            continue;
         }
         else
            assert(rsm->tagn[j] != rsm->tagn[j]);
         /* if reduced costs of xN[j] (i.e. dual variable) slightly
            violates its bound, temp is negative; in this case it is
            assumed that reduced cost is exactly on its bound (i.e.
            equal to zero), therefore temp is replaced by zero */
         if (temp < 0.0) temp = 0.0;
         /* apply minimal ratio test */
         if (teta > temp || teta == temp && big < fabs(rsm->ap[j]))
         {  rsm->q = j;
            teta = temp;
            big = fabs(rsm->ap[j]);
         }
      }
      /* restore original signs of the coefficients ap[j] */
      if (rsm->tagp == LP_NU)
         for (j = 1; j <= n; j++) rsm->ap[j] = - rsm->ap[j];
      return teta;
}

/* eof */
