/* glpgel/min_bump.c */

/*----------------------------------------------------------------------
-- This file is a part of the GLPK package.
--
-- Copyright (C) 2000, 2001 Andrew Makhorin <mao@mai2.rcnet.ru>,
--                          Department for Applied Informatics,
--                          Moscow Aviation Institute, Moscow, Russia.
--                          All rights reserved.
--
-- This code is free software; you can redistribute it and/or modify
-- it under the terms of the GNU General Public License as published by
-- the Free Software Foundation; either version 2 of the License, or
-- (at your option) any later version.
--
-- This software is distributed "as is" in the hope that it will be
-- useful, but WITHOUT ANY WARRANTY; without even the implied warranty
-- of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
-- General Public License for more details.
--
-- You should have received a copy of the GNU General Public License
-- along with this program; if not, write to the Free Software
-- Foundation, 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA.
----------------------------------------------------------------------*/

#include <assert.h>
#include <stddef.h>
#include "glpgel.h"
#include "glpset.h"

/*----------------------------------------------------------------------
-- min_bump - minimize bump size of given matrix (Reid's technique).
--
-- *Synopsis*
--
-- #include "glpgel.h"
-- void min_bump(PER *P, MAT *V, PER *Q, int *k1, int *k2,
--    DUFF *rs, DUFF *cs, int prev[], int next[]);
--
-- *Description*
--
-- The min_bump routine performs symmetric permutations of rows and
-- columns of the given matrix U = P*V*Q in order to minimize the size
-- of the bump of this matrix saving its special structure (see below).
-- The routine is based on the transformation proposed by J.Reid.
--
-- Actually the matrix V remains unchanged, since row and column
-- permutations are implicit and accumulated in permutation matrices
-- P and Q.
--
-- It is assumed that on entry the matrix U = P*V*Q has the following
-- special structure:
--
--     1   k1       k2   m
-- 1   x x x x x x x x x x
--     . x x x x x x x x x
-- k1  . . + * * * * * x x
--     . . + * * * * * x x
--     . . + . * * * * x x
--     . . + . . * * * x x
--     . . + . . . * * x x
-- k2  . . + . . . . * x x
--     . . . . . . . . x x
-- m   . . . . . . . . . x
--
-- where rows and columns k1, k1+1, ..., k2 form so called bump. So,
-- the matrix U differs from upper triangular matrix by k1-th column,
-- which is called spike. Additionally it is assumed that all diagonal
-- elements of the matrix U (except may be the element u[k1,k1]) are
-- non-zero. The element u[k2,k1] also should be non-zero (otherwise
-- the initial bump size may be decreased).
--
-- If in the bump there is a column which is not the spike and which
-- has only one non-zero (column singlet), this element is placed on
-- the main diagonal. Let u[k,k] is the column singlet. Then the routine
-- shifts rows and columns with numbers k1, k1+1, ..., k-1 by one
-- position respectively to the right and downwards, and places the row
-- and the column with number k to the position k1. This transformation
-- allows to decrease the bump size by one:
--
--     1   k1    k  k2   m        1     k1     k2   m
-- 1   x x x x x x x x x x    1   x x x x x x x x x x
--     . x x x x x x x x x        . x x x x x x x x x
-- k1  . . + * * . * * x x        . . s + . . * * x x
--     . . + * * . * * x x    k1  . . . + * * * * x x
--     . . + . * . * * x x        . . . + * * * * x x
-- k   . . + . . s * * x x        . . . + . * * * x x
--     . . + . . . * * x x        . . . + . . * * x x
-- k2  . . + . . . . * x x    k2  . . . + . . . * x x
--     . . . . . . . . x x        . . . . . . . . x x
-- m   . . . . . . . . . x    m   . . . . . . . . . x
--
-- Analogously, if in the bump there is a row which is not the row that
-- is symmetric to the spike and which has only one non-zero (row
-- singlet), this element is placed on the main diagonal. Let u[k,k] is
-- the row singlet. The the routine shifts rows and columns with numbers
-- k+1, k+2, ..., k2 by one position respectively to the left and
-- upwards, and places the row and the column with number k to the
-- position k2. This transformation again allows to decrease the bump
-- size by one:
--
--     1   k1    k  k2   m        1   k1     k2     m
-- 1   x x x x x x x x x x    1   x x x x x x x x x x
--     . x x x x x x x x x        . x x x x x x x x x
-- k1  . . + * * * * * x x    k1  . . + * * * * * x x
--     . . + * * * * * x x        . . + * * * * * x x
--     . . + . * * * * x x        . . + . * * * * x x
-- k   . . . . . s . . x x        . . + . . * * . x x
--     . . + . . . * * x x    k2  . . + . . . * . x x
-- k2  . . + . . . . * x x        . . . . . . . s x x
--     . . . . . . . . x x        . . . . . . . . x x
-- m   . . . . . . . . . x    m   . . . . . . . . . x
--
-- Removing row or column from the bump may involve new singlets to be
-- appeared, so the routine repeats these elementary transformations as
-- long as possible.
--
-- After the matrix U has been transformed the routine assigns to the
-- parameter k1 and k2 positions of the final bump. (The case k1 = k2
-- would mean that the matrix U has became fully upper triangular and
-- that all its diagonal elements, except may be the diagonal element
-- forming the final bump, are non-zero.)
--
-- The following details are important for efficient implementation.
-- Explicit row and column permutations of the matrix U (this matrix
-- has implicit representation U = P*V*Q) would be very expensive, since
-- one shift of row or column would require up to n transpositions (i.e.
-- up to n movements of elements of the matrices P and Q), where n is
-- the size of the current bump. Hence, the total time estimation would
-- be O(n^2). Therefore the min_bump routine performs all elementary
-- transformations on other matrix U', which differs from the matrix U
-- only by order of rows and columns. For representing the matrix U' the
-- routine doesn't use permutation matrices (like P and Q). Instead that
-- it uses double-linked list, elements of which are numbers of rows and
-- columns of the matrix U and the order of which is the same as the
-- order of rows and columns of the matrix U' (it's enough to have only
-- one linked list, because all permutations are symmetric). Initially
-- the matrix U' is identical to the matrix U. And after the matrix U
-- has been transformed this double-linked list is used to reorder rows
-- and columns of the matrix U. Such technique allows to perform any
-- movement of rows and columns (of the matrix U') for a fixed time,
-- because one movement is equivalent to one exclusion from and one
-- inclusion to the linked list, so in this case the time estimation is
-- O(nz), where nz is number of non-zeros.
--
-- To find row and column singlets for a fixed time the routine uses
-- Duff's search technique. The parameters rs and cs should specify Duff
-- schemes for holding lists of active (i.e. belonging to the current
-- bump) rows and columns. The initial contents of these schemes is not
-- essential; on exit these schemes will contain respectively numbers of
-- rows and columns of the matrix U (not V!), which belong to the final
-- (minimized) bump. These schemes may be created by the statements
-- rs = create_duff(n, n) and cs = create_duff(n, n), where n is the
-- order of the matrix U. To hold double-linked lists the routine also
-- uses two working array prev and next, which should have at least 1+n
-- elements.
--
-- *Reference*
--
-- Reid J.K. A Sparsity-Exploiting Variant of the Bartels-Golub
-- Decomposition for Linear Programming Bases. Math.Prog., 24, 1982,
-- pp. 55-69. */

#define iU(i) (P->col[i])
/* converts row number of V to row number of U */

#define iV(i) (P->row[i])
/* converts row number of U to row number of V */

#define jU(j) (Q->row[j])
/* converts column number of V to column number of U */

#define jV(j) (Q->col[j])
/* converts column number of U to column number of V */

void min_bump(PER *P, MAT *V, PER *Q, int *_k1, int *_k2,
      DUFF *rs, DUFF *cs, int prev[], int next[])
{     ELEM *e;
      int n = V->m, k1 = *_k1, k2 = *_k2, head, tail;
      int i, j, k, nz, t, t1, t2, t1new, t2new;
      int *old_iV = &iU(0), *old_jV = &jU(0);
      if (V->m != V->n)
         fault("min_bump: transformed matrix is not square");
      if (!(P->n == n && Q->n == n))
         fault("min_bump: permutation matrices have invalid order");
      if (!(1 <= k1 && k1 <= k2 && k2 <= n))
         fault("min_bump: initial bump has invalid position");
      /* build linked lists containing rows and columns of the matrix U
         (initially the matrix U' is identical to the matrix U) */
      head = 1, tail = n;
      for (t = 1; t <= n; t++)
         prev[t] = t-1, next[t] = (t < n ? t+1 : 0);
      /* build active lists containing rows and columns of the matrix U
         (initially these are rows and columns of the initial bump, but
         except the spike and the symmetric row, because their choice
         should be locked) */
      reset_duff(rs);
      for (i = k1+1; i <= k2; i++)
      {  nz = 0;
         for (e = V->row[iV(i)]; e != NULL; e = e->row)
         {  j = jU(e->j);
            if (j < k1)
err:           fault("min_bump: transformed matrix has invalid pattern")
                  ;
            if (j <= k2) nz++;
         }
         include_obj(rs, i, nz);
      }
      reset_duff(cs);
      for (j = k1+1; j <= k2; j++)
      {  nz = 0;
         for (e = V->col[jV(j)]; e != NULL; e = e->col)
         {  i = iU(e->i);
            if (i > k2) goto err;
            if (i >= k1) nz++;
         }
         include_obj(cs, j, nz);
      }
      /* row and column numbers t1 and t2 of the matrix U correspond to
         bounds k1 and k2 of the current bump of the matrix U' */
      t1 = k1, t2 = k2;
loop: /* main loop starts here */
      t = cs->head[1];
      if (t != 0)
      {  /* the element u[t,t] is in the bump and is a column singlet */
         for (e = V->col[jV(t)]; e != NULL; e = e->col)
         {  /* the active row is a sign of the singlet */
            i = iU(e->i);
            if (rs->len[i] >= 0) break;
         }
         if (!(e != NULL && i == t)) goto err;
         /* determine bounds of new bump */
         t1new = (t1 == t ? next[t1] : t1);
         t2new = (t2 == t ? prev[t2] : t2);
         /* t-th column and row of the matrix U should be removed from
            the bump and placed before t1-th row and column */
         if (t != t1)
         {  /* remove the element t from the linked list */
            if (prev[t] == 0)
               head = next[t];
            else
               next[prev[t]] = next[t];
            if (next[t] == 0)
               tail = prev[t];
            else
               prev[next[t]] = prev[t];
            /* insert the element t before the element t1 */
            prev[t] = prev[t1];
            next[t] = t1;
            if (prev[t1] == 0)
               head = t;
            else
               next[prev[t1]] = t;
            prev[t1] = t;
         }
         /* correct counts of active columns of the matrix U */
         for (e = V->row[iV(t)]; e != NULL; e = e->row)
         {  j = jU(e->j);
            nz = cs->len[j];
            if (nz >= 0)
               exclude_obj(cs, j), include_obj(cs, j, nz-1);
         }
         /* remove t-th row and column from active lists */
         exclude_obj(rs, t), exclude_obj(cs, t);
         /* continue the search */
         t1 = t1new, t2 = t2new, k1++;
         goto loop;
      }
      t = rs->head[1];
      if (t != 0)
      {  /* the element u[t,t] is in the bump and is a row singlet */
         for (e = V->row[iV(t)]; e != NULL; e = e->row)
         {  /* the active column is a sign of the singlet */
            j = jU(e->j);
            if (cs->len[j] >= 0) break;
         }
         if (!(e != NULL && j == t)) goto err;
         /* determine bounds of new bump */
         t1new = (t1 == t ? next[t1] : t1);
         t2new = (t2 == t ? prev[t2] : t2);
         /* t-th column and row of the matrix U should be removed from
            the bump and placed after t2-th row and column */
         if (t != t2)
         {  /* remove the element t from the linked list */
            if (prev[t] == 0)
               head = next[t];
            else
               next[prev[t]] = next[t];
            if (next[t] == 0)
               tail = prev[t];
            else
               prev[next[t]] = prev[t];
            /* insert the element t after the element t2 */
            prev[t] = t2;
            next[t] = next[t2];
            if (next[t2] == 0)
               tail = t;
            else
               prev[next[t2]] = t;
            next[t2] = t;
         }
         /* correct counts of active rows of the matrix U */
         for (e = V->col[jV(t)]; e != NULL; e = e->col)
         {  i = iU(e->i);
            nz = rs->len[i];
            if (nz >= 0)
               exclude_obj(rs, i), include_obj(rs, i, nz-1);
         }
         /* remove t-th row and column from active lists */
         exclude_obj(rs, t), exclude_obj(cs, t);
         /* continue the search */
         t1 = t1new, t2 = t2new, k2--;
         goto loop;
      }
      assert(tail == tail); /* relax compiler about never used */
      /* igitur, the matrix U' has desired form, where the order of
         rows and columns of the original matrix U that form the matrix
         ing U' is determined by the linked list; it allows performing
         implicit permutations of rows and columns of the matrix U in
         order to turn it into the matrix U' */
      for (i = 1; i <= n; i++) old_iV[i] = iV(i);
      for (k = 0, i = head; i != 0; i = next[i])
         k++, iV(k) = old_iV[i];
      for (j = 1; j <= n; j++) old_jV[j] = jV(j);
      for (k = 0, j = head; j != 0; j = next[j])
         k++, jV(k) = old_jV[j];
      for (k = 1; k <= n; k++) iU(iV(k)) = k, jU(jV(k)) = k;
      /* k1 and k2 are bounds of the final bump of the matrix U which
         now is identical to the matrix U' (k1 <= k2, moreover k1 = k2
         only if the final bump contains only one spike) */
      *_k1 = k1, *_k2 = k2;
      return;
}

/* eof */
