/* glpmat/ut_solve.c */

/*----------------------------------------------------------------------
-- This file is a part of the GNU LPK package.
--
-- Copyright (C) 2000 Andrew Makhorin <mao@mai2.rcnet.ru>,
--                    Department for Applied Informatics,
--                    Moscow Aviation Institute, Moscow, Russia.
--                    All rights reserved.
--
-- This code is free software; you can redistribute it and/or modify
-- it under the terms of the GNU General Public License as published by
-- the Free Software Foundation; either version 2 of the License, or
-- (at your option) any later version.
--
-- This software is distributed "as is" in the hope that it will be
-- useful, but WITHOUT ANY WARRANTY; without even the implied warranty
-- of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
-- General Public License for more details.
--
-- You should have received a copy of the GNU General Public License
-- along with this program; if not, write to the Free Software
-- Foundation, 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA.
----------------------------------------------------------------------*/

#include <stddef.h>
#include "glpmat.h"
#include "glpset.h"

/*----------------------------------------------------------------------
-- ut_solve - solve transposed upper triangular system U'*x = b.
--
-- *Synopsis*
--
-- #include "glpmat.h"
-- double *ut_solve(MAT *U, double x[]);
--
-- *Description*
--
-- The ut_solve routine solves the system U'*x = b, where U' is a matrix
-- transposed to the given upper triangular matrix U, x is dense vector
-- of unknowns, b is dense vector of right-hand sides. The matrix U
-- should be non-singular, i.e. all its diagonal elements should be
-- non-zeros. Before a call the array x should contain elements of the
-- vector b in locations x[1], x[2], ..., x[n], where n is the order of
-- the system. After a call this array will contain the vector x in the
-- same locations.
--
-- *Returns*
--
-- The ut_solve routine returns a pointer to the array x. */

double *ut_solve(MAT *U, double x[])
{     ELEM *e;
      int flag = 1, i;
      double piv;
      if (U->m != U->n)
         fault("ut_solve: matrix is not quadratic");
      for (i = 1; i <= U->m; i++)
      {  /* flag = 1 means that x[1] = ... = x[i-1] = 0; therefore if
            b[i] = 0 then x[i] = 0 */
         if (flag && x[i] == 0.0) continue;
         piv = 0.0;
         for (e = U->col[i]; e != NULL; e = e->col)
         {  if (e->i > i)
               fault("ut_solve: matrix is not upper triangular");
            if (e->i == i)
               piv = e->val;
            else
               x[i] -= e->val * x[e->i];
         }
         if (piv == 0.0)
            fault("ut_solve: matrix is singular");
         x[i] /= piv;
         if (x[i] != 0.0) flag = 0;
      }
      return x;
}

/* eof */
