/* glpmat/aat_symb.c */

/*----------------------------------------------------------------------
-- This file is a part of the GNU LPK package.
--
-- Copyright (C) 2000 Andrew Makhorin <mao@mai2.rcnet.ru>,
--                    Department for Applied Informatics,
--                    Moscow Aviation Institute, Moscow, Russia.
--                    All rights reserved.
--
-- This code is free software; you can redistribute it and/or modify
-- it under the terms of the GNU General Public License as published by
-- the Free Software Foundation; either version 2 of the License, or
-- (at your option) any later version.
--
-- This software is distributed "as is" in the hope that it will be
-- useful, but WITHOUT ANY WARRANTY; without even the implied warranty
-- of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
-- General Public License for more details.
--
-- You should have received a copy of the GNU General Public License
-- along with this program; if not, write to the Free Software
-- Foundation, 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA.
----------------------------------------------------------------------*/

#include <stddef.h>
#include "glpmat.h"
#include "glpset.h"

/*----------------------------------------------------------------------
-- aat_symb - compute matrix product S = A * A'; symbolic phase
--
-- *Synopsis*
--
-- #include "glpmat.h"
-- MAT *aat_symb(MAT *S, MAT *A, char work[]);
--
-- *Description*
--
-- The aat_symb computes the matrix product S = A * A' symbolically,
-- i.e. it only determines distribution of non-zeros of the matrix S not
-- computing their numeric values.
--
-- Input matrix A remains unchanged on exit. Numeric values of elements
-- of this matrix are ignored (they all are considered as non-zeros).
--
-- The resultant matrix S should not coincide with the matrix A. Order
-- of S should be equal number of rows of A. On entry the contents of S
-- is ignored. So far as S is symmetric, the routine computes only upper
-- triangular part of the matrix S.
--
-- The auxiliary array work should contain at least 1+m elements, where
-- m is order of S (number of rows of A). If this parameter is NULL, the
-- routine automatically allocates and frees the working array.
--
-- *Returns*
--
-- The aat_symb routine returns a pointer to the matrix S. */

MAT *aat_symb(MAT *S, MAT *A, char _work[])
{     ELEM *e, *ee;
      int i, j, k;
      char *work = _work;
      if (S == A)
         fault("aat_symb: invalid specification of resultant matrix");
      if (!(S->m == S->n && S->m == A->m))
         fault("aat_symb: inconsistent dimension; product undefined");
      clear_mat(S);
      if (_work == NULL) work = ucalloc(1+S->n, sizeof(char));
      for (j = 1; j <= S->n; j++) work[j] = 0;
      for (i = 1; i <= S->m; i++)
      {  /* compute i-th row of S */
         for (e = A->row[i]; e != NULL; e = e->row)
         {  k = e->j;
            for (ee = A->col[k]; ee != NULL; ee = ee->col)
            {  j = ee->i;
               /* a[i,k] != 0 and a[j,k] != 0 therefore s[i,j] != 0 */
               if (i > j) continue; /* skip subdiagonal element */
               if (work[j] == 0) new_elem(S, i, j, 1.0), work[j] = 1;
            }
         }
         /* clear working array */
         for (e = S->row[i]; e != NULL; e = e->row) work[e->j] = 0;
      }
      if (_work == NULL) ufree(work);
      return S;
}

/* eof */
