/* glpavl/rotate_avl.c */

/*----------------------------------------------------------------------
-- This file is a part of the GNU LPK package.
--
-- Copyright (C) 2000 Andrew Makhorin <mao@mai2.rcnet.ru>,
--                    Department for Applied Informatics,
--                    Moscow Aviation Institute, Moscow, Russia.
--                    All rights reserved.
--
-- This code is free software; you can redistribute it and/or modify
-- it under the terms of the GNU General Public License as published by
-- the Free Software Foundation; either version 2 of the License, or
-- (at your option) any later version.
--
-- This software is distributed "as is" in the hope that it will be
-- useful, but WITHOUT ANY WARRANTY; without even the implied warranty
-- of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
-- General Public License for more details.
--
-- You should have received a copy of the GNU General Public License
-- along with this program; if not, write to the Free Software
-- Foundation, 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA.
----------------------------------------------------------------------*/

#include <stddef.h>
#include "glpavl.h"

/*----------------------------------------------------------------------
-- rotate_avl - restore correct balance of AVL-subtree.
--
-- *Synopsis*
--
-- #include "glpavl.h"
-- AVLNODE *rotate_avl(AVLTREE *tree, AVLNODE *node);
--
-- The rotate_avl routine restores the correct balance of the subtree
-- specified by its root which node points to. It's assumed that the
-- converted subtree belongs to the AVL-tree specified by the parameter
-- tree.
--
-- The rotate_avl is auxiliary routine and not intended for independent
-- usage.
--
-- *Returns*
--
-- The rotate_avl routine returns a pointer to the new root node of the
-- converted subtree. */

AVLNODE *rotate_avl(AVLTREE *tree, AVLNODE *node)
{     AVLNODE *f, *p = node, *q, *r, *x, *y;
      /* determine the kind of rotation */
      if (p->bal < 0) goto neg; else goto pos;
neg:  /* negative (left) rotation is needed */
      f = p->up; q = p->left; r = q->right;
      if (q->bal <= 0) goto neg1; else goto neg2;
neg1: /* perform single negative rotation */
      if (f == NULL)
         tree->root = q;
      else
         if (p->flag == 0) f->left = q; else f->right = q;
      p->rank -= q->rank;
      q->up = f; q->flag = p->flag; q->bal++; q->right = p;
      p->up = q; p->flag = 1;
      p->bal = (short int)(-q->bal); p->left = r;
      if (r != NULL) r->up = p, r->flag = 0;
      return q;
neg2: /* perform double negative rotation */
      x = r->left; y = r->right;
      if (f == NULL)
         tree->root = r;
      else
         if (p->flag == 0) f->left = r; else f->right = r;
      p->rank -= (q->rank + r->rank);
      r->rank += q->rank;
      p->bal = (short int)(r->bal >= 0 ? 0 : +1);
      q->bal = (short int)(r->bal <= 0 ? 0 : -1);
      r->up = f; r->flag = p->flag; r->bal = 0;
      r->left = q; r->right = p;
      p->up = r; p->flag = 1; p->left = y;
      q->up = r; q->flag = 0; q->right = x;
      if (x != NULL) x->up = q, x->flag = 1;
      if (y != NULL) y->up = p, y->flag = 0;
      return r;
pos:  /* positive (right) rotation is needed */
      f = p->up; q = p->right; r = q->left;
      if (q->bal >= 0) goto pos1; else goto pos2;
pos1: /* perform single positive rotation */
      if (f == NULL)
         tree->root = q;
      else
         if (p->flag == 0) f->left = q; else f->right = q;
      q->rank += p->rank;
      q->up = f; q->flag = p->flag; q->bal--; q->left = p;
      p->up = q; p->flag = 0;
      p->bal = (short int)(-q->bal); p->right = r;
      if (r != NULL) r->up = p, r->flag = 1;
      return q;
pos2: /* perform double positive rotation */
      x = r->left; y = r->right;
      if (f == NULL)
         tree->root = r;
      else
         if (p->flag == 0) f->left = r; else f->right = r;
      q->rank -= r->rank;
      r->rank += p->rank;
      p->bal = (short int)(r->bal <= 0 ? 0 : -1);
      q->bal = (short int)(r->bal >= 0 ? 0 : +1);
      r->up = f; r->flag = p->flag; r->bal = 0;
      r->left = p; r->right = q;
      p->up = r; p->flag = 0; p->right = x;
      q->up = r; q->flag = 1; q->left = y;
      if (x != NULL) x->up = p, x->flag = 1;
      if (y != NULL) y->up = q, y->flag = 0;
      return r;
}

/* eof */
