/* glpapi/glp_find_item.c */

/*----------------------------------------------------------------------
-- This file is a part of the GNU LPK package.
--
-- Copyright (C) 2000 Andrew Makhorin <mao@mai2.rcnet.ru>,
--                    Department for Applied Informatics,
--                    Moscow Aviation Institute, Moscow, Russia.
--                    All rights reserved.
--
-- This code is free software; you can redistribute it and/or modify
-- it under the terms of the GNU General Public License as published by
-- the Free Software Foundation; either version 2 of the License, or
-- (at your option) any later version.
--
-- This software is distributed "as is" in the hope that it will be
-- useful, but WITHOUT ANY WARRANTY; without even the implied warranty
-- of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
-- General Public License for more details.
--
-- You should have received a copy of the GNU General Public License
-- along with this program; if not, write to the Free Software
-- Foundation, 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA.
----------------------------------------------------------------------*/

#include <stddef.h>
#include "glpapi.h"

/*----------------------------------------------------------------------
-- glp_find_item - find row or column by name.
--
-- *Synopsis*
--
-- #include "glpk.h"
-- int glp_find_item(int what, char *name);
--
-- *Description*
--
-- If what = GLP_ROW, the glp_find_item routine finds the row with the
-- given name and makes it be the current row.
--
-- If what = GLP_COL, the glp_find_item routine finds the column with
-- the given name and makes it be the current column.
--
-- *Complexity*
--
-- This operation has time complexity O(log n), where n is the current
-- number of rows or columns in the workspace.
--
-- *Returns*
--
-- 0 - no errors;
-- 1 - row/column with the given name doesn't exist;
-- 2 - the parameter what or the given name is invalid. */

int glp_find_item(int what, char *name)
{     if (what == GLP_ROW)
      {  /* find row */
         STR *row_name;
         AVLNODE *node;
         if (glp_check_name(name)) return 2;
         row_name = set_str(create_str(glp->str_pool), name);
         node = find_by_key(glp->row_tab, row_name);
         delete_str(row_name);
         if (node == NULL) return 1;
         glp->this_row = node->link;
      }
      else if (what == GLP_COL)
      {  /* find column */
         STR *col_name;
         AVLNODE *node;
         if (glp_check_name(name)) return 2;
         col_name = set_str(create_str(glp->str_pool), name);
         node = find_by_key(glp->col_tab, col_name);
         delete_str(col_name);
         if (node == NULL) return 1;
         glp->this_col = node->link;
      }
      else
      {  /* invalid parameter */
         return 2;
      }
      return 0;
}

/* eof */
