;;; erc-track.el --- Track modified channel buffers

;; Copyright (C) 2002  Mario Lang <mlang@delysid.org>

;; Author: Mario Lang <mlang@delysid.org>
;; Keywords: comm, faces
;; URL: http://www.emacswiki.org/cgi-bin/wiki.pl?ErcChannelTracking

;; This file is free software; you can redistribute it and/or modify
;; it under the terms of the GNU General Public License as published by
;; the Free Software Foundation; either version 2, or (at your option)
;; any later version.

;; This file is distributed in the hope that it will be useful,
;; but WITHOUT ANY WARRANTY; without even the implied warranty of
;; MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
;; GNU General Public License for more details.

;; You should have received a copy of the GNU General Public License
;; along with GNU Emacs; see the file COPYING.  If not, write to
;; the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
;; Boston, MA 02111-1307, USA.

;;; Commentary:

;; Highlights keywords and pals (friends), and hides or hilights fools
;; (using a dark color).  Add to your ~/.emacs:

;; (require 'erc-track)
;; (erc-track-mode 1)

;; Todo:
;; * Add extensibility so that custom functions can track
;;   custom modification types.

(require 'erc)
(require 'erc-compat)

;;; Code:

(defconst erc-track-version "$Revision: 1.27 $"
  "ERC track mode revision")

(defcustom erc-track-exclude nil
  "A list targets (channel names or query targets) which should not be tracked."
  :group 'erc
  :type '(repeat string))

(defcustom erc-track-exclude-types '("NICK")
  "*List of message types to be ignored.
This list could look like '(\"JOIN\" \"PART\")."
  :group 'erc
  :type '(repeat string))

(defcustom erc-track-shorten-start 1
  "This number specifies the minimum number of characters a channel name in
the mode-line should be reduced to."
  :group 'erc
  :type 'number)

(defcustom erc-track-shorten-cutoff 4
  "All channel names longer than this value will be shortened."
  :group 'erc
  :type 'number)

(defcustom erc-track-shorten-aggressively nil
  "*If non-nil, channel names will be shortened more aggressively.
Usually, names are not shortened if this will save only one character.
Example: If there are two channels, #linux-de and #linux-fr, then
normally these will not be shortened.  When shortening aggressively,
however, these will be shortened to #linux-d and #linux-f.

If this variable is set to `max', then channel names will be shortened
to the max.  Usually, shortened channel names will remain unique for a
given set of existing channels.  When shortening to the max, the shortened
channel names will be unique for the set of active channels only.
Example: If there are tow active channels #emacs and #vi, and two inactive
channels #electronica and #folk, then usually the active channels are
shortened to #em and #v.  When shortening to the max, however, #emacs is
not compared to #electronica -- only to #vi, therefore it can be shortened
even more and the result is #e and #v.

This setting is used by `erc-track-shorten-names'."
  :group 'erc
  :type '(choice (const :tag "No" nil)
		 (const :tag "Yes" t)
		 (const :tag "Max" max)))

(defcustom erc-track-shorten-function 'erc-track-shorten-names
  "*This function will be used to reduce the channel names before display.
It takes one argument, CHANNEL-NAMES which is a list of strings.
It should return a list of strings of the same number of elements."
  :group 'erc
  :type 'function)

(defcustom erc-track-use-faces t
  "*If non-nil, use faces to indicate current nick/pal/fool/keyword/dangerous
host activities in the mode-line.
The faces used are the same as used for text in the buffers.
\(e.g. `erc-pal-face' is used if a pal sent a message to that channel.)"
  :group 'erc
  :type 'boolean)

(defvar erc-modified-channels-string ""
  "Internal string used for displaying modified channels in the mode line.")

(defvar erc-modified-channels-alist nil
  "An ALIST used for tracking channel modification activity. Each element
looks like (BUFFER FACE...) where BUFFER is a buffer object of the channel
the entry corresponds to, followed by zero or more faces.

Entries in this list should only happen for buffers where activity occured
while the buffer was not visible.")

;;; Shortening of names

(defun erc-track-shorten-names (channel-names)
  "Call `erc-unique-channel-names' with the correct parameters.
This function is a good value for `erc-track-shorten-function'.
The list of all channels is returned by `erc-all-buffers'.
CHANNEL-NAMES is the list of active channel names.
Only channel names longer than `erc-track-shorten-cutoff' are
actually shortened, and they are only shortened to a minimum
of `erc-track-shorten-start' characters."
  (erc-unique-channel-names
   (erc-all-buffer-names)
   channel-names
   (lambda (s)
     (> (length s) erc-track-shorten-cutoff))
   erc-track-shorten-start))

(defvar erc-default-recipients)

(defun erc-all-buffer-names ()
  "Return all channel or query buffer names.
Note that we cannot use `erc-channel-list' with a nil argument,
because that does not return query buffers."
  (save-excursion
    (let (result)
      (dolist (buf (buffer-list))
	(set-buffer buf)
	(when erc-default-recipients
	  (setq result (cons (buffer-name) result))))
      result)))

(defun erc-unique-channel-names (all active &optional predicate start)
  "Return a list of unique channel names.
ALL is the list of all channel and query buffer names.
ACTIVE is the list of active buffer names.
PREDICATE is a predicate that should return non-nil if a name needs
  no shortening.
START is the minimum length of the name used."
  (if (eq 'max erc-track-shorten-aggressively)
      ;; return the unique substrings of all active channels
      (erc-unique-substrings active predicate start)
    ;; else determine the unique substrings of all channels, and for
    ;; every active channel, return the corresponding substring.
    (let ((all-substrings (erc-unique-substrings all predicate start))
	  result)
      (dolist (channel active)
	(let ((substrings all-substrings)
	      candidate
	      winner)
	  (while (and substrings (not winner))
	    (setq candidate (car substrings)
		  substrings (cdr substrings))
	    (when (string= candidate (substring channel 0 (length candidate)))
	      (setq winner candidate)))
	  (setq result (cons winner result))))
      (nreverse result))))

(defun erc-unique-substrings (strings &optional predicate start)
  "Return a list of unique substrings of STRINGS."
  (if (or (not (numberp start))
	  (< start 0))
      (setq start 0))
  (mapcar
   (lambda (str)
     (let* ((others (delete str (copy-sequence strings)))
	    (maxlen (length str))
	    (i start)
	    candidate
	    done)
       (if (and (functionp predicate) (not (funcall predicate str)))
	   ;; do not shorten if a predicate exists and it returns nil
	   str
	 ;; Start with smallest substring candidate, ie. length 1.
	 ;; Then check all the others and see wether any of them starts
	 ;; with the same substring.  While there is such another
	 ;; element in the list, increase the length of the candidate.
	 (while (not done)
	   (if (> i maxlen)
	       (setq done t)
	     (setq candidate (substring str 0 i)
		   done (not (erc-unique-substring-1 candidate others))))
	   (setq i (1+ i)))
	 (if (and (= (length candidate) (1- maxlen))
		  (not erc-track-shorten-aggressively))
	     str
	   candidate))))
   strings))

(defun erc-unique-substring-1 (candidate others)
  "Return non-nil when any string in OTHERS starts with CANDIDATE."
  (let (result other (maxlen (length candidate)))
    (while (and others
		(not result))
      (setq other (car others)
	    others (cdr others))
      (when (and (>= (length other) maxlen)
		 (string= candidate (substring other 0 maxlen)))
	(setq result other)))
    result))

;;; Test:

(assert
 (and
  ;; verify examples from the doc strings
  (equal (let ((erc-track-shorten-aggressively nil))
	   (erc-unique-channel-names
	    '("#emacs" "#vi" "#electronica" "#folk")
	    '("#emacs" "#vi")))
	 '("#em" "#vi")); emacs is different from electronica
  (equal (let ((erc-track-shorten-aggressively t))
	   (erc-unique-channel-names
	    '("#emacs" "#vi" "#electronica" "#folk")
	    '("#emacs" "#vi")))
	 '("#em" "#v")); vi is shortened by one letter
  (equal (let ((erc-track-shorten-aggressively 'max))
	   (erc-unique-channel-names
	    '("#emacs" "#vi" "#electronica" "#folk")
	    '("#emacs" "#vi")))
	 '("#e" "#v")); emacs need not be different from electronica
  (equal (let ((erc-track-shorten-aggressively nil))
	   (erc-unique-channel-names
	    '("#linux-de" "#linux-fr")
	    '("#linux-de" "#linux-fr")))
	 '("#linux-de" "#linux-fr")); shortening by one letter is too aggressive
  (equal (let ((erc-track-shorten-aggressively t))
	   (erc-unique-channel-names
	    '("#linux-de" "#linux-fr")
	    '("#linux-de" "#linux-fr")))
	 '("#linux-d" "#linux-f")); now we want to be agressive
  ;; specific problems
  (equal (let ((erc-track-shorten-aggressively nil))
	   (erc-unique-substrings
	    '("#emacs" "#vi" "#electronica" "#folk")))
	 '("#em" "#vi" "#el" "#f"))
  (equal (let ((erc-track-shorten-aggressively t))
	   (erc-unique-substrings
	    '("#emacs" "#vi" "#electronica" "#folk")))
	 '("#em" "#v" "#el" "#f"))
  (equal (let ((erc-track-shorten-aggressively nil))
	   (erc-unique-channel-names
	    '("#emacs" "#burse" "+linux.de" "#starwars"
	      "#bitlbee" "+burse" "#ratpoison")
	    '("+linux.de" "#starwars" "#burse")))
	 '("+l" "#s" "#bu"))
  (equal (let ((erc-track-shorten-aggressively nil))
	   (erc-unique-channel-names
	    '("fsbot" "#emacs" "deego")
	    '("fsbot")))
	 '("f"))
  (equal (let ((erc-track-shorten-aggressively nil))
	   (erc-unique-channel-names
	    '("fsbot" "#emacs" "deego")
	    '("fsbot")
	    (lambda (s)
	      (> (length s) 4))
	    1))
	 '("f"))
  (equal (let ((erc-track-shorten-aggressively nil))
	   (erc-unique-channel-names
	    '("fsbot" "#emacs" "deego")
	    '("fsbot")
	    (lambda (s)
	      (> (length s) 4))
	    2))
	 '("fs"))
  ;; general examples
  (let ((erc-track-shorten-aggressively t))
    (and (equal (erc-unique-substring-1 "abc" '("ab" "abcd")) "abcd")
	 (not (erc-unique-substring-1 "a" '("xyz" "xab")))
	 (equal (erc-unique-substrings '("abc" "xyz" "xab")) 
		'("a" "xy" "xa"))
	 (equal (erc-unique-substrings '("abc" "abcdefg"))
		'("abc" "abcd"))))
  (let ((erc-track-shorten-aggressively nil))
    (and (equal (erc-unique-substring-1 "abc" '("ab" "abcd")) "abcd")
	 (not (erc-unique-substring-1 "a" '("xyz" "xab")))
	 (equal (erc-unique-substrings '("abc" "xyz" "xab")) 
		'("a" "xyz" "xab"))
	 (equal (erc-unique-substrings '("abc" "abcdefg")) 
		'("abc" "abcd"))))))

;;; Module

;;;###autoload (autoload 'erc-track-mode "erc-track")
(define-erc-module track track-modified-channels
  "This mode tracks ERC channel buffers with activity."
  ((or global-mode-string
       (setq global-mode-string '("")))
   (or (memq 'erc-modified-channels-string global-mode-string)
       (setq global-mode-string
	     (append global-mode-string
		     '(erc-modified-channels-string))))
   (setq erc-modified-channels-string "")
   (erc-update-mode-line)
   (add-hook 'erc-insert-post-hook
	     'erc-track-modified-channels)
   (add-hook 'window-configuration-change-hook
	     'erc-modified-channels-update))
  ((setq global-mode-string
	 (delq 'erc-modified-channels-string global-mode-string))
   (remove-hook 'erc-insert-post-hook
		'erc-track-modified-channels)
   (remove-hook 'window-configuration-change-hook
		'erc-modified-channels-update)))

;;; Tracking the channel modifications

(defvar erc-modified-channels-update-inside nil
  "Variable to prevent running `erc-modified-channels-update' multiple
times.  Without it, you cannot debug `erc-modified-channels-display',
because the debugger also cases changes to the window-configuration.")

(defun erc-modified-channels-update ()
  "This function updates the information in `erc-modified-channels-alist'
according to buffer visibility. It calls `erc-modified-channels-display' at the
end. This should usually be called via `window-configuration-change-hook'."
  (interactive)
  (unless erc-modified-channels-update-inside
    (let ((erc-modified-channels-update-inside t))
      (mapcar (lambda (elt)
		(let ((buffer (car elt)))
		  (when (or (not (bufferp buffer))
			    (not (buffer-live-p buffer))
			    (get-buffer-window buffer t))
		    (erc-modified-channels-remove-buffer buffer))))
	      erc-modified-channels-alist)
      (erc-modified-channels-display))))
    
(defun erc-make-mode-line-buffer-name (string buffer &optional faces)
  "Return STRING as a button that switches to BUFFER when clicked.
If FACES are provided, color STRING with them."
  (let ((map (make-sparse-keymap))
	(name (copy-sequence string)))
    (define-key map (vector 'mode-line 'mouse-2)
      `(lambda (e)
	 (interactive "e")
	 (save-selected-window
	   (select-window
	    (posn-window (event-start e)))
	   (switch-to-buffer ,buffer))))
    (put-text-property 0 (length name) 'local-map map name)
    (when (and faces erc-track-use-faces)
      (put-text-property 0 (length name) 'face faces name))
    name))

(defun erc-modified-channels-display ()
  "Set `erc-modified-channels-string'
according to `erc-modified-channels-alist'.
Use `erc-make-mode-line-buffer-name' to create buttons."
  (if (null erc-modified-channels-alist)
      (setq erc-modified-channels-string "")
    ;; erc-modified-channels-alist is an alist with buffers in the
    ;; CAR.  We use alist to point to its the elements.  We extract
    ;; the buffer-names (bufnames), shorten them (tmplist), and create
    ;; a new-list, where the CAR is a shortened buffer-name (from
    ;; templist), and the CDR is the buffer, followed by faces (from
    ;; alist).
    (let* ((bufnames (mapcar (lambda (x) (buffer-name (car x)))
			    erc-modified-channels-alist))
	   (tmplist (if (functionp erc-track-shorten-function)
			(funcall erc-track-shorten-function bufnames)
		      bufnames))
	   (alist erc-modified-channels-alist)
	   new-list)
      (while alist
	(setq new-list (cons (cons (car tmplist)
				   (car alist))
			     new-list)
	      alist (cdr alist)
	      tmplist (cdr tmplist)))
      (setq new-list (nreverse new-list))
      (setq erc-modified-channels-string
	    (concat "["
		    (mapconcat (lambda (elt)
				 (erc-make-mode-line-buffer-name
				  (car elt) (nth 1 elt) (nthcdr 2 elt)))
			       new-list ",")
		    "] ")))))

(defun erc-modified-channels-remove-buffer (buffer)
  "Remove BUFFER from `erc-modified-channels-alist'."
  (interactive "bBuffer: ")
  (setq erc-modified-channels-alist
	(delete (assq buffer erc-modified-channels-alist)
		erc-modified-channels-alist))
  (when (interactive-p)
    (erc-modified-channels-display)))

(defun erc-track-modified-channels ()
  "Hook function for `erc-insert-post-hook' to check if the current
buffer should be added to the modeline as a hidden, modified
channel.  Assumes it will only be called when current-buffer
is in `erc-mode'."
  (let ((this-channel (erc-default-target)))
    (if (and (not (get-buffer-window (current-buffer) t))
	     this-channel
	     (not (member this-channel erc-track-exclude))
	     (not (erc-message-type-member (point-min)
					   erc-track-exclude-types)))
	;; If the active buffer is not visible (not shown in a
	;; window), and not to be excluded, determine the kinds of
	;; faces used in the current message, and add the buffer to
	;; the erc-modified-channels-alist, if it is not already
	;; there.  If it the buffer is already on the list, add to the
	;; list of faces to reflect the latest values.
	(let ((faces (erc-faces-in (buffer-string ))))
	  (if (not (assq (current-buffer) erc-modified-channels-alist))
	      ;; Add buffer and faces
	      (setq erc-modified-channels-alist
		    (cons (cons (current-buffer) faces)
			  erc-modified-channels-alist))
	    ;; Else modify the list of faces for the buffer, if new
	    ;; faces appeared.
	    (when faces
 	      (let* ((cell (assq (current-buffer)
				 erc-modified-channels-alist))
 		     (value (cdr cell)))
 		(dolist (face faces)
 		  (add-to-list 'value face))
 		(setcdr cell value))))
	  ;; And display it
	  (erc-modified-channels-display))
      ;; Else if the active buffer is the current buffer, remove it
      ;; from our list.
      (when (or (get-buffer-window (current-buffer) t)
		(and this-channel
		     (assq (current-buffer) erc-modified-channels-alist)
		     (member this-channel erc-track-exclude)))
	;; Remove it from mode-line if buffer is visible or
	;; channel was added to erc-track-exclude recently.
	(erc-modified-channels-remove-buffer (current-buffer))
	(erc-modified-channels-display)))))

(defun erc-faces-in (str)
  "Return a list of all faces used in STR."
  (let ((i 0)
	(m (length str))
	(faces (erc-list (get-text-property 0 'face str))))
    (while (and (setq i (next-single-property-change i 'face str m))
		(not (= i m)))
      (dolist (face (erc-list (get-text-property i 'face str)))
	(add-to-list 'faces face)))
    faces))

(assert
 (let ((str "is bold"))
   (put-text-property 3 (length str)
		      'face '(bold erc-current-nick-face)
		      str)
   (erc-faces-in str)))

;;; Buffer switching

(defcustom erc-track-switch-from-erc t
  "If non-nil, `erc-track-switch-buffer' will return to the last non-erc buffer
when there are no more active channels."
  :type 'boolean
  :group 'erc-track)

(defcustom erc-track-switch-direction 'oldest
  "Direction `erc-track-switch-buffer' should switch.
'oldest will find the oldest active buffer.
'newest finds the latest."
  :group 'erc-track
  :type '(choice (const oldest) (const newest)))

(defvar erc-track-last-non-erc-buffer nil
  "Stores the name of the last buffer you were in before activating
`erc-track-switch-buffers'")

(defun erc-track-get-active-buffer (arg)
  "Return the buffer name of `arg' in `erc-modified-channels-alist'.
Negative arguments index in the opposite direction.  This direction is
relative to `erc-track-switch-direction'"
  (let ((dir erc-track-switch-direction)
        offset)
    (if (< arg 0)
        (progn
          (cond
           ((eq 'oldest dir) (setq dir 'newest))
           ((eq 'newest dir) (setq dir 'oldest)))
          (setq arg (* -1 arg))))

    (setq arg (- arg 1))
    (setq offset (cond
                  ((eq 'newest dir)
                   (+ arg 0))
                  ((eq 'oldest dir)
                   (- (- (length erc-modified-channels-alist) 1) arg))
                  (t 0)))
    ;; normalise out of range user input
    (if (>= offset (length erc-modified-channels-alist))
        (setq offset (- (length erc-modified-channels-alist) 1))
    (if (< offset 0)
        (setq offset 0))

    (car (nth offset erc-modified-channels-alist)))))

(defun erc-track-switch-buffer (arg)
  "Switch to the next active ERC buffer, or if there are no active buffers,
switch back to the last non-ERC buffer visited.  Next is defined by
`erc-track-switch-direction', a negative argument will reverse this."
  (interactive "p")
  (when erc-track-mode
    (let ((dir erc-track-switch-direction))
      (if erc-modified-channels-alist
	  (progn
	    ;; if we're not in erc-mode, set this buffer to return to
	    (unless (eq major-mode 'erc-mode)
	      (setq erc-track-last-non-erc-buffer (current-buffer)))
	    ;; and jump to the next active channel
	    (switch-to-buffer (erc-track-get-active-buffer arg)))
	;; if no active channels, switch back to what we were doing before
	(when (and erc-track-last-non-erc-buffer
		   erc-track-switch-from-erc
		   (buffer-live-p erc-track-last-non-erc-buffer))
	  (switch-to-buffer erc-track-last-non-erc-buffer))))))

(global-set-key (kbd "C-c C-@") 'erc-track-switch-buffer)
(global-set-key (kbd "C-c C-SPC") 'erc-track-switch-buffer)

(provide 'erc-track)

;;; erc-track.el ends here
