;;; erc-autoaway.el --- Provides autoaway for ERC

;; Copyright (C) 2002  Free Software Foundation, Inc.

;; Author: Jorgen Schaefer <forcer@forcix.cx>
;; URL: http://www.emacswiki.org/cgi-bin/wiki.pl?ErcAutoAway

;; This file is free software; you can redistribute it and/or modify
;; it under the terms of the GNU General Public License as published by
;; the Free Software Foundation; either version 2, or (at your option)
;; any later version.

;; This file is distributed in the hope that it will be useful,
;; but WITHOUT ANY WARRANTY; without even the implied warranty of
;; MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
;; GNU General Public License for more details.

;; You should have received a copy of the GNU General Public License
;; along with GNU Emacs; see the file COPYING.  If not, write to
;; the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
;; Boston, MA 02111-1307, USA.

;;; Commentary:

;; TODO:
;; - Legacy names: erc-auto-discard-away, erc-auto-set-away

;;; Code:

(require 'erc)

(defvar erc-autoaway-idletimer nil
  "The emacs idletimer.")

;;;###autoload (autoload 'erc-autoaway-mode "erc-autoaway")
(define-erc-module autoaway nil
  "In ERC autoaway mode, the client can set you automatically away
after some specific idletime, and set you back when you type
something."
  ;; Enable:
  ((add-hook 'erc-send-completed-hook 'erc-autoaway-reset-idletime)
   (add-hook 'erc-timer-hook 'erc-autoaway-possibly-set-away)
   (erc-autoaway-reestablish-idletimer))
  ;; Disable:
  ((remove-hook 'erc-send-completed-hook 'erc-autoaway-reset-idletime)
   (remove-hook 'erc-timer-hook 'erc-autoaway-possibly-set-away)
   (when erc-autoaway-idletimer
     (cancel-timer erc-autoaway-idletimer)
     (setq erc-autoaway-idletimer nil))))

(defgroup erc-autoaway nil
  "Set yourself automatically away after some idletime and set
yourself back when you type something."
  :group 'erc)

(defcustom erc-autoaway-idle-seconds 1800
  "*Number of seconds after which ERC will set you automatically away.
If `erc-autoaway-use-emacs-idle' is non-nil, you have to run
`erc-autoaway-reestablish-idletimer' after changing this variable."
  :group 'erc-autoaway
  :type 'number)

(defcustom erc-autoaway-message "I'm gone (autoaway after %i seconds of idletime)"
  "*Message ERC will use when he sets you automatically away. It's
used as a `format' string with the argument of the idletime in
seconds."
  :group 'erc-autoaway
  :type 'string)

(defcustom erc-auto-set-away nil
  "*If non-nil, set away after `erc-autoaway-idle-seconds' seconds of
idletime in ERC."
  :group 'erc-autoaway
  :type 'boolean)

(defcustom erc-autoaway-use-emacs-idle nil
  "*If non-nil, set away after Emacs has been idle for
`erc-autoaway-idle-seconds' seconds."
  :group 'erc-autoaway
  :type 'boolean)

(defcustom erc-auto-discard-away t
  "*If non-nil, sending anything when away automatically discards away state."
  :group 'erc-autoaway
  :type 'boolean)

(defvar erc-autoaway-last-sent-time (erc-current-time)
  "The last time the user sent something.")

(defun erc-autoaway-reset-idletime (l)
  "Reset the stored idletime for the user. This is one global variable
since a user talking on one net can talk on another net too."
  (when erc-auto-discard-away
    (erc-autoaway-set-back l))
  (setq erc-autoaway-last-sent-time (erc-current-time)))

(defun erc-autoaway-reestablish-idletimer ()
  "Reestablish the emacs idletimer. You have to call this function
each time you change `erc-autoaway-idle-seconds."
  (interactive)
  (when erc-autoaway-idletimer
    (cancel-timer erc-autoaway-idletimer))
  (setq erc-autoaway-idletimer
	(run-with-idle-timer erc-autoaway-idle-seconds
			     t
			     'erc-autoaway-set-away
			     erc-autoaway-idle-seconds)))

(defun erc-autoaway-set-back (l)
  (when (erc-away-p)
    (setq erc-autoaway-last-sent-time (erc-current-time))
    (erc-cmd-GAWAY "" t)))

(defun erc-autoaway-possibly-set-away (current-time)
  "Set autoaway when `erc-auto-set-away' is true and the idletime is
exceeds `erc-autoaway-idle-seconds'."
  (let ((idle-time (erc-time-diff erc-autoaway-last-sent-time current-time)))
    (when (and erc-auto-set-away
               (>= idle-time erc-autoaway-idle-seconds)
               (not (erc-away-p)))
      (erc-display-message
       nil 'notice nil
       (format "Setting automatically away after %i seconds of idle-time"
               idle-time))
      (erc-autoaway-set-away idle-time))))

(defun erc-autoaway-set-away (idle-time)
  (when (not (erc-away-p))
    (erc-cmd-GAWAY (format erc-autoaway-message idle-time))))

(provide 'erc-autoaway)

;;; erc-autoaway.el ends here
