;;; erc-speak.el -- Speech-enable the ERC chat client

;; Copyright 2001 Mario Lang <mlang@delysid.org>

;; This file is protected by the GNU GPL, same conditions as with
;; Emacs apply, although this file is not part of GNU Emacs.

;;  Commentary:

;; This file contains code to speech enable ERC using Emacspeaks functionality
;; to access a speech synthesizer.
;;
;; It tries to be intelligent and produce actually understandable
;; audio streams :). Hopefully it does. I use it on #debian at irc.debian.org
;; with about 200 users, and I am amazed how easy it works.
;;
;; Currently, erc-speak is only written to listen to channels.
;; There is no special functionality for interaction in the erc buffers.
;; Although this shouldn't be hard. Look at the Todo list, there are
;; definitely many things this script could do nicely to make a better
;; IRC experience for anyone.
;;
;; More info? Read the code. It isn't that complicated.
;;

;;  ChangeLog:

;; This release justifies the removal of ACTION handling
;; and acronym handling from the Todo-list.

;;  Installation:

;; Currently, change the (load ...) calls at the beginning of code to point
;; to your erc.el and edo-tools.el files.
;;
;; And load this file.

;;  Bugs:

;; erc-speak-rate doesn't seem to work here on outloud. Can anyone enlighten
;; me on the use of dtk-interp-queue-set-rate or equivalents?

;;  Todo:

;; * Rewrite erc-speak-join and write erc-speak-part/quit to also
;;   use the queue and use erc-speak-debug instead of poor format.
;; * Write smiley handling functions (smileys should
;;   use a separate voice too, perhaps the action voice.
;; * Optimize erc-speak-to-whom for more intelligence
;; * Introduce nickname mapping to something alternatively (names
;;   or sounds) - does someone want this. I figure it is fun
;;   to have the speech synth decide how the nicks of people should sound :).
;; * Implement bold and perhaps colors
;; * Convert defvar to defcustom where useful.
;; * Check library header to make something more LCD like.
;; * Add your wish here.
;;

;;  Code:

;(load "~/edo-tools.el")
;(load "~/erc.el")

(require 'erc)
(require 'emacspeak)

(defvar erc-speak-moderator 'harry
  "Voice used for the moderator (who writes where to whom)")

(defvar erc-speak-person 'paul
  "Voice used for messages")

(defvar erc-speak-action-voice 'paul-italic
  "Voice used for ACTIONs")

(defvar erc-speak-debug t
  "Define whether or not erc-speak should log the content
it receives via the hooks.

We are simply using erc-log-aux and ignoree the erc-debug settings.")

(defvar erc-speak-rate 85
  "The speech rate for IRC traffic.
BUG: dtk-interp-queue-set-rate seems to not work. Any hints?")

(defvar erc-speak-quiet nil
  "Set this to prevent erc-speak from actually talking.
nil    - Happily talk away
'maybe - Don't talk, but keep things in the queue for analysis or later playback.
't - Don't talk and discard messages.

You can manipulate the queue when this variable is either nil nor t.")

(defvar erc-speak-acronyms 
  ;; We need a better way of doing this. Currently punctuation
  ;; and/or whitespaces are not handled. This means if someone
  ;; happens to write a word which contains on of the
  ;; REGEXs the word gets mangled :-).. Any idea how to do this
  ;; better?
  '((t "brb" "be right back")
    (t "bbl" "be back later")
    (t "wtf" "what the fuck"))
  "List of common acronyms to replace with their respective meaning.
Format is (CHANNEL REGEX STRING) where CHANNEL is a string containing
a channel name or t for all channels, REGEX is the regular expression
to search and STRING is the string to replace the match with.

If you add any useful acronym definitions here, please
send them to the maintainer of this package so that he can
incorporate your findings in the next release.")

(defvar erc-speak-message-queue nil
  "A FIFO for messages.")

(defun erc-speak-acronym-replace (string target)
  (let ((list erc-speak-acronyms))
    (while list
      (let ((item (car list)))
	(if (or (eq (car item) t) (string= target (car item)))
	    (let ((from (car (cdr item)))
		  (to (car (nthcdr 2 item))))
	      (while (string-match from string)
		(setq string (replace-match to t nil string)))))
	(setq list (cdr list))))
    string))

(defun erc-speak-queue-message (message target nick buffer-name host login spec)
  "Appends the values received by erc-message-hook to erc-speak-message-queue."
  (if erc-speak-debug
      (erc-log-aux (erc-speak-debug 'erc-speak-message
				    message 'message
				    target 'target
				    nick 'nick
				    buffer-name 'buffer
				    host 'host
				    login 'login
				    spec 'spec)))
  (if (erc-speak-channel-p target)
      (setq erc-speak-message-queue
	    (append 
	     erc-speak-message-queue
	     (list (list message target nick buffer-name host login spec))))
    (setq erc-speak-message-queue
	  (append 
	   (list (list message target nick buffer-name host login spec))
	   erc-speak-message-queue))))

(defun erc-speak-to-whom (message nicks)
  "Find out if message addresses one of nicks nicknames.
If true, nickname is returned, otherwise nil."
  ;; Currently a very long regex is generated out of all channel-members
  ;; Perhaps we can come up with a more generic regex
  ;; which doesn't depend on the nick using the exactly
  ;; right nick name as target..
;  (if (> (length nicks) 30)
;      (setq nicks (nthcdr (- (length nicks) 30) nicks)))
  (if (string-match
       (concat "^\\("
;	       (substring
;		(apply 'concat 
	;	       (mapcar (lambda (x) (concat "\\|" x )) nicks)) 2)
	       "[a-zA-Z_0-9-]+\\) ?[:,>]")
       message)
      (substring message (match-beginning 1) (match-end 1))
    nil))

(defun erc-speak-channel-p (target)
  "Returns nil if TARGET is not a channel."
  (eq (elt target 0) ?#))

(defun erc-speak-message ()
  "This function is called via the timer every once in a while.
It pops one or more items of the queue and send them to the speech
server. (intelligence not implemented yet)"
  (if (eq erc-speak-quiet 'nil)
      (if erc-speak-message-queue
	  (let* ((item (car erc-speak-message-queue))
		 (message (car item))
		 (target (car (cdr item)))
		 (nick (car (nthcdr 2 item)))
		 (buffer (car (nthcdr 3 item))))
	    (dtk-interp-set-punctuations "some")
	    (dtk-interp-queue-set-rate erc-speak-rate)
	    (if (string-match (format "^\\(%cACTION \\).*%c$" 1 1) message)
		(dtk-speak-using-voice
		 erc-speak-action-voice
		 (format "%s %s"
			 nick
			 (erc-speak-acronym-replace
			  (substring message 
				     (match-end 1)
				     (- (length message) 1)) target)))
	      (save-excursion
		(set-buffer buffer)
		(let* ((members	(mapcar (lambda (x) (car x)) channel-members))
		       (to-whom (erc-speak-to-whom message members)))
		  (dtk-speak-using-voice
		   erc-speak-moderator
		   (if (erc-speak-channel-p target)
		       (format "%s: %s"
			       (concat "" (substring target 1) ":")
			       (if to-whom
				   (format "%s writes to %s" nick to-whom)
				 nick))
		     (format "Message from %s" nick)))
		  (dtk-speak-using-voice
		   erc-speak-person
		   (format "%s" 
			   (if to-whom
			       (erc-speak-acronym-replace 
				(substring message (match-end 0)) target)
			     (erc-speak-acronym-replace message target)))))))
	    (dtk-force)
	    (setq erc-speak-message-queue (cdr erc-speak-message-queue))))))


(defvar erc-speak-message-timer nil
  "Holds the timer object used to speak messages.
Use it with cancel-timer to remove erc-speak-queue-message")

(defvar erc-speak-idle-time 4
  "Start speaking IRC information after specified time.
Actually this currently means \"Try to pop from the FIFO every SEC seconds\".
Idle time behaviour still has to be written (how to find out how long Emacs
is idle?")

(defun erc-speak-setup ()
  "Initialize the timer functions and add hooks."
  (setq erc-speak-message-timer
	(run-at-time erc-speak-idle-time
		     erc-speak-idle-time 'erc-speak-message))
  (add-hook 'erc-message-hook 'erc-speak-queue-message)
  ;; Join and part behaviour has still to be integrated into the queue
  ;; Wondering if this makes sense anyway.
  (add-hook 'erc-join-hook 'erc-speak-join))
  
(defun erc-speak-join (channel nick buffer ip login spec)
  "Attempts to voicify join events."
  (save-excursion
    (set-buffer buffer)
    (if erc-speak-debug
	(erc-log-aux
	 (concat "(erc-speak-join\n"
		 (format "\t\"%s\" ;; channel\n\t\"%s\" ;; nick\n\t\"%s\" ;; buffer\n\t\"%s\" ;; ip\n\t\"%s\" ;; login\n\t\"%s\" ;; spec)\n"
			 channel nick buffer ip login spec))))
    (if (not erc-speak-quiet)
	(progn
	  (dtk-interp-set-punctuations "some")
	  (dtk-interp-queue-set-rate erc-speak-rate)
	  (dtk-speak-using-voice
	   erc-speak-moderator
	   (format "%s joined channel %s"
		   nick channel))
	  (dtk-force)))))

(defun erc-speak-debug (name &rest list)
  "Return useful debug message."
  (let ((item list)
	(result (concat "'(" (symbol-name name) "\n")))
    (while item
      (setq result (concat result
			   "\t;; "
			   (symbol-name (car (cdr item)))
			   "\n\t\""
			   (car item)
			   "\"\n"))
      (setq item (nthcdr 2 item)))
    (concat result ")")))

(erc-speak-setup)
(provide 'erc-speak)

