/*
 * TestIPAddress.cs - Test class for "System.Net.IPAddress" 
 *
 * Copyright (C) 2002  Southern Storm Software, Pty Ltd.
 * 
 * Authors : Autogenerated using csdoc2test 
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

using CSUnit;
using System;
using System.Net;
using System.Net.Sockets;

public class TestIPAddress : TestCase
 {
	// Constructor.
	public TestIPAddress(String name)	: base(name)
	{
		// Nothing to do here.
	}

	// Set up for the tests.
	protected override void Setup()
	{
		// Nothing to do here.
	}

	// Clean up after the tests.
	protected override void Cleanup()
	{
		// Nothing to do here.
	}

	// Determine if the platform is little-endian.
	private static bool IsLittleEndian
	{
		get
		{
		#if ECMA_COMPAT
			// We cannot use "BitConverter", so use a different approach.
			return (IPAddress.HostToNetworkOrder(0x01020304) != 0x01020304);
		#else
			return BitConverter.IsLittleEndian;
		#endif
		}
	}

	public void TestIPAddressConstructor()
	{
		IPAddress ip=null;
		try
		{
			ip=new IPAddress(-1);
			Fail("IPAddress(-1) should throw an ArgumentOutOfRangeException");
		}
		catch(ArgumentOutOfRangeException)
		{
			//  OK !
		}
		try
		{
			ip=new IPAddress(0x00000001FFFFFFFF);
			Fail("IPAddress(0x00000001FFFFFFFF) should throw an ArgumentOutOfRangeException");
		}
		catch(ArgumentOutOfRangeException)
		{
			//  OK !
		}
		AssertEquals ("IPAddress.Any.Address == 0l", IPAddress.Any.Address, 
						(long) 0);
		AssertEquals ("IPAddress.Broadcast.Address == 0xFFFFFFFF", 
			IPAddress.Broadcast.Address, 
			(long) 0xFFFFFFFF);
		AssertEquals ("IPAddress.Loopback.Address == loopback", 
			IPAddress.Loopback.Address, 
			IPAddress.HostToNetworkOrder(0x7f000001));
	}

	public void TestIPAddressEquals()
	{
	/*TODO*/
	}

	public void TestIPAddressGetHashCode()
	{
	/*TODO*/
	}

	public void TestIPAddressHostToNetworkOrder()
	{
		long [] bigEndian = new long [] { 0, 1, 1, 1};
		long [] littleEndian = new long [] {0, 256, 16777216, 
									72057594037927936 };

		long [] test = IsLittleEndian ? littleEndian : bigEndian;

		short s1=IPAddress.HostToNetworkOrder((short)test[0]);
		AssertEquals("HostToNetworkOrder((short)0)",(short)bigEndian[0],s1);
		
		s1=IPAddress.HostToNetworkOrder((short)test[1]);
		AssertEquals("HostToNetworkOrder((short)1)",(short)bigEndian[1],s1);
		
		int i1 = IPAddress.HostToNetworkOrder((int)test[0]);
		AssertEquals("HostToNetworkOrder((int)0)",(int)bigEndian[0],i1);
		
		i1 = IPAddress.HostToNetworkOrder((int)test[2]);
		AssertEquals("HostToNetworkOrder((int)1)",(int)bigEndian[2],i1);
		
		long l1 = IPAddress.HostToNetworkOrder(test[0]);
		AssertEquals("HostToNetworkOrder((long)0)",bigEndian[0],l1);
		
		l1 = IPAddress.HostToNetworkOrder(test[3]);
		AssertEquals("HostToNetworkOrder((long)1)",bigEndian[3],l1);
	}

	public void TestIPAddressIsLoopback()
	{
		IPAddress ip = IPAddress.Parse ("127.0.0.1");
		AssertEquals ("IPAddress.IsLoopback(127.0.0.1)",true,IPAddress.IsLoopback (ip));
		ip = IPAddress.Parse("192.168.99.1");
		AssertEquals ("IPAddress.IsLoopback(192.168.99.1)",false,IPAddress.IsLoopback (ip));

		ip = IPAddress.Parse("127.127.127.127");
		AssertEquals ("IPAddress.IsLoopback(127.127.127.127)",true,IPAddress.IsLoopback (ip));
	}

	public void TestIPAddressNetworkToHostOrder()
	{
		long [] bigEndian = new long [] { 0, 1, 1, 1};
		long [] littleEndian = new long [] {0, 256, 16777216, 
									72057594037927936 };

		long [] result = IsLittleEndian ? littleEndian : bigEndian;

		short s1=IPAddress.NetworkToHostOrder((short)bigEndian[0]);
		AssertEquals("NetworkToHostOrder((short)0)",(short)result[0],s1);
		
		s1=IPAddress.NetworkToHostOrder((short)bigEndian[1]);
		AssertEquals("NetworkToHostOrder((short)1)",(short)result[1],s1);
		
		int i1 = IPAddress.NetworkToHostOrder((int)bigEndian[0]);
		AssertEquals("NetworkToHostOrder((int)0)",(int)result[0],i1);
		
		i1 = IPAddress.NetworkToHostOrder((int)bigEndian[2]);
		AssertEquals("NetworkToHostOrder((int)1)",(int)result[2],i1);
		
		long l1 = IPAddress.NetworkToHostOrder(bigEndian[0]);
		AssertEquals("NetworkToHostOrder((long)0)",result[0],l1);
		
		l1 = IPAddress.NetworkToHostOrder(bigEndian[3]);
		AssertEquals("NetworkToHostOrder((long)1)",result[3],l1);
	}

	public void TestIPAddressParse()
	{
		IPAddress ip= IPAddress.Parse("192.168.99.1");
		AssertEquals("ip.ToString()==192.168.99.1","192.168.99.1",
			ip.ToString());
			
#if false
		/* 
		* Might support this ... as Int8.Parse("0x7f")==127 
		* Out of spec ..
		*/
		ip = IPAddress.Parse("0x7f.0x00.0x00.0x01");
		AssertEquals("ip.ToString()==127.0.0.1","127.0.0.1",
			ip.ToString());
#endif
	
		ip = IPAddress.Parse("0.0.0.0");
		AssertEquals("0.0.0.0 == IPAddress.Any",IPAddress.Any,
			ip);

		ip = IPAddress.Parse("255.255.255.255");
		AssertEquals("255.255.255.255 == IPAddress.Broadcast",
			IPAddress.Broadcast,ip);
		AssertEquals("255.255.255.255 == IPAddress.None",
			IPAddress.None,ip);

		ip = IPAddress.Parse("127.0.0.1");
		AssertEquals("127.0.0.1 == IPAddress.Loopback",
			IPAddress.Loopback,ip);
		/* Errors */

		try
		{
			ip = IPAddress.Parse(null);
			Fail("IPAddress.Parse(null) should have raised an ArgumentNullException");
		}
		catch(ArgumentNullException)
		{
			/* The crowd goes wild as pnetlib passes test by test */
		}
		
		try
		{
			ip = IPAddress.Parse(" ");
			Fail("IPAddress.Parse(\" \") should have thrown a FormatException");
		}
		catch(FormatException)
		{
			/* Today csunit, tomorrow the world */
		}

		try
		{
			ip = IPAddress.Parse("127.0.0.0.1");
			Fail("IPAddress.Parse(\"127.0.0.0.1\") should have thrown a FormatException");
		}
		catch(FormatException)
		{
			/* My, My .. aren't we good ! */
		}

		try
		{
			ip = IPAddress.Parse("127.0.");
			Fail("IPAddress.Parse(\"127.0.\") should have thrown a FormatException");
		}
		catch(FormatException)
		{
			/* Give me more ! */
		}

		try
		{
			ip = IPAddress.Parse("127:0:0:1");
			Fail("IPAddress.Parse(\"127:0:0:1\") should have thrown a FormatException");
		}
		catch(FormatException)
		{
			/* Oh, the slings and arrows of outrageous fortune !*/
		}
	}

	public void TestIPAddressToString()
	{
		String s;
		s=IPAddress.Loopback.ToString();
		AssertEquals("IPAddress.Loopback.ToString == 127.0.0.1","127.0.0.1",
			s);

		s=IPAddress.Any.ToString();
		AssertEquals("IPAddress.Any.ToString == 0.0.0.0","0.0.0.0",
			s);
		
		s=IPAddress.Broadcast.ToString();
		AssertEquals("IPAddress.Broadcast.ToString == 255.255.255.255",
		"255.255.255.255",
			s);

		s=(IPAddress.Parse("010.000.000.001")).ToString();
		AssertEquals("010.000.000.001 == 10.0.0.1",
		"10.0.0.1",
			s);
	}

	public void TestIPAddressAddress()
	{
		IPAddress ip = new IPAddress(0);

		// ip = IPAddress.Loopback;
		// AssertEquals("Loopback.Address == 0x0100007f",0x0100007f,ip.Address);

		try
		{
			ip.Address=-1;
			Fail("ip.Address=-1 should throw an ArgumentOutOfRangeException");
		}
		catch(ArgumentOutOfRangeException)
		{
			/* Business as usual */
		}
		ip.Address=0xFFFFFFFF;
		AssertEquals("ip == IPAddress.None",IPAddress.None,ip);
	}
	public void TestIPAddressAddressFamily()
	{
		IPAddress ip=IPAddress.Loopback;
		AssertEquals("IPAddress.Loopback.AddressFamily == AddressFamily.InterNetwork", AddressFamily.InterNetwork,ip.AddressFamily);
	}	
}
