// Copyright (C) 2000 Open Source Telecom Corporation.
//  
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
// 
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
// 
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software 
// Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.

#include "driver.h"

char VPBTrunk::status[16 * 4];

VPBTrunk::VPBTrunk(VPBThread *thread, int ts) :
Trunk(ts), TimerPort()
{
	TrunkEvent event;
	service = thread;
	handler = NULL;
	lastring = 0;

	handle = vpb_open(thread->id, ts + 1);
	thread->handles[ts] = this;

	event.id = TRUNK_ENTER_STATE;
	handler = &VPBTrunk::idleHandler;
	postEvent(&event);
}

VPBTrunk::~VPBTrunk()
{
	handler = NULL;
	endTimer();
	
	vpb_close(handle);
	slog(SLOG_INFO) << "vpb" << id << ": device stopped" << endl;
}

void VPBTrunk::getName(char *buffer)
{
	sprintf(buffer, "vpb%d", id);
}

void VPBTrunk::Exit(void)
{
	handler = &VPBTrunk::hangupHandler;
}

void VPBTrunk::setTimer(timeout_t timeout, trunkevent_t event)
{
	notify = event;
	TimerPort::setTimer(timeout);
}

void VPBTrunk::setOffhook(void)
{
	timeout_t timer;
	if(!flags.offhook)
	{
		vpb_sethook_async(handle, VPB_OFFHOOK);
		timer = vpbivr.getHookTimer();
	}
	flags.offhook = true;
	setTimer(timer, TRUNK_OFF_HOOK);
}

bool VPBTrunk::postEvent(TrunkEvent *event)
{
	bool rtn = true;
	trunkhandler_t prior;

	EnterMutex();
	switch(event->id)
	{
	case TRUNK_TIMER_EXPIRED:
		if(!getTimer())
			rtn = false;
		break;
	case TRUNK_DTMF_KEYUP:
		if(!flags.dtmf)
			rtn = false;
		break;
	}	
	if(!rtn)
	{
		LeaveMutex();
		return false;
	}

	if(!handler)
	{
		slog(SLOG_WARNING) << "vpb" << id;
		slog() << ": no handler active; event=" << event->id << endl;
		LeaveMutex();
		return false;
	}

retry:
	debug->DebugEvent(this, event);
	prior = handler;
	rtn = (this->*handler)(event);
	if(rtn)
	{
		if(handler != prior)
		{
			event->id = TRUNK_ENTER_STATE;
			goto retry;
		}
		LeaveMutex();
		return true;
	}

	// default handler

	rtn = true;
	switch(event->id)
	{
	case TRUNK_RINGING_ON:
		++rings;
		break;
	case TRUNK_ENTER_STATE:
		if(flags.offhook)
			setDTMFDetect();
		else
			setDTMFDetect(false);
		endTimer();
		break;
	case TRUNK_LINE_WINK:
		if(!flags.offhook)
			break;
	case TRUNK_CPA_DIALTONE:
	case TRUNK_STOP_DISCONNECT:
		TrunkSignal(TRUNK_SIGNAL_HANGUP);
		event->id = TRUNK_STOP_STATE;
		goto retry;
		break;
	case TRUNK_TIMER_EXPIRED:
		if(!TrunkSignal(TRUNK_SIGNAL_TIMEOUT))
			TrunkSignal(TRUNK_SIGNAL_STEP);
		event->id = TRUNK_STOP_STATE;
		goto retry;
		break;
	case TRUNK_DTMF_KEYUP:
		if(digits < 32)
			dtmf.bin.data[digits++] = digit[event->parm.dtmf.digit];
		dtmf.bin.data[digits] = 0;
		if(TrunkSignal((trunksignal_t)(event->parm.dtmf.digit + TRUNK_SIGNAL_0)))
		{
			event->id = TRUNK_STOP_STATE;
			goto retry;
		}
		break;
	case TRUNK_EXIT_SHELL:
		tgi.pid = 0;
		break;
	case TRUNK_STOP_STATE:
		endTimer();
		handler = &VPBTrunk::stepHandler;
		break;
	case TRUNK_EXIT_STATE:
		break;
	case TRUNK_MAKE_BUSY:
		handler = &VPBTrunk::busyHandler;
		break;
	case TRUNK_MAKE_IDLE:
		handler = &VPBTrunk::idleHandler;
		break;
	default:
		rtn = false;
	}
	if(handler != prior)
	{
		event->id = TRUNK_ENTER_STATE;
		goto retry;
	}
	LeaveMutex();
	return rtn;
}

unsigned long VPBTrunk::getIdleTime(void)
{
	time_t now;

	time(&now);
	if(handler == &VPBTrunk::idleHandler)
		return now - idle;

	return 0;
}

