// Copyright (C) 2000 Open Source Telecom Corporation.
//  
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
// 
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
// 
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software 
// Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.

#include "driver.h"

VPBRecord::VPBRecord(VPBTrunk *trunk, int h) :
Service((Trunk *)trunk, keythreads.priAudio()), AudioFile()
{
	handle = h;
	reset = false;
	stopped = false;
}

VPBRecord::~VPBRecord()
{
	char buffer[12];
	struct stat ino;
	int trim;

	Terminate();
	sprintf(buffer, "%ld", getPosition());
	trunk->setSymbol(SYM_RECORDED, buffer);
	trim = tobytes(getEncoding(), atoi(trunk->getSymbol(SYM_TRIM)));
	stat(data->record.name, &ino);
	truncate(data->record.name, ino.st_size - trim);
	chown(data->record.name, keyserver.getUid(), keyserver.getGid());

	Close();
	if(reset)
	{
		vpb_record_terminate(handle);
		dspReset();
	}
}

void VPBRecord::Initial(void)
{
	int codec;
	char buffer[32];
	audioinfo_t info;
	char *ext = strrchr(data->record.name, '.');

	trunk->getName(buffer);
	info.format = AUDIO_FORMAT_RAW;
	info.encoding = MULAW_AUDIO_ENCODING;
	info.order = 0;
	info.annotation = trunk->getSymbol(SYM_ANNOTATION);
	info.rate = 8000;
	
	if(!ext)
		ext = trunk->getSymbol(SYM_EXTENSION);

	if(!stricmp(ext, ".al"))
		info.encoding = ALAW_AUDIO_ENCODING;
	else if(!stricmp(ext, ".au"))
	{
		info.format = AUDIO_FORMAT_SUN;
		info.order = __BIG_ENDIAN;
	}
	else if(!stricmp(ext, ".wav"))
	{
		info.format = AUDIO_FORMAT_RIFF;
		info.order = __LITTLE_ENDIAN;
	}

	if(data->record.append)
		Open(data->record.name);
	else
		Create(data->record.name, &info);	

	if(!isOpen())
	{
		slog(SLOG_ERROR) << data->record.name << ": cannot open" << endl;
		Failure();
	}
	if(data->record.append)
		setPosition();

	switch(getEncoding())
	{
	case G721_ADPCM_ENCODING:
	case OKI_ADPCM_ENCODING:
		codec = VPB_OKIADPCM;
		bufsize = 80;
		samples = 160;
		break;
	case G723_3BIT_ENCODING:
		codec = VPB_OKIADPCM24;
		bufsize = 60;
		samples = 160;
		break;
	case MULAW_AUDIO_ENCODING:
		codec = VPB_MULAW;
		bufsize = 160;
		samples = 160;
		break;
	case ALAW_AUDIO_ENCODING:
		codec = VPB_ALAW;
		bufsize = 160;
		samples = 160;
		break;
	case PCM16_AUDIO_ENCODING:
		codec = VPB_LINEAR;
		bufsize = 320;
		samples = 160;
		break;
	default:
		slog(SLOG_ERROR) << buffer << ": unsupported codec required" << endl;
		Failure();
	}

	reset = true;
	vpb_record_buf_start(handle, codec);
}

void VPBRecord::Run(void)
{
	char name[33];
	char buffer[bufsize];
	audioerror_t status = AUDIO_SUCCESS;
	int len = 0;

	trunk->getName(name);

	while(status == AUDIO_SUCCESS && !stopped)
	{	
		if(vpb_record_buf_sync(handle, buffer, bufsize) != VPB_OK)
			status = AUDIO_READ_INCOMPLETE;
		else
			status = putSamples(buffer, samples);
	}
	if(status == AUDIO_WRITE_FAILURE)
		slog(SLOG_ERROR) << name << ": failed record" << endl;
	vpb_record_buf_finish(handle);
	reset = false;
	Success();
}

bool VPBTrunk::recordHandler(TrunkEvent *event)
{
	unsigned short mask;

	switch(event->id)
	{
	case TRUNK_DTMF_KEYUP:
		mask = (1 << event->parm.dtmf.digit);
		if(!(mask & data.record.term))
			return false;
	case TRUNK_SERVICE_SUCCESS:
		TrunkSignal(TRUNK_SIGNAL_STEP);
		handler = &VPBTrunk::stepHandler;
		return true;
	case TRUNK_SERVICE_FAILURE:
		setSymbol(SYM_ERROR, "record-failed");
		TrunkSignal(TRUNK_SIGNAL_ERROR);
		handler = &VPBTrunk::stepHandler;
		return true;
	case TRUNK_ENTER_STATE:
		debug->DebugState(this, "record");
		flags.dsp = DSP_MODE_VOICE;
		status[id] = 'r';
		if(!flags.offhook)
		{
			flags.offhook = true;
			vpb_sethook_async(handle, VPB_OFFHOOK);
			setTimer(vpbivr.getHookTimer());
			return true;
		}
	case TRUNK_TIMER_EXPIRED:
		if(data.record.term)
			setDTMFDetect(true);
		else
			setDTMFDetect();
		thread = new VPBRecord(this, handle);
		thread->Start();
		setTimer(data.record.timeout);
		return true;
	}
	return false;
}

