// Copyright (C) 2000 Open Source Telecom Corporation.
//  
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
// 
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
// 
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software 
// Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
//
// As a special exception to the GNU General Public License, permission is
// granted for additional uses of the text contained in its release
// of Bayonne as noted here.
//
// This exception is that permission is hereby granted to link Bayonne 
// with  the Pika MonteCarlo libraries to produce a executable image
// without requiring MonteCarlo itself to be supplied in source form so
// long as each source file so linked contains this exclusion.
//
// This exception does not however invalidate any other reasons why
// the resulting executable file might be covered by the GNU General
// public license or invalidate the licensing requirements of any
// other component or library.
//
// This exception applies only to the code released by OST under the
// name Bayonne.  If you copy code from other releases into a copy of
// Bayonne, as the General Public License permits, the exception does not
// apply to the code that you add in this way.  To avoid misleading
// anyone as to the status of such modified files, you must delete
// this exception notice from them.
//
// If you write modifications of your own to Bayonne, it is your choice
// whether to permit this exception to apply to your modifications.
// If you do not wish that, delete this exception notice, at which
// point the terms of your modification would be covered under the GPL
// as explicitly stated in "COPYING".

#include <bayonne.h>

class UDPStat : public StatMon, public MappedFile, public UDPSocket
{
private:
	statnode_t *map;
	time_t last;
	char *getPath(void);
	statnode_t *getNode(const char *name);
	InetAddress &getInterface(void);
	BroadcastAddress &getBroadcast(void);
	tpport_t getPort(void);
public:
	UDPStat();
	void Refresh(int secs);
} udpstat;

UDPStat::UDPStat() :
StatMon(), UDPSocket(getInterface(), getPort()),
MappedFile(getPath(), FILE_ACCESS_READWRITE)
{
	statnode_t node;
	int i, nodes = keyserver.getNodeCount();
	char *addr = NULL;

	time(&last);
	UDPSocket::setError(false);
	RandomFile::setError(false);
	setBroadcast(true);
	setPeer((InetHostAddress &)getBroadcast(), getPort());

	if(!isBroadcast())
	{
		map = NULL;
		slog(SLOG_WARNING) << "udpstat: broadcast failed" << endl;
		return;
	}

	memset(&node, 0, sizeof(node));
	for(i = 0; i < nodes; ++i)
		::write(fd, &node, sizeof(node)); 

	map = (statnode_t *)MappedFile::Fetch(0, nodes * sizeof(node));
	if(!map)
		slog(SLOG_ERROR) << "udpstat: mapping failed" << endl;
}

statnode_t *UDPStat::getNode(const char *name)
{
	statnode_t *list = map;

	while(list->name[0])
	{
		if(!strncmp(list->name, name, 16))
			return list;
		++list;
	}
	return NULL;
}

tpport_t UDPStat::getPort(void)
{
	const char *cp = getLast("port");
	if(!cp)
		cp = "5811";

	return (tpport_t)atoi(cp);
}

InetAddress &UDPStat::getInterface(void)
{
	static InetAddress addr;
	const char *cp = getLast("interface");

	if(!cp)
		cp = "*";

	if(!strcmp(cp, "*"))
		cp = "0.0.0.0";

	addr = cp;
	return addr;
}

BroadcastAddress &UDPStat::getBroadcast(void)
{
	static BroadcastAddress addr;
	const char *cp = getLast("broadcast");

	if(!cp)
		cp = "127.0.0.1";

	if(!strcmp(cp, "*"))
		cp = "255.255.255.255";

	addr = cp;
	return addr;
}

char *UDPStat::getPath(void)
{
	static char path[256];

	if(canModify(keypaths.getRunfiles()))
		sprintf(path, "%s/bayonne.nodes", keypaths.getRunfiles());
	else
		sprintf(path, "%s/.bayonne.nodes", getenv("HOME"));
	return path;
}

void UDPStat::Refresh(int secs)
{
	time_t now;
	time_t start;
	long diff;
	int refresh = keythreads.getRefresh();
	statnode_t node;
	int ports = driver->getTrunkCount();
	statnode_t *list;

	time(&start);

	if(!map)
	{
		ccxx_sleep(secs * 1000);
		return;
	}
	
	for(;;)
	{
		time(&now);
		if(now >= last + refresh)
		{
			strncpy(node.name, keyserver.getNode(), sizeof(node.name));
			strncpy((char *)&node.update, "STAT", 4);
			node.ports = ports;
			driver->getStatus(node.stat);
			Send(&node, sizeof(node) - (sizeof(node.stat) - ports));
			time(&last);
		}

		diff = last + refresh - now;
		if(diff + now > start + secs)
			diff = start + secs - now;

		if(diff < 1)
			return;

		if(isPending(SOCKET_PENDING_INPUT, diff * 1000))
		{
			memset(&node, 0, sizeof(node));
			Recv(&node, sizeof(node));
			time(&node.update);
			list = map;
			if(memcmp(&node.update, "STAT", 4))
				continue;
			while(list->name[0])
			{
				if(!memcmp(list->name, node.name, sizeof(node.name)))
					break;
				++list;
			}
			memcpy(list, &node, sizeof(node));
		}
	}
}

