/*
 * Copyright (c) 2005-2008 Substance Kirill Grouchnikov. All Rights Reserved.
 *
 * Redistribution and use in source and binary forms, with or without 
 * modification, are permitted provided that the following conditions are met:
 * 
 *  o Redistributions of source code must retain the above copyright notice, 
 *    this list of conditions and the following disclaimer. 
 *     
 *  o Redistributions in binary form must reproduce the above copyright notice, 
 *    this list of conditions and the following disclaimer in the documentation 
 *    and/or other materials provided with the distribution. 
 *     
 *  o Neither the name of Substance Kirill Grouchnikov nor the names of 
 *    its contributors may be used to endorse or promote products derived 
 *    from this software without specific prior written permission. 
 *     
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" 
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, 
 * THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR 
 * PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR 
 * CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, 
 * EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, 
 * PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; 
 * OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, 
 * WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE 
 * OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, 
 * EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE. 
 */
package test;

import java.awt.*;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.image.BufferedImage;
import java.util.HashMap;
import java.util.Map;

import javax.swing.*;
import javax.swing.event.ChangeEvent;
import javax.swing.event.ChangeListener;

import org.jvnet.lafwidget.LafWidget;
import org.jvnet.lafwidget.LafWidgetUtilities2;
import org.jvnet.lafwidget.layout.TransitionLayout;
import org.jvnet.lafwidget.preview.DefaultPreviewPainter;
import org.jvnet.substance.SubstanceLookAndFeel;
import org.jvnet.substance.painter.AlphaControlBackgroundComposite;
import org.jvnet.substance.utils.SubstanceCoreUtilities;
import org.jvnet.substance.utils.SubstanceConstants.ScrollPaneButtonPolicyKind;

public class ScrollDemo extends JFrame {
	public class ScrollPanel extends JPanel {
		private JScrollPane sp;

		private JPanel panel;

		private Map<String, BufferedImage> shots;

		public ScrollPanel() {
			this.shots = new HashMap<String, BufferedImage>();

			this.panel = new JPanel() {
				@Override
				protected void paintComponent(Graphics g) {
					Graphics2D graphics = (Graphics2D) g.create();
					graphics.setComposite(TransitionLayout
							.getAlphaComposite(panel));

					BufferedImage shot = getShot(this.getWidth(), this
							.getHeight());
					graphics.drawImage(shot, 0, 0, null);
					graphics.dispose();
				}
			};
			// this.panel.putClientProperty(LafWidget.COMPONENT_PREVIEW_PAINTER,
			// new DefaultPreviewPainter());
			this.panel.setPreferredSize(new Dimension(1200, 800));
			this.panel.setSize(this.getPreferredSize());
			this.panel.setMinimumSize(this.getPreferredSize());
			this.sp = new JScrollPane(this.panel,
					ScrollPaneConstants.VERTICAL_SCROLLBAR_AS_NEEDED,
					ScrollPaneConstants.HORIZONTAL_SCROLLBAR_AS_NEEDED);
			this.sp.putClientProperty(SubstanceLookAndFeel.OVERLAY_PROPERTY,
					Boolean.TRUE);
			this.sp.putClientProperty(
					SubstanceLookAndFeel.PAINT_ACTIVE_PROPERTY, Boolean.TRUE);
			this.sp.putClientProperty(LafWidget.COMPONENT_PREVIEW_PAINTER,
					new DefaultPreviewPainter());
			// this.sp.putClientProperty(
			// SubstanceLookAndFeel.PAINT_ACTIVE_PROPERTY, Boolean.FALSE);
			this.setLayout(new BorderLayout());
			this.add(this.sp, BorderLayout.CENTER);

			JPanel controlsSliders = new JPanel(new GridLayout(2, 1));

			final JSlider sliderInactive = new JSlider(0, 100, 50);
			sliderInactive.setMajorTickSpacing(20);
			sliderInactive.setMinorTickSpacing(5);
			sliderInactive.setPaintLabels(true);
			sliderInactive.setPaintTicks(true);
			sliderInactive.setSnapToTicks(true);

			final JSlider sliderActive = new JSlider(0, 100, 80);
			sliderActive.setMajorTickSpacing(20);
			sliderActive.setMinorTickSpacing(5);
			sliderActive.setPaintLabels(true);
			sliderActive.setPaintTicks(true);
			sliderActive.setSnapToTicks(true);

			sliderInactive.addChangeListener(new ChangeListener() {
				public void stateChanged(ChangeEvent e) {
					int valInactive = sliderInactive.getValue();
					int valActive = sliderActive.getValue();
					sp.putClientProperty(
							SubstanceLookAndFeel.BACKGROUND_COMPOSITE,
							new AlphaControlBackgroundComposite(
									valInactive / 100.0f, valActive / 100.0f));
					sp.repaint();
				}
			});
			sliderActive.addChangeListener(new ChangeListener() {
				public void stateChanged(ChangeEvent e) {
					int valInactive = sliderInactive.getValue();
					int valActive = sliderActive.getValue();
					sp.putClientProperty(
							SubstanceLookAndFeel.BACKGROUND_COMPOSITE,
							new AlphaControlBackgroundComposite(
									valInactive / 100.0f, valActive / 100.0f));
					sp.repaint();
				}
			});
			int valInactive = sliderInactive.getValue();
			int valActive = sliderActive.getValue();
			sp.putClientProperty(SubstanceLookAndFeel.BACKGROUND_COMPOSITE,
					new AlphaControlBackgroundComposite(valInactive / 100.0f,
							valActive / 100.0f));

			controlsSliders.add(sliderActive);
			controlsSliders.add(sliderInactive);
			this.add(controlsSliders, BorderLayout.SOUTH);

			JPanel controls = new JPanel();
			controls.setLayout(new FlowLayout(FlowLayout.CENTER));
			final JCheckBox isEnabled = new JCheckBox("enabled");
			isEnabled.setSelected(true);
			isEnabled.addActionListener(new ActionListener() {
				public void actionPerformed(ActionEvent e) {
					boolean toEnable = isEnabled.isSelected();
					sp.setEnabled(toEnable);
					updateEnabledState(sp, toEnable);
					Check.out("Scroll pane is " + toEnable);
				}
			});
			// controls.add(isEnabled);

			final JCheckBox hasCustomPreview = new JCheckBox("preview");
			hasCustomPreview.setSelected(LafWidgetUtilities2
					.getComponentPreviewPainter(sp) != null);
			hasCustomPreview.addActionListener(new ActionListener() {
				public void actionPerformed(ActionEvent e) {
					sp
							.putClientProperty(
									LafWidget.COMPONENT_PREVIEW_PAINTER,
									hasCustomPreview.isSelected() ? new DefaultPreviewPainter()
											: null);
					sp.repaint();
				}
			});
			controls.add(hasCustomPreview);

			final JComboBox buttonPolicyCombo = new JComboBox(new Object[] {
					ScrollPaneButtonPolicyKind.NONE,
					ScrollPaneButtonPolicyKind.OPPOSITE,
					ScrollPaneButtonPolicyKind.ADJACENT,
					ScrollPaneButtonPolicyKind.MULTIPLE,
					ScrollPaneButtonPolicyKind.MULTIPLE_BOTH });
			buttonPolicyCombo
					.setSelectedItem(ScrollPaneButtonPolicyKind.OPPOSITE);
			buttonPolicyCombo.addActionListener(new ActionListener() {
				public void actionPerformed(ActionEvent e) {
					ScrollPaneButtonPolicyKind buttonPolicy = (ScrollPaneButtonPolicyKind) buttonPolicyCombo
							.getSelectedItem();
					sp.putClientProperty(
							SubstanceLookAndFeel.SCROLL_PANE_BUTTONS_POLICY,
							buttonPolicy);
					sp.repaint();
				}
			});
			// controls.add(new JLabel("buttons"));
			controls.add(buttonPolicyCombo);

			final JCheckBox isNever = new JCheckBox("never");
			isNever.addActionListener(new ActionListener() {
				public void actionPerformed(ActionEvent e) {
					sp.putClientProperty(
							SubstanceLookAndFeel.BUTTON_PAINT_NEVER_PROPERTY,
							isNever.isSelected() ? Boolean.TRUE : null);
					sp.repaint();
				}
			});
			controls.add(isNever);

			final JCheckBox isFlat = new JCheckBox("flat");
			isFlat.addActionListener(new ActionListener() {
				public void actionPerformed(ActionEvent e) {
					sp.putClientProperty(SubstanceLookAndFeel.FLAT_PROPERTY,
							isFlat.isSelected() ? Boolean.TRUE : null);
					sp.repaint();
				}
			});
			controls.add(isFlat);

			final JCheckBox isActive = new JCheckBox("active");
			isActive
					.setSelected(Boolean.TRUE
							.equals(sp
									.getClientProperty(SubstanceLookAndFeel.PAINT_ACTIVE_PROPERTY)));
			isActive.addActionListener(new ActionListener() {
				public void actionPerformed(ActionEvent e) {
					sp.putClientProperty(
							SubstanceLookAndFeel.PAINT_ACTIVE_PROPERTY,
							isActive.isSelected() ? Boolean.TRUE : null);
					sp.repaint();
				}
			});
			controls.add(isActive);

			final JCheckBox isOverlay = new JCheckBox("overlay");
			isOverlay.setSelected(Boolean.TRUE.equals(sp
					.getClientProperty(SubstanceLookAndFeel.OVERLAY_PROPERTY)));
			isOverlay.addActionListener(new ActionListener() {
				public void actionPerformed(ActionEvent e) {
					sp.putClientProperty(SubstanceLookAndFeel.OVERLAY_PROPERTY,
							isOverlay.isSelected() ? Boolean.TRUE : null);
					sp.repaint();
				}
			});
			controls.add(isOverlay);

			this.add(controls, BorderLayout.NORTH);
		}

		private synchronized BufferedImage getShot(int width, int height) {
			String key = width + ":" + height;
			BufferedImage result = this.shots.get(key);
			if (result == null) {
				result = SubstanceCoreUtilities.getBlankImage(width, height);

				Graphics2D graphics = result.createGraphics();
				int cols = 1 + width / 10;
				int rows = 1 + height / 10;
				if (SubstanceCoreUtilities.isThemeDark(SubstanceLookAndFeel
						.getTheme()))
					graphics.setColor(Color.black);
				else
					graphics.setColor(Color.white);
				graphics.fillRect(0, 0, width, height);
				for (int i = 0; i < cols; i++) {
					for (int j = 0; j < rows; j++) {
						if (((i + j) % 2) == 0) {
							float val = (i + j) / 100.f;
							val -= Math.floor(val);
							boolean isDark = SubstanceCoreUtilities
									.isThemeDark(SubstanceLookAndFeel
											.getTheme());
							float brightness = isDark ? 0.1f : 0.9f;
							float saturation = 0.2f;
							graphics.setColor(new Color(Color.HSBtoRGB(val,
									saturation, brightness)));
							graphics.fillRect(i * 10, j * 10, 10, 10);
						}
					}
				}

				graphics.setColor(Color.gray);
				graphics.setFont(new Font("Arial", Font.PLAIN, 11));
				rows = 1 + height / 25;
				for (int i = 0; i < rows; i++) {
					for (int j = 0; j < width / 25; j++) {
						graphics.drawString("" + (i + j), j * 25, 25 * i);
					}
				}

				this.shots.put(key, result);
			}
			return result;
		}
	}

	public ScrollDemo() {
		super("Scroll overlay demo");
		this.setLayout(new BorderLayout());
		this.add(new ScrollPanel(), BorderLayout.CENTER);
		this.setSize(450, 400);
		this.setLocationRelativeTo(null);
		this.setDefaultCloseOperation(JFrame.EXIT_ON_CLOSE);
	}

	void updateEnabledState(Container c, boolean enabled) {
		for (int counter = c.getComponentCount() - 1; counter >= 0; counter--) {
			Component child = c.getComponent(counter);

			child.setEnabled(enabled);
			if (child instanceof Container) {
				updateEnabledState((Container) child, enabled);
			}
		}
	}

	public static void main(String[] args) throws Exception {
		// UIManager.setLookAndFeel(new SubstanceModerateLookAndFeel());
		UIManager.setLookAndFeel(new SubstanceLookAndFeel());
		SwingUtilities.invokeLater(new Runnable() {
			public void run() {
				new ScrollDemo().setVisible(true);
			}
		});
	}
}