/*
 * Copyright (c) 2005-2008 Substance Kirill Grouchnikov. All Rights Reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 *  o Redistributions of source code must retain the above copyright notice,
 *    this list of conditions and the following disclaimer.
 *
 *  o Redistributions in binary form must reproduce the above copyright notice,
 *    this list of conditions and the following disclaimer in the documentation
 *    and/or other materials provided with the distribution.
 *
 *  o Neither the name of Substance Kirill Grouchnikov nor the names of
 *    its contributors may be used to endorse or promote products derived
 *    from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO,
 * THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR
 * CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
 * EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
 * PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS;
 * OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
 * WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE
 * OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE,
 * EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */
package org.jvnet.substance.painter.text;

import java.awt.*;
import java.awt.geom.AffineTransform;

import javax.swing.JComponent;
import javax.swing.plaf.basic.BasicGraphicsUtils;

import org.jvnet.lafwidget.utils.RenderingUtils;
import org.jvnet.substance.utils.SubstanceCoreUtilities;

/**
 * The default core implementation of {@link SubstanceTextPainter}.
 * 
 * @author Kirill Grouchnikov
 */
public class DefaultTextPainter extends AbstractTextPainter {
	/*
	 * (non-Javadoc)
	 * 
	 * @see org.jvnet.substance.painter.text.SubstanceTextPainter#needsBackgroundImage()
	 */
	public boolean needsBackgroundImage() {
		return false;
	}

	public void setBackgroundFill(JComponent comp, Color backgroundFillColor,
			boolean toOverlayWatermark, int watermarkOffsetX,
			int watermarkOffsetY) {
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.jvnet.substance.painter.text.SubstanceTextPainter#renderSurface(java.awt.Graphics)
	 */
	public void renderSurface(Graphics g) {
		Graphics2D g2d = (Graphics2D) g.create();
		SubstanceCoreUtilities.workaroundBug6576507(g2d);
		RenderingUtils.installDesktopHints(g2d);

		for (TextLineInfo lineInfo : textLines) {
			g2d.setFont(lineInfo.font);
			g2d.setColor(lineInfo.color);
			if (lineInfo.clip != null)
				g2d.setClip(lineInfo.clip);
			AffineTransform currTransform = g2d.getTransform();
			if (lineInfo.transform != null)
				g2d.transform(lineInfo.transform);
			String text = lineInfo.text;
			if ((text != null) && (text.length() > 0)) {
				BasicGraphicsUtils.drawStringUnderlineCharAt(g2d,
						lineInfo.text, lineInfo.mnemonicIndex,
						lineInfo.textRect.x, lineInfo.textRect.y
								+ g2d.getFontMetrics().getAscent());
			}
			g2d.setTransform(currTransform);
		}
		g2d.dispose();
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.jvnet.substance.painter.text.SubstanceTextPainter#attachVerticalText(javax.swing.JComponent,
	 *      java.awt.Rectangle, java.lang.String, int, java.awt.Font,
	 *      java.awt.Color, java.awt.Rectangle, boolean)
	 */
	public void attachVerticalText(JComponent comp, Rectangle textRect,
			String text, int mnemonicIndex, Font font, Color color,
			Rectangle clip, boolean isFromBottomToTop) {
		AffineTransform at = null;

		if (!isFromBottomToTop) {
			at = AffineTransform.getTranslateInstance(textRect.x
					+ textRect.width, textRect.y);
			at.rotate(Math.PI / 2);
		} else {
			at = AffineTransform.getTranslateInstance(textRect.x, textRect.y
					+ textRect.height);
			at.rotate(-Math.PI / 2);
		}
		Rectangle newRect = new Rectangle(0, 0, textRect.width, textRect.height);
		this.textLines.add(new TextLineInfo(newRect, text, mnemonicIndex, font,
				color, clip, at));
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.jvnet.substance.painter.text.SubstanceTextPainter#isNative()
	 */
	public boolean isNative() {
		return false;
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.jvnet.substance.painter.text.SubstanceTextPainter#dispose()
	 */
	public void dispose() {
	}

	public java.awt.Dimension getTextBounds(Component comp, java.awt.Font font,
			String text) {
		FontMetrics fm = comp.getFontMetrics(font);
		return new Dimension(fm.stringWidth(text), fm.getHeight());
	}
}
