/*********************************************************************

    m6847.c

    Implementation of Motorola 6847 video hardware chip

    Sources:
    M6847 data sheet
    M6847T1 info from Rainbow magazine (10/1986-12/1986)


    AG  AS  INTEXT  INV  GM2  GM1  GM0
    --  --  ------  ---  ---  ---  ---
     0   0       0    0    X    X    X  Internal Alphanumerics
     0   0       0    1    X    X    X  Internal Alphanumerics Inverted
     0   0       1    0    X    X    X  External Alphanumerics
     0   0       1    1    X    X    X  External Alphanumerics Inverted
     0   1       0    X    X    X    X  Semigraphics 4
     0   1       1    X    X    X    X  Semigraphics 6
     1   X       X    X    0    0    0  Graphics CG1 (64x64x4)    (16 bpr)
     1   X       X    X    0    0    1  Graphics RG1 (128x64x2)   (16 bpr)
     1   X       X    X    0    1    0  Graphics CG2 (128x64x4)   (32 bpr)
     1   X       X    X    0    1    1  Graphics RG2 (128x96x2)   (16 bpr)
     1   X       X    X    1    0    0  Graphics CG3 (128x96x4)   (32 bpr)
     1   X       X    X    1    0    1  Graphics RG3 (128x192x2)  (16 bpr)
     1   X       X    X    1    1    0  Graphics CG6 (128x192x4)  (32 bpr)
     1   X       X    X    1    1    1  Graphics RG6 (256x192x2)  (32 bpr)

    Note: The M6847 relies on an external source (typically a 6883 SAM chip)
    to feed it bytes; so the BPR (bytes per row) figures are effectively
    suggestions.  Mismatching modes is responsible for the semigraphic modes
    on the CoCo.

    Timing:
    (source Motorola M6847 Manual)

    Horizontal Sync:  Total Period: 227.5 clock cycles
        @ CLK(0) + DHS_F            - falling edge (high to low)
        @ CLK(16.5) + DHS_R         - rising edge (low to high)
        @ CLK(42)                   - left border start
        @ CLK(71.5)                 - body start
        @ CLK(199.5)                - right border start
        @ CLK(227.5) + DHS_F        - falling edge (high to low)
        ...

    Field Sync: Total Period 262*227.5 clock cycles
        @ CLK(0) + DFS_F            - falling edge (high to low)
        @ CLK(32*227.5) + DFS_R     - rising edge (low to high)
        @ CLK(262*227.5) + DFS_F    - falling edge (high to low) (262.5 for the M6847Y)

    DHS_F:  550ns
    DHS_R:  740ns
    DFS_F:  520ns
    DFS_R:  500ns

    The M6847T1 is a later variant of the M6847 chip that implements lower
    case support and some other nifty features.  This chip is in the CoCo 2B.
    I have not been able to find a pinout diagram for this chip so I am
    assuming that the extra text modes on the CoCo 2B are activated by the
    GM2-0 pins.  This needs to be confirmed.

    TODO: Implement PAL support

**********************************************************************/

#include "emu.h"
#include "m6847.h"

#include "debug/debugcpu.h"
#include "debug/debugcon.h"

enum
{
	M6847_AG		= 0x80,
	M6847_AS		= 0x40,
	M6847_INTEXT	= 0x20,
	M6847_INV		= 0x10,
	M6847_CSS		= 0x08,
	M6847_GM2		= 0x04,
	M6847_GM1		= 0x02,
	M6847_GM0		= 0x01
};



#define LOG_FS			0
#define LOG_HS			0
#define LOG_STATS		0
#define LOG_PREPARE		0

typedef struct _m6847_variant m6847_variant;
struct _m6847_variant
{
	const UINT8 *fontdata;
	unsigned int has_lowercase : 1;
	unsigned int text_offset : 1;

	int frames_per_second;
	double scanlines_per_frame;
	double clocks_per_scanline;

	double vblank_scanlines;
	int top_border_scanlines;
	int display_scanlines;
	int bottom_border_scanlines;
	int vretrace_scanlines;

	double field_sync_scanlines;
};

typedef struct _m6847_pixel m6847_pixel;
struct _m6847_pixel
{
	UINT8 data;
	UINT8 attr;
};

typedef struct _mc6847_state mc6847_state;
struct _mc6847_state
{
	/* callbacks */
	devcb_resolved_read8 in_dd_func;

	devcb_resolved_read_line in_gm2_func;
	devcb_resolved_read_line in_gm1_func;
	devcb_resolved_read_line in_gm0_func;
	devcb_resolved_read_line in_intext_func;
	devcb_resolved_read_line in_inv_func;
	devcb_resolved_read_line in_as_func;
	devcb_resolved_read_line in_ag_func;
	devcb_resolved_read_line in_css_func;

	devcb_resolved_write_line out_fs_func;
	devcb_resolved_write_line out_hs_func;
	devcb_resolved_write_line out_rs_func;

	/* state of mode control lines */
	unsigned int gm2 : 1;
	unsigned int gm1 : 1;
	unsigned int gm0 : 1;
	unsigned int intext : 1;
	unsigned int inv : 1;
	unsigned int as : 1;
	unsigned int ag : 1;
	unsigned int css : 1;

	/* sync control line status */
	unsigned int fs : 1;
	unsigned int hs : 1;

	/* to be cleaned up... */
	int (*new_frame_callback)(running_machine &machine);	/* returns whether the M6847 is in charge of this frame */
	void (*custom_prepare_scanline)(int scanline);
	UINT8 (*get_char_rom)(running_machine &machine, UINT8 ch,int line);

	/* timers */
	emu_timer *fs_rise_timer;
	emu_timer *fs_fall_timer;
	emu_timer *hs_rise_timer;
	emu_timer *hs_fall_timer;

	/* things that vary according to chip version */
	attoseconds_t clock_period;
	attoseconds_t scanline_period;
	attoseconds_t field_sync_period;
	attoseconds_t horizontal_sync_period;
	attoseconds_t vblank_period;
	int top_border_scanlines;
	int display_scanlines;
	int bottom_border_scanlines;
	const UINT32 *palette;

	/* flags */
	unsigned int has_lowercase : 1;			/* M6847T1 has lower case */
	unsigned int has_custom_palette : 1;	/* needed for CoCo 3 */
	unsigned int text_offset : 2;			/* needed for CoCo 3 */

	/* video state; every scanline the video state for the scanline is copied
     * here and only rendered in SCREEN_UPDATE */
	int dirty;
	int using_custom;
	UINT32 border[384];
	UINT8 attrs[384];
	m6847_pixel screendata[192][32];

	UINT32 saved_artifacting, saved_c0, saved_c1;
	UINT32 expanded_colors[128];

	/* saved palette; used with CoCo 3 */
	UINT32 saved_palette[16];

	/* 2^7 modes, 256/16 character groups, background/foreground */
	UINT8 colordata[128][256/16][2];

	/* 2^7 modes, 256 characters, 12 scanlines */
	UINT8 fontdata[128][256][12];
};



static void apply_artifacts(device_t *device, UINT32 *line);


static const UINT8 pal_round_fontdata8x12[] =
{
	0x00, 0x00, 0x38, 0x44, 0x04, 0x34, 0x4C, 0x4C, 0x38, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x10, 0x28, 0x44, 0x44, 0x7C, 0x44, 0x44, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x78, 0x24, 0x24, 0x38, 0x24, 0x24, 0x78, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x38, 0x44, 0x40, 0x40, 0x40, 0x44, 0x38, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x78, 0x24, 0x24, 0x24, 0x24, 0x24, 0x78, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x7C, 0x40, 0x40, 0x70, 0x40, 0x40, 0x7C, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x7C, 0x40, 0x40, 0x70, 0x40, 0x40, 0x40, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x38, 0x44, 0x40, 0x40, 0x4C, 0x44, 0x38, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x44, 0x44, 0x44, 0x7C, 0x44, 0x44, 0x44, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x38, 0x10, 0x10, 0x10, 0x10, 0x10, 0x38, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x04, 0x04, 0x04, 0x04, 0x04, 0x44, 0x38, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x44, 0x48, 0x50, 0x60, 0x50, 0x48, 0x44, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x40, 0x40, 0x40, 0x40, 0x40, 0x40, 0x7C, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x44, 0x6C, 0x54, 0x54, 0x44, 0x44, 0x44, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x44, 0x44, 0x64, 0x54, 0x4C, 0x44, 0x44, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x38, 0x44, 0x44, 0x44, 0x44, 0x44, 0x38, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x78, 0x44, 0x44, 0x78, 0x40, 0x40, 0x40, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x38, 0x44, 0x44, 0x44, 0x54, 0x48, 0x34, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x78, 0x44, 0x44, 0x78, 0x50, 0x48, 0x44, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x38, 0x44, 0x40, 0x38, 0x04, 0x44, 0x38, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x7C, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x44, 0x44, 0x44, 0x44, 0x44, 0x44, 0x38, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x44, 0x44, 0x44, 0x28, 0x28, 0x10, 0x10, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x44, 0x44, 0x44, 0x44, 0x54, 0x6C, 0x44, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x44, 0x44, 0x28, 0x10, 0x28, 0x44, 0x44, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x44, 0x44, 0x28, 0x10, 0x10, 0x10, 0x10, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x7C, 0x04, 0x08, 0x10, 0x20, 0x40, 0x7C, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x38, 0x20, 0x20, 0x20, 0x20, 0x20, 0x38, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x40, 0x20, 0x10, 0x08, 0x04, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x38, 0x08, 0x08, 0x08, 0x08, 0x08, 0x38, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x10, 0x38, 0x54, 0x10, 0x10, 0x10, 0x10, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x10, 0x20, 0x7C, 0x20, 0x10, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x10, 0x10, 0x10, 0x10, 0x10, 0x00, 0x10, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x28, 0x28, 0x28, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x28, 0x28, 0x7C, 0x28, 0x7C, 0x28, 0x28, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x10, 0x3C, 0x50, 0x38, 0x14, 0x78, 0x10, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x60, 0x64, 0x08, 0x10, 0x20, 0x4C, 0x0C, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x20, 0x50, 0x50, 0x20, 0x54, 0x48, 0x34, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x10, 0x10, 0x20, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x08, 0x10, 0x20, 0x20, 0x20, 0x10, 0x08, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x20, 0x10, 0x08, 0x08, 0x08, 0x10, 0x20, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x10, 0x54, 0x38, 0x38, 0x54, 0x10, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x10, 0x10, 0x7C, 0x10, 0x10, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x20, 0x20, 0x40, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x7C, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x10, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x04, 0x08, 0x10, 0x20, 0x40, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x38, 0x44, 0x4C, 0x54, 0x64, 0x44, 0x38, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x10, 0x30, 0x10, 0x10, 0x10, 0x10, 0x38, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x38, 0x44, 0x04, 0x38, 0x40, 0x40, 0x7C, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x38, 0x44, 0x04, 0x08, 0x04, 0x44, 0x38, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x08, 0x18, 0x28, 0x48, 0x7C, 0x08, 0x08, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x7C, 0x40, 0x78, 0x04, 0x04, 0x44, 0x38, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x38, 0x40, 0x40, 0x78, 0x44, 0x44, 0x38, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x7C, 0x04, 0x08, 0x10, 0x20, 0x40, 0x40, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x38, 0x44, 0x44, 0x38, 0x44, 0x44, 0x38, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x38, 0x44, 0x44, 0x3C, 0x04, 0x04, 0x38, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x10, 0x00, 0x00, 0x10, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x10, 0x00, 0x00, 0x10, 0x10, 0x20, 0x00, 0x00,
	0x00, 0x00, 0x08, 0x10, 0x20, 0x40, 0x20, 0x10, 0x08, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x7C, 0x00, 0x7C, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x20, 0x10, 0x08, 0x04, 0x08, 0x10, 0x20, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x38, 0x44, 0x04, 0x08, 0x10, 0x00, 0x10, 0x00, 0x00, 0x00,

	/* Lower case */
	0x00, 0x00, 0x18, 0x24, 0x20, 0x70, 0x20, 0x24, 0x78, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x38, 0x04, 0x3C, 0x44, 0x3C, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x40, 0x40, 0x58, 0x64, 0x44, 0x64, 0x58, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x38, 0x44, 0x40, 0x44, 0x38, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x04, 0x04, 0x34, 0x4C, 0x44, 0x4C, 0x34, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x38, 0x44, 0x7C, 0x40, 0x38, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x08, 0x14, 0x10, 0x38, 0x10, 0x10, 0x10, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x34, 0x4C, 0x44, 0x4C, 0x34, 0x04, 0x38, 0x00,
	0x00, 0x00, 0x40, 0x40, 0x58, 0x64, 0x44, 0x44, 0x44, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x10, 0x00, 0x30, 0x10, 0x10, 0x10, 0x38, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x04, 0x00, 0x04, 0x04, 0x04, 0x04, 0x44, 0x38, 0x00, 0x00,
	0x00, 0x00, 0x40, 0x40, 0x48, 0x50, 0x60, 0x50, 0x48, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x30, 0x10, 0x10, 0x10, 0x10, 0x10, 0x38, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x78, 0x54, 0x54, 0x54, 0x54, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x58, 0x64, 0x44, 0x44, 0x44, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x38, 0x44, 0x44, 0x44, 0x38, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x78, 0x44, 0x44, 0x44, 0x78, 0x40, 0x40, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x3C, 0x44, 0x44, 0x44, 0x3C, 0x04, 0x04, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x58, 0x64, 0x40, 0x40, 0x40, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x3C, 0x40, 0x38, 0x04, 0x78, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x20, 0x20, 0x70, 0x20, 0x20, 0x24, 0x18, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x44, 0x44, 0x44, 0x4C, 0x34, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x44, 0x44, 0x44, 0x28, 0x10, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x44, 0x54, 0x54, 0x28, 0x28, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x44, 0x28, 0x10, 0x28, 0x44, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x44, 0x44, 0x44, 0x3C, 0x04, 0x38, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x7C, 0x08, 0x10, 0x20, 0x7C, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x08, 0x10, 0x10, 0x20, 0x10, 0x10, 0x08, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x10, 0x10, 0x10, 0x00, 0x10, 0x10, 0x10, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x20, 0x10, 0x10, 0x08, 0x10, 0x10, 0x20, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x20, 0x54, 0x08, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x7C, 0x00, 0x00, 0x00,

	/* Semigraphics 6 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x0F, 0x0F, 0x0F, 0x0F,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xF0, 0xF0, 0xF0, 0xF0,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xFF, 0xFF, 0xFF, 0xFF,
	0x00, 0x00, 0x00, 0x00, 0x0F, 0x0F, 0x0F, 0x0F, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F,
	0x00, 0x00, 0x00, 0x00, 0x0F, 0x0F, 0x0F, 0x0F, 0xF0, 0xF0, 0xF0, 0xF0,
	0x00, 0x00, 0x00, 0x00, 0x0F, 0x0F, 0x0F, 0x0F, 0xFF, 0xFF, 0xFF, 0xFF,
	0x00, 0x00, 0x00, 0x00, 0xF0, 0xF0, 0xF0, 0xF0, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0xF0, 0xF0, 0xF0, 0xF0, 0x0F, 0x0F, 0x0F, 0x0F,
	0x00, 0x00, 0x00, 0x00, 0xF0, 0xF0, 0xF0, 0xF0, 0xF0, 0xF0, 0xF0, 0xF0,
	0x00, 0x00, 0x00, 0x00, 0xF0, 0xF0, 0xF0, 0xF0, 0xFF, 0xFF, 0xFF, 0xFF,
	0x00, 0x00, 0x00, 0x00, 0xFF, 0xFF, 0xFF, 0xFF, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0xFF, 0xFF, 0xFF, 0xFF, 0x0F, 0x0F, 0x0F, 0x0F,
	0x00, 0x00, 0x00, 0x00, 0xFF, 0xFF, 0xFF, 0xFF, 0xF0, 0xF0, 0xF0, 0xF0,
	0x00, 0x00, 0x00, 0x00, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
	0x0F, 0x0F, 0x0F, 0x0F, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x0F, 0x0F, 0x0F, 0x0F, 0x00, 0x00, 0x00, 0x00, 0x0F, 0x0F, 0x0F, 0x0F,
	0x0F, 0x0F, 0x0F, 0x0F, 0x00, 0x00, 0x00, 0x00, 0xF0, 0xF0, 0xF0, 0xF0,
	0x0F, 0x0F, 0x0F, 0x0F, 0x00, 0x00, 0x00, 0x00, 0xFF, 0xFF, 0xFF, 0xFF,
	0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x00, 0x00, 0x00, 0x00,
	0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F,
	0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0xF0, 0xF0, 0xF0, 0xF0,
	0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0xFF, 0xFF, 0xFF, 0xFF,
	0x0F, 0x0F, 0x0F, 0x0F, 0xF0, 0xF0, 0xF0, 0xF0, 0x00, 0x00, 0x00, 0x00,
	0x0F, 0x0F, 0x0F, 0x0F, 0xF0, 0xF0, 0xF0, 0xF0, 0x0F, 0x0F, 0x0F, 0x0F,
	0x0F, 0x0F, 0x0F, 0x0F, 0xF0, 0xF0, 0xF0, 0xF0, 0xF0, 0xF0, 0xF0, 0xF0,
	0x0F, 0x0F, 0x0F, 0x0F, 0xF0, 0xF0, 0xF0, 0xF0, 0xFF, 0xFF, 0xFF, 0xFF,
	0x0F, 0x0F, 0x0F, 0x0F, 0xFF, 0xFF, 0xFF, 0xFF, 0x00, 0x00, 0x00, 0x00,
	0x0F, 0x0F, 0x0F, 0x0F, 0xFF, 0xFF, 0xFF, 0xFF, 0x0F, 0x0F, 0x0F, 0x0F,
	0x0F, 0x0F, 0x0F, 0x0F, 0xFF, 0xFF, 0xFF, 0xFF, 0xF0, 0xF0, 0xF0, 0xF0,
	0x0F, 0x0F, 0x0F, 0x0F, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
	0xF0, 0xF0, 0xF0, 0xF0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0xF0, 0xF0, 0xF0, 0xF0, 0x00, 0x00, 0x00, 0x00, 0x0F, 0x0F, 0x0F, 0x0F,
	0xF0, 0xF0, 0xF0, 0xF0, 0x00, 0x00, 0x00, 0x00, 0xF0, 0xF0, 0xF0, 0xF0,
	0xF0, 0xF0, 0xF0, 0xF0, 0x00, 0x00, 0x00, 0x00, 0xFF, 0xFF, 0xFF, 0xFF,
	0xF0, 0xF0, 0xF0, 0xF0, 0x0F, 0x0F, 0x0F, 0x0F, 0x00, 0x00, 0x00, 0x00,
	0xF0, 0xF0, 0xF0, 0xF0, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F,
	0xF0, 0xF0, 0xF0, 0xF0, 0x0F, 0x0F, 0x0F, 0x0F, 0xF0, 0xF0, 0xF0, 0xF0,
	0xF0, 0xF0, 0xF0, 0xF0, 0x0F, 0x0F, 0x0F, 0x0F, 0xFF, 0xFF, 0xFF, 0xFF,
	0xF0, 0xF0, 0xF0, 0xF0, 0xF0, 0xF0, 0xF0, 0xF0, 0x00, 0x00, 0x00, 0x00,
	0xF0, 0xF0, 0xF0, 0xF0, 0xF0, 0xF0, 0xF0, 0xF0, 0x0F, 0x0F, 0x0F, 0x0F,
	0xF0, 0xF0, 0xF0, 0xF0, 0xF0, 0xF0, 0xF0, 0xF0, 0xF0, 0xF0, 0xF0, 0xF0,
	0xF0, 0xF0, 0xF0, 0xF0, 0xF0, 0xF0, 0xF0, 0xF0, 0xFF, 0xFF, 0xFF, 0xFF,
	0xF0, 0xF0, 0xF0, 0xF0, 0xFF, 0xFF, 0xFF, 0xFF, 0x00, 0x00, 0x00, 0x00,
	0xF0, 0xF0, 0xF0, 0xF0, 0xFF, 0xFF, 0xFF, 0xFF, 0x0F, 0x0F, 0x0F, 0x0F,
	0xF0, 0xF0, 0xF0, 0xF0, 0xFF, 0xFF, 0xFF, 0xFF, 0xF0, 0xF0, 0xF0, 0xF0,
	0xF0, 0xF0, 0xF0, 0xF0, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
	0xFF, 0xFF, 0xFF, 0xFF, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0xFF, 0xFF, 0xFF, 0xFF, 0x00, 0x00, 0x00, 0x00, 0x0F, 0x0F, 0x0F, 0x0F,
	0xFF, 0xFF, 0xFF, 0xFF, 0x00, 0x00, 0x00, 0x00, 0xF0, 0xF0, 0xF0, 0xF0,
	0xFF, 0xFF, 0xFF, 0xFF, 0x00, 0x00, 0x00, 0x00, 0xFF, 0xFF, 0xFF, 0xFF,
	0xFF, 0xFF, 0xFF, 0xFF, 0x0F, 0x0F, 0x0F, 0x0F, 0x00, 0x00, 0x00, 0x00,
	0xFF, 0xFF, 0xFF, 0xFF, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F,
	0xFF, 0xFF, 0xFF, 0xFF, 0x0F, 0x0F, 0x0F, 0x0F, 0xF0, 0xF0, 0xF0, 0xF0,
	0xFF, 0xFF, 0xFF, 0xFF, 0x0F, 0x0F, 0x0F, 0x0F, 0xFF, 0xFF, 0xFF, 0xFF,
	0xFF, 0xFF, 0xFF, 0xFF, 0xF0, 0xF0, 0xF0, 0xF0, 0x00, 0x00, 0x00, 0x00,
	0xFF, 0xFF, 0xFF, 0xFF, 0xF0, 0xF0, 0xF0, 0xF0, 0x0F, 0x0F, 0x0F, 0x0F,
	0xFF, 0xFF, 0xFF, 0xFF, 0xF0, 0xF0, 0xF0, 0xF0, 0xF0, 0xF0, 0xF0, 0xF0,
	0xFF, 0xFF, 0xFF, 0xFF, 0xF0, 0xF0, 0xF0, 0xF0, 0xFF, 0xFF, 0xFF, 0xFF,
	0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0x00, 0x00, 0x00, 0x00,
	0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0x0F, 0x0F, 0x0F, 0x0F,
	0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xF0, 0xF0, 0xF0, 0xF0,
	0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,

	/* Block Graphics (Semigraphics 4 Graphics ) */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xF0, 0xF0, 0xF0, 0xF0, 0xF0, 0xF0,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
	0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F,
	0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0xF0, 0xF0, 0xF0, 0xF0, 0xF0, 0xF0,
	0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
	0xF0, 0xF0, 0xF0, 0xF0, 0xF0, 0xF0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0xF0, 0xF0, 0xF0, 0xF0, 0xF0, 0xF0, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F,
	0xF0, 0xF0, 0xF0, 0xF0, 0xF0, 0xF0, 0xF0, 0xF0, 0xF0, 0xF0, 0xF0, 0xF0,
	0xF0, 0xF0, 0xF0, 0xF0, 0xF0, 0xF0, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
	0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F,
	0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xF0, 0xF0, 0xF0, 0xF0, 0xF0, 0xF0,
	0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
};



static const UINT8 pal_square_fontdata8x12[] =
{
	0x00, 0x00, 0x00, 0x1C, 0x22, 0x02, 0x1A, 0x2A, 0x2A, 0x1C, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x08, 0x14, 0x22, 0x22, 0x3E, 0x22, 0x22, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x3C, 0x12, 0x12, 0x1C, 0x12, 0x12, 0x3C, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x1C, 0x22, 0x20, 0x20, 0x20, 0x22, 0x1C, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x3C, 0x12, 0x12, 0x12, 0x12, 0x12, 0x3C, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x3E, 0x20, 0x20, 0x3C, 0x20, 0x20, 0x3E, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x3E, 0x20, 0x20, 0x3C, 0x20, 0x20, 0x20, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x1E, 0x20, 0x20, 0x26, 0x22, 0x22, 0x1E, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x22, 0x22, 0x22, 0x3E, 0x22, 0x22, 0x22, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x1C, 0x08, 0x08, 0x08, 0x08, 0x08, 0x1C, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x02, 0x02, 0x02, 0x02, 0x22, 0x22, 0x1C, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x22, 0x24, 0x28, 0x30, 0x28, 0x24, 0x22, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x3E, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x22, 0x36, 0x2A, 0x2A, 0x22, 0x22, 0x22, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x22, 0x32, 0x2A, 0x26, 0x22, 0x22, 0x22, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x3E, 0x22, 0x22, 0x22, 0x22, 0x22, 0x3E, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x3C, 0x22, 0x22, 0x3C, 0x20, 0x20, 0x20, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x1C, 0x22, 0x22, 0x22, 0x2A, 0x24, 0x1A, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x3C, 0x22, 0x22, 0x3C, 0x28, 0x24, 0x22, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x1C, 0x22, 0x10, 0x08, 0x04, 0x22, 0x1C, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x3E, 0x08, 0x08, 0x08, 0x08, 0x08, 0x08, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x22, 0x22, 0x22, 0x22, 0x22, 0x22, 0x1C, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x22, 0x22, 0x22, 0x14, 0x14, 0x08, 0x08, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x22, 0x22, 0x22, 0x2A, 0x2A, 0x36, 0x22, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x22, 0x22, 0x14, 0x08, 0x14, 0x22, 0x22, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x22, 0x22, 0x14, 0x08, 0x08, 0x08, 0x08, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x3E, 0x02, 0x04, 0x08, 0x10, 0x20, 0x3E, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x38, 0x20, 0x20, 0x20, 0x20, 0x20, 0x38, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x20, 0x20, 0x10, 0x08, 0x04, 0x02, 0x02, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x0E, 0x02, 0x02, 0x02, 0x02, 0x02, 0x0E, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x08, 0x1C, 0x2A, 0x08, 0x08, 0x08, 0x08, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x08, 0x10, 0x3E, 0x10, 0x08, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x08, 0x08, 0x08, 0x08, 0x08, 0x00, 0x08, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x14, 0x14, 0x14, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x14, 0x14, 0x36, 0x00, 0x36, 0x14, 0x14, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x08, 0x1E, 0x20, 0x1C, 0x02, 0x3C, 0x08, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x32, 0x32, 0x04, 0x08, 0x10, 0x26, 0x26, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x10, 0x28, 0x28, 0x10, 0x2A, 0x24, 0x1A, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x18, 0x18, 0x18, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x08, 0x10, 0x20, 0x20, 0x20, 0x10, 0x08, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x08, 0x04, 0x02, 0x02, 0x02, 0x04, 0x08, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x08, 0x1C, 0x3E, 0x1C, 0x08, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x08, 0x08, 0x3E, 0x08, 0x08, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x30, 0x30, 0x10, 0x20, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x3E, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x30, 0x30, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x02, 0x02, 0x04, 0x08, 0x10, 0x20, 0x20, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x18, 0x24, 0x24, 0x24, 0x24, 0x24, 0x18, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x08, 0x18, 0x08, 0x08, 0x08, 0x08, 0x1C, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x1C, 0x22, 0x02, 0x1C, 0x20, 0x20, 0x3E, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x1C, 0x22, 0x02, 0x0C, 0x02, 0x22, 0x1C, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x04, 0x0C, 0x14, 0x3E, 0x04, 0x04, 0x04, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x3E, 0x20, 0x3C, 0x02, 0x02, 0x22, 0x1C, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x1C, 0x20, 0x20, 0x3C, 0x22, 0x22, 0x1C, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x3E, 0x02, 0x04, 0x08, 0x10, 0x20, 0x20, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x1C, 0x22, 0x22, 0x1C, 0x22, 0x22, 0x1C, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x1C, 0x22, 0x22, 0x1E, 0x02, 0x02, 0x1C, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x18, 0x18, 0x00, 0x18, 0x18, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x18, 0x18, 0x00, 0x18, 0x18, 0x08, 0x10, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x04, 0x08, 0x10, 0x20, 0x10, 0x08, 0x04, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x3E, 0x00, 0x3E, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x10, 0x08, 0x04, 0x02, 0x04, 0x08, 0x10, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x18, 0x24, 0x04, 0x08, 0x08, 0x00, 0x08, 0x00, 0x00,

	/* Lower case */
	0x00, 0x00, 0x00, 0x0C, 0x12, 0x10, 0x38, 0x10, 0x12, 0x3C, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x1C, 0x02, 0x1E, 0x22, 0x1E, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x20, 0x20, 0x3C, 0x22, 0x22, 0x22, 0x3C, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x1C, 0x20, 0x20, 0x20, 0x1C, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x02, 0x02, 0x1E, 0x22, 0x22, 0x22, 0x1E, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x1C, 0x22, 0x3E, 0x20, 0x1C, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x0C, 0x12, 0x10, 0x38, 0x10, 0x10, 0x10, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x1E, 0x22, 0x22, 0x22, 0x1E, 0x02, 0x1C,
	0x00, 0x00, 0x00, 0x20, 0x20, 0x3C, 0x22, 0x22, 0x22, 0x22, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x08, 0x00, 0x18, 0x08, 0x08, 0x08, 0x1C, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x04, 0x00, 0x0C, 0x04, 0x04, 0x04, 0x04, 0x24, 0x18,
	0x00, 0x00, 0x00, 0x20, 0x20, 0x24, 0x28, 0x38, 0x24, 0x22, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x18, 0x08, 0x08, 0x08, 0x08, 0x08, 0x1C, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x3C, 0x2A, 0x2A, 0x2A, 0x2A, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x2C, 0x32, 0x22, 0x22, 0x22, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x1C, 0x22, 0x22, 0x22, 0x1C, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x3C, 0x22, 0x22, 0x22, 0x3C, 0x20, 0x20,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x1E, 0x22, 0x22, 0x22, 0x1E, 0x02, 0x03,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x2C, 0x32, 0x20, 0x20, 0x20, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x1E, 0x20, 0x1C, 0x02, 0x3C, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x10, 0x3C, 0x10, 0x10, 0x10, 0x12, 0x0C, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x22, 0x22, 0x22, 0x26, 0x1A, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x22, 0x22, 0x14, 0x14, 0x08, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x22, 0x2A, 0x2A, 0x1C, 0x14, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x22, 0x14, 0x08, 0x14, 0x22, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x22, 0x22, 0x22, 0x22, 0x1E, 0x02, 0x1C,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x3E, 0x04, 0x08, 0x10, 0x3E, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x08, 0x10, 0x10, 0x20, 0x10, 0x10, 0x08, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x08, 0x08, 0x08, 0x00, 0x08, 0x08, 0x08, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x08, 0x04, 0x04, 0x02, 0x04, 0x04, 0x08, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x08, 0x08, 0x08, 0x08, 0x2A, 0x1C, 0x08, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x08, 0x04, 0x3E, 0x04, 0x08, 0x00, 0x00, 0x00, 0x00,

	/* Semigraphics 6 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x0f, 0x0f, 0x0f, 0x0f,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xf0, 0xf0, 0xf0, 0xf0,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xff, 0xff, 0xff, 0xff,
	0x00, 0x00, 0x00, 0x00, 0x0f, 0x0f, 0x0f, 0x0f, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x0f, 0x0f, 0x0f, 0x0f, 0x0f, 0x0f, 0x0f, 0x0f,
	0x00, 0x00, 0x00, 0x00, 0x0f, 0x0f, 0x0f, 0x0f, 0xf0, 0xf0, 0xf0, 0xf0,
	0x00, 0x00, 0x00, 0x00, 0x0f, 0x0f, 0x0f, 0x0f, 0xff, 0xff, 0xff, 0xff,
	0x00, 0x00, 0x00, 0x00, 0xf0, 0xf0, 0xf0, 0xf0, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0xf0, 0xf0, 0xf0, 0xf0, 0x0f, 0x0f, 0x0f, 0x0f,
	0x00, 0x00, 0x00, 0x00, 0xf0, 0xf0, 0xf0, 0xf0, 0xf0, 0xf0, 0xf0, 0xf0,
	0x00, 0x00, 0x00, 0x00, 0xf0, 0xf0, 0xf0, 0xf0, 0xff, 0xff, 0xff, 0xff,
	0x00, 0x00, 0x00, 0x00, 0xff, 0xff, 0xff, 0xff, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0xff, 0xff, 0xff, 0xff, 0x0f, 0x0f, 0x0f, 0x0f,
	0x00, 0x00, 0x00, 0x00, 0xff, 0xff, 0xff, 0xff, 0xf0, 0xf0, 0xf0, 0xf0,
	0x00, 0x00, 0x00, 0x00, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
	0x0f, 0x0f, 0x0f, 0x0f, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x0f, 0x0f, 0x0f, 0x0f, 0x00, 0x00, 0x00, 0x00, 0x0f, 0x0f, 0x0f, 0x0f,
	0x0f, 0x0f, 0x0f, 0x0f, 0x00, 0x00, 0x00, 0x00, 0xf0, 0xf0, 0xf0, 0xf0,
	0x0f, 0x0f, 0x0f, 0x0f, 0x00, 0x00, 0x00, 0x00, 0xff, 0xff, 0xff, 0xff,
	0x0f, 0x0f, 0x0f, 0x0f, 0x0f, 0x0f, 0x0f, 0x0f, 0x00, 0x00, 0x00, 0x00,
	0x0f, 0x0f, 0x0f, 0x0f, 0x0f, 0x0f, 0x0f, 0x0f, 0x0f, 0x0f, 0x0f, 0x0f,
	0x0f, 0x0f, 0x0f, 0x0f, 0x0f, 0x0f, 0x0f, 0x0f, 0xf0, 0xf0, 0xf0, 0xf0,
	0x0f, 0x0f, 0x0f, 0x0f, 0x0f, 0x0f, 0x0f, 0x0f, 0xff, 0xff, 0xff, 0xff,
	0x0f, 0x0f, 0x0f, 0x0f, 0xf0, 0xf0, 0xf0, 0xf0, 0x00, 0x00, 0x00, 0x00,
	0x0f, 0x0f, 0x0f, 0x0f, 0xf0, 0xf0, 0xf0, 0xf0, 0x0f, 0x0f, 0x0f, 0x0f,
	0x0f, 0x0f, 0x0f, 0x0f, 0xf0, 0xf0, 0xf0, 0xf0, 0xf0, 0xf0, 0xf0, 0xf0,
	0x0f, 0x0f, 0x0f, 0x0f, 0xf0, 0xf0, 0xf0, 0xf0, 0xff, 0xff, 0xff, 0xff,
	0x0f, 0x0f, 0x0f, 0x0f, 0xff, 0xff, 0xff, 0xff, 0x00, 0x00, 0x00, 0x00,
	0x0f, 0x0f, 0x0f, 0x0f, 0xff, 0xff, 0xff, 0xff, 0x0f, 0x0f, 0x0f, 0x0f,
	0x0f, 0x0f, 0x0f, 0x0f, 0xff, 0xff, 0xff, 0xff, 0xf0, 0xf0, 0xf0, 0xf0,
	0x0f, 0x0f, 0x0f, 0x0f, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
	0xf0, 0xf0, 0xf0, 0xf0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0xf0, 0xf0, 0xf0, 0xf0, 0x00, 0x00, 0x00, 0x00, 0x0f, 0x0f, 0x0f, 0x0f,
	0xf0, 0xf0, 0xf0, 0xf0, 0x00, 0x00, 0x00, 0x00, 0xf0, 0xf0, 0xf0, 0xf0,
	0xf0, 0xf0, 0xf0, 0xf0, 0x00, 0x00, 0x00, 0x00, 0xff, 0xff, 0xff, 0xff,
	0xf0, 0xf0, 0xf0, 0xf0, 0x0f, 0x0f, 0x0f, 0x0f, 0x00, 0x00, 0x00, 0x00,
	0xf0, 0xf0, 0xf0, 0xf0, 0x0f, 0x0f, 0x0f, 0x0f, 0x0f, 0x0f, 0x0f, 0x0f,
	0xf0, 0xf0, 0xf0, 0xf0, 0x0f, 0x0f, 0x0f, 0x0f, 0xf0, 0xf0, 0xf0, 0xf0,
	0xf0, 0xf0, 0xf0, 0xf0, 0x0f, 0x0f, 0x0f, 0x0f, 0xff, 0xff, 0xff, 0xff,
	0xf0, 0xf0, 0xf0, 0xf0, 0xf0, 0xf0, 0xf0, 0xf0, 0x00, 0x00, 0x00, 0x00,
	0xf0, 0xf0, 0xf0, 0xf0, 0xf0, 0xf0, 0xf0, 0xf0, 0x0f, 0x0f, 0x0f, 0x0f,
	0xf0, 0xf0, 0xf0, 0xf0, 0xf0, 0xf0, 0xf0, 0xf0, 0xf0, 0xf0, 0xf0, 0xf0,
	0xf0, 0xf0, 0xf0, 0xf0, 0xf0, 0xf0, 0xf0, 0xf0, 0xff, 0xff, 0xff, 0xff,
	0xf0, 0xf0, 0xf0, 0xf0, 0xff, 0xff, 0xff, 0xff, 0x00, 0x00, 0x00, 0x00,
	0xf0, 0xf0, 0xf0, 0xf0, 0xff, 0xff, 0xff, 0xff, 0x0f, 0x0f, 0x0f, 0x0f,
	0xf0, 0xf0, 0xf0, 0xf0, 0xff, 0xff, 0xff, 0xff, 0xf0, 0xf0, 0xf0, 0xf0,
	0xf0, 0xf0, 0xf0, 0xf0, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
	0xff, 0xff, 0xff, 0xff, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0xff, 0xff, 0xff, 0xff, 0x00, 0x00, 0x00, 0x00, 0x0f, 0x0f, 0x0f, 0x0f,
	0xff, 0xff, 0xff, 0xff, 0x00, 0x00, 0x00, 0x00, 0xf0, 0xf0, 0xf0, 0xf0,
	0xff, 0xff, 0xff, 0xff, 0x00, 0x00, 0x00, 0x00, 0xff, 0xff, 0xff, 0xff,
	0xff, 0xff, 0xff, 0xff, 0x0f, 0x0f, 0x0f, 0x0f, 0x00, 0x00, 0x00, 0x00,
	0xff, 0xff, 0xff, 0xff, 0x0f, 0x0f, 0x0f, 0x0f, 0x0f, 0x0f, 0x0f, 0x0f,
	0xff, 0xff, 0xff, 0xff, 0x0f, 0x0f, 0x0f, 0x0f, 0xf0, 0xf0, 0xf0, 0xf0,
	0xff, 0xff, 0xff, 0xff, 0x0f, 0x0f, 0x0f, 0x0f, 0xff, 0xff, 0xff, 0xff,
	0xff, 0xff, 0xff, 0xff, 0xf0, 0xf0, 0xf0, 0xf0, 0x00, 0x00, 0x00, 0x00,
	0xff, 0xff, 0xff, 0xff, 0xf0, 0xf0, 0xf0, 0xf0, 0x0f, 0x0f, 0x0f, 0x0f,
	0xff, 0xff, 0xff, 0xff, 0xf0, 0xf0, 0xf0, 0xf0, 0xf0, 0xf0, 0xf0, 0xf0,
	0xff, 0xff, 0xff, 0xff, 0xf0, 0xf0, 0xf0, 0xf0, 0xff, 0xff, 0xff, 0xff,
	0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x00, 0x00, 0x00, 0x00,
	0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x0f, 0x0f, 0x0f, 0x0f,
	0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xf0, 0xf0, 0xf0, 0xf0,
	0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,

	/* Block Graphics (Semigraphics 4 Graphics ) */
	0x00,0x00,0x00,0x00,0x00,0x00, 0x00,0x00,0x00,0x00,0x00,0x00,
	0x00,0x00,0x00,0x00,0x00,0x00, 0x0f,0x0f,0x0f,0x0f,0x0f,0x0f,
	0x00,0x00,0x00,0x00,0x00,0x00, 0xf0,0xf0,0xf0,0xf0,0xf0,0xf0,
	0x00,0x00,0x00,0x00,0x00,0x00, 0xff,0xff,0xff,0xff,0xff,0xff,
	0x0f,0x0f,0x0f,0x0f,0x0f,0x0f, 0x00,0x00,0x00,0x00,0x00,0x00,
	0x0f,0x0f,0x0f,0x0f,0x0f,0x0f, 0x0f,0x0f,0x0f,0x0f,0x0f,0x0f,
	0x0f,0x0f,0x0f,0x0f,0x0f,0x0f, 0xf0,0xf0,0xf0,0xf0,0xf0,0xf0,
	0x0f,0x0f,0x0f,0x0f,0x0f,0x0f, 0xff,0xff,0xff,0xff,0xff,0xff,
	0xf0,0xf0,0xf0,0xf0,0xf0,0xf0, 0x00,0x00,0x00,0x00,0x00,0x00,
	0xf0,0xf0,0xf0,0xf0,0xf0,0xf0, 0x0f,0x0f,0x0f,0x0f,0x0f,0x0f,
	0xf0,0xf0,0xf0,0xf0,0xf0,0xf0, 0xf0,0xf0,0xf0,0xf0,0xf0,0xf0,
	0xf0,0xf0,0xf0,0xf0,0xf0,0xf0, 0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff, 0x00,0x00,0x00,0x00,0x00,0x00,
	0xff,0xff,0xff,0xff,0xff,0xff, 0x0f,0x0f,0x0f,0x0f,0x0f,0x0f,
	0xff,0xff,0xff,0xff,0xff,0xff, 0xf0,0xf0,0xf0,0xf0,0xf0,0xf0,
	0xff,0xff,0xff,0xff,0xff,0xff, 0xff,0xff,0xff,0xff,0xff,0xff
};



static const UINT8 ntsc_round_fontdata8x12[] =
{
	0x00, 0x00, 0x38, 0x44, 0x04, 0x34, 0x4C, 0x4C, 0x38, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x10, 0x28, 0x44, 0x44, 0x7C, 0x44, 0x44, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x78, 0x24, 0x24, 0x38, 0x24, 0x24, 0x78, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x38, 0x44, 0x40, 0x40, 0x40, 0x44, 0x38, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x78, 0x24, 0x24, 0x24, 0x24, 0x24, 0x78, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x7C, 0x40, 0x40, 0x70, 0x40, 0x40, 0x7C, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x7C, 0x40, 0x40, 0x70, 0x40, 0x40, 0x40, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x38, 0x44, 0x40, 0x40, 0x4C, 0x44, 0x38, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x44, 0x44, 0x44, 0x7C, 0x44, 0x44, 0x44, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x38, 0x10, 0x10, 0x10, 0x10, 0x10, 0x38, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x04, 0x04, 0x04, 0x04, 0x04, 0x44, 0x38, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x44, 0x48, 0x50, 0x60, 0x50, 0x48, 0x44, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x40, 0x40, 0x40, 0x40, 0x40, 0x40, 0x7C, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x44, 0x6C, 0x54, 0x54, 0x44, 0x44, 0x44, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x44, 0x44, 0x64, 0x54, 0x4C, 0x44, 0x44, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x38, 0x44, 0x44, 0x44, 0x44, 0x44, 0x38, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x78, 0x44, 0x44, 0x78, 0x40, 0x40, 0x40, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x38, 0x44, 0x44, 0x44, 0x54, 0x48, 0x34, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x78, 0x44, 0x44, 0x78, 0x50, 0x48, 0x44, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x38, 0x44, 0x40, 0x38, 0x04, 0x44, 0x38, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x7C, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x44, 0x44, 0x44, 0x44, 0x44, 0x44, 0x38, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x44, 0x44, 0x44, 0x28, 0x28, 0x10, 0x10, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x44, 0x44, 0x44, 0x44, 0x54, 0x6C, 0x44, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x44, 0x44, 0x28, 0x10, 0x28, 0x44, 0x44, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x44, 0x44, 0x28, 0x10, 0x10, 0x10, 0x10, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x7C, 0x04, 0x08, 0x10, 0x20, 0x40, 0x7C, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x38, 0x20, 0x20, 0x20, 0x20, 0x20, 0x38, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x40, 0x20, 0x10, 0x08, 0x04, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x38, 0x08, 0x08, 0x08, 0x08, 0x08, 0x38, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x10, 0x38, 0x54, 0x10, 0x10, 0x10, 0x10, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x10, 0x20, 0x7C, 0x20, 0x10, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x10, 0x10, 0x10, 0x10, 0x10, 0x00, 0x10, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x28, 0x28, 0x28, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x28, 0x28, 0x7C, 0x28, 0x7C, 0x28, 0x28, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x10, 0x3C, 0x50, 0x38, 0x14, 0x78, 0x10, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x60, 0x64, 0x08, 0x10, 0x20, 0x4C, 0x0C, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x20, 0x50, 0x50, 0x20, 0x54, 0x48, 0x34, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x10, 0x10, 0x20, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x08, 0x10, 0x20, 0x20, 0x20, 0x10, 0x08, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x20, 0x10, 0x08, 0x08, 0x08, 0x10, 0x20, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x10, 0x54, 0x38, 0x38, 0x54, 0x10, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x10, 0x10, 0x7C, 0x10, 0x10, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x20, 0x20, 0x40, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x7C, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x10, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x04, 0x08, 0x10, 0x20, 0x40, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x38, 0x44, 0x4C, 0x54, 0x64, 0x44, 0x38, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x10, 0x30, 0x10, 0x10, 0x10, 0x10, 0x38, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x38, 0x44, 0x04, 0x38, 0x40, 0x40, 0x7C, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x38, 0x44, 0x04, 0x08, 0x04, 0x44, 0x38, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x08, 0x18, 0x28, 0x48, 0x7C, 0x08, 0x08, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x7C, 0x40, 0x78, 0x04, 0x04, 0x44, 0x38, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x38, 0x40, 0x40, 0x78, 0x44, 0x44, 0x38, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x7C, 0x04, 0x08, 0x10, 0x20, 0x40, 0x40, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x38, 0x44, 0x44, 0x38, 0x44, 0x44, 0x38, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x38, 0x44, 0x44, 0x3C, 0x04, 0x04, 0x38, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x10, 0x00, 0x00, 0x10, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x10, 0x00, 0x00, 0x10, 0x10, 0x20, 0x00, 0x00,
	0x00, 0x00, 0x08, 0x10, 0x20, 0x40, 0x20, 0x10, 0x08, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x7C, 0x00, 0x7C, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x20, 0x10, 0x08, 0x04, 0x08, 0x10, 0x20, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x38, 0x44, 0x04, 0x08, 0x10, 0x00, 0x10, 0x00, 0x00, 0x00,

	/* Lower case */
	0x00, 0x00, 0x10, 0x28, 0x44, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x38, 0x04, 0x3C, 0x44, 0x3C, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x40, 0x40, 0x58, 0x64, 0x44, 0x64, 0x58, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x38, 0x44, 0x40, 0x44, 0x38, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x04, 0x04, 0x34, 0x4C, 0x44, 0x4C, 0x34, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x38, 0x44, 0x7C, 0x40, 0x38, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x08, 0x14, 0x10, 0x38, 0x10, 0x10, 0x10, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x34, 0x4C, 0x44, 0x4C, 0x34, 0x04, 0x38, 0x00,
	0x00, 0x00, 0x40, 0x40, 0x58, 0x64, 0x44, 0x44, 0x44, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x10, 0x00, 0x30, 0x10, 0x10, 0x10, 0x38, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x04, 0x00, 0x04, 0x04, 0x04, 0x04, 0x44, 0x38, 0x00, 0x00,
	0x00, 0x00, 0x40, 0x40, 0x48, 0x50, 0x60, 0x50, 0x48, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x30, 0x10, 0x10, 0x10, 0x10, 0x10, 0x38, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x78, 0x54, 0x54, 0x54, 0x54, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x58, 0x64, 0x44, 0x44, 0x44, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x38, 0x44, 0x44, 0x44, 0x38, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x78, 0x44, 0x44, 0x44, 0x78, 0x40, 0x40, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x3C, 0x44, 0x44, 0x44, 0x3C, 0x04, 0x04, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x58, 0x64, 0x40, 0x40, 0x40, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x3C, 0x40, 0x38, 0x04, 0x78, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x20, 0x20, 0x70, 0x20, 0x20, 0x24, 0x18, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x44, 0x44, 0x44, 0x4C, 0x34, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x44, 0x44, 0x44, 0x28, 0x10, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x44, 0x54, 0x54, 0x28, 0x28, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x44, 0x28, 0x10, 0x28, 0x44, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x44, 0x44, 0x44, 0x3C, 0x04, 0x38, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x7C, 0x08, 0x10, 0x20, 0x7C, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x08, 0x10, 0x10, 0x20, 0x10, 0x10, 0x08, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x10, 0x10, 0x10, 0x00, 0x10, 0x10, 0x10, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x20, 0x10, 0x10, 0x08, 0x10, 0x10, 0x20, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x20, 0x54, 0x08, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x7C, 0x00, 0x00, 0x00,

	/* Semigraphics 6 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x0F, 0x0F, 0x0F, 0x0F,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xF0, 0xF0, 0xF0, 0xF0,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xFF, 0xFF, 0xFF, 0xFF,
	0x00, 0x00, 0x00, 0x00, 0x0F, 0x0F, 0x0F, 0x0F, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F,
	0x00, 0x00, 0x00, 0x00, 0x0F, 0x0F, 0x0F, 0x0F, 0xF0, 0xF0, 0xF0, 0xF0,
	0x00, 0x00, 0x00, 0x00, 0x0F, 0x0F, 0x0F, 0x0F, 0xFF, 0xFF, 0xFF, 0xFF,
	0x00, 0x00, 0x00, 0x00, 0xF0, 0xF0, 0xF0, 0xF0, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0xF0, 0xF0, 0xF0, 0xF0, 0x0F, 0x0F, 0x0F, 0x0F,
	0x00, 0x00, 0x00, 0x00, 0xF0, 0xF0, 0xF0, 0xF0, 0xF0, 0xF0, 0xF0, 0xF0,
	0x00, 0x00, 0x00, 0x00, 0xF0, 0xF0, 0xF0, 0xF0, 0xFF, 0xFF, 0xFF, 0xFF,
	0x00, 0x00, 0x00, 0x00, 0xFF, 0xFF, 0xFF, 0xFF, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0xFF, 0xFF, 0xFF, 0xFF, 0x0F, 0x0F, 0x0F, 0x0F,
	0x00, 0x00, 0x00, 0x00, 0xFF, 0xFF, 0xFF, 0xFF, 0xF0, 0xF0, 0xF0, 0xF0,
	0x00, 0x00, 0x00, 0x00, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
	0x0F, 0x0F, 0x0F, 0x0F, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x0F, 0x0F, 0x0F, 0x0F, 0x00, 0x00, 0x00, 0x00, 0x0F, 0x0F, 0x0F, 0x0F,
	0x0F, 0x0F, 0x0F, 0x0F, 0x00, 0x00, 0x00, 0x00, 0xF0, 0xF0, 0xF0, 0xF0,
	0x0F, 0x0F, 0x0F, 0x0F, 0x00, 0x00, 0x00, 0x00, 0xFF, 0xFF, 0xFF, 0xFF,
	0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x00, 0x00, 0x00, 0x00,
	0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F,
	0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0xF0, 0xF0, 0xF0, 0xF0,
	0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0xFF, 0xFF, 0xFF, 0xFF,
	0x0F, 0x0F, 0x0F, 0x0F, 0xF0, 0xF0, 0xF0, 0xF0, 0x00, 0x00, 0x00, 0x00,
	0x0F, 0x0F, 0x0F, 0x0F, 0xF0, 0xF0, 0xF0, 0xF0, 0x0F, 0x0F, 0x0F, 0x0F,
	0x0F, 0x0F, 0x0F, 0x0F, 0xF0, 0xF0, 0xF0, 0xF0, 0xF0, 0xF0, 0xF0, 0xF0,
	0x0F, 0x0F, 0x0F, 0x0F, 0xF0, 0xF0, 0xF0, 0xF0, 0xFF, 0xFF, 0xFF, 0xFF,
	0x0F, 0x0F, 0x0F, 0x0F, 0xFF, 0xFF, 0xFF, 0xFF, 0x00, 0x00, 0x00, 0x00,
	0x0F, 0x0F, 0x0F, 0x0F, 0xFF, 0xFF, 0xFF, 0xFF, 0x0F, 0x0F, 0x0F, 0x0F,
	0x0F, 0x0F, 0x0F, 0x0F, 0xFF, 0xFF, 0xFF, 0xFF, 0xF0, 0xF0, 0xF0, 0xF0,
	0x0F, 0x0F, 0x0F, 0x0F, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
	0xF0, 0xF0, 0xF0, 0xF0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0xF0, 0xF0, 0xF0, 0xF0, 0x00, 0x00, 0x00, 0x00, 0x0F, 0x0F, 0x0F, 0x0F,
	0xF0, 0xF0, 0xF0, 0xF0, 0x00, 0x00, 0x00, 0x00, 0xF0, 0xF0, 0xF0, 0xF0,
	0xF0, 0xF0, 0xF0, 0xF0, 0x00, 0x00, 0x00, 0x00, 0xFF, 0xFF, 0xFF, 0xFF,
	0xF0, 0xF0, 0xF0, 0xF0, 0x0F, 0x0F, 0x0F, 0x0F, 0x00, 0x00, 0x00, 0x00,
	0xF0, 0xF0, 0xF0, 0xF0, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F,
	0xF0, 0xF0, 0xF0, 0xF0, 0x0F, 0x0F, 0x0F, 0x0F, 0xF0, 0xF0, 0xF0, 0xF0,
	0xF0, 0xF0, 0xF0, 0xF0, 0x0F, 0x0F, 0x0F, 0x0F, 0xFF, 0xFF, 0xFF, 0xFF,
	0xF0, 0xF0, 0xF0, 0xF0, 0xF0, 0xF0, 0xF0, 0xF0, 0x00, 0x00, 0x00, 0x00,
	0xF0, 0xF0, 0xF0, 0xF0, 0xF0, 0xF0, 0xF0, 0xF0, 0x0F, 0x0F, 0x0F, 0x0F,
	0xF0, 0xF0, 0xF0, 0xF0, 0xF0, 0xF0, 0xF0, 0xF0, 0xF0, 0xF0, 0xF0, 0xF0,
	0xF0, 0xF0, 0xF0, 0xF0, 0xF0, 0xF0, 0xF0, 0xF0, 0xFF, 0xFF, 0xFF, 0xFF,
	0xF0, 0xF0, 0xF0, 0xF0, 0xFF, 0xFF, 0xFF, 0xFF, 0x00, 0x00, 0x00, 0x00,
	0xF0, 0xF0, 0xF0, 0xF0, 0xFF, 0xFF, 0xFF, 0xFF, 0x0F, 0x0F, 0x0F, 0x0F,
	0xF0, 0xF0, 0xF0, 0xF0, 0xFF, 0xFF, 0xFF, 0xFF, 0xF0, 0xF0, 0xF0, 0xF0,
	0xF0, 0xF0, 0xF0, 0xF0, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
	0xFF, 0xFF, 0xFF, 0xFF, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0xFF, 0xFF, 0xFF, 0xFF, 0x00, 0x00, 0x00, 0x00, 0x0F, 0x0F, 0x0F, 0x0F,
	0xFF, 0xFF, 0xFF, 0xFF, 0x00, 0x00, 0x00, 0x00, 0xF0, 0xF0, 0xF0, 0xF0,
	0xFF, 0xFF, 0xFF, 0xFF, 0x00, 0x00, 0x00, 0x00, 0xFF, 0xFF, 0xFF, 0xFF,
	0xFF, 0xFF, 0xFF, 0xFF, 0x0F, 0x0F, 0x0F, 0x0F, 0x00, 0x00, 0x00, 0x00,
	0xFF, 0xFF, 0xFF, 0xFF, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F,
	0xFF, 0xFF, 0xFF, 0xFF, 0x0F, 0x0F, 0x0F, 0x0F, 0xF0, 0xF0, 0xF0, 0xF0,
	0xFF, 0xFF, 0xFF, 0xFF, 0x0F, 0x0F, 0x0F, 0x0F, 0xFF, 0xFF, 0xFF, 0xFF,
	0xFF, 0xFF, 0xFF, 0xFF, 0xF0, 0xF0, 0xF0, 0xF0, 0x00, 0x00, 0x00, 0x00,
	0xFF, 0xFF, 0xFF, 0xFF, 0xF0, 0xF0, 0xF0, 0xF0, 0x0F, 0x0F, 0x0F, 0x0F,
	0xFF, 0xFF, 0xFF, 0xFF, 0xF0, 0xF0, 0xF0, 0xF0, 0xF0, 0xF0, 0xF0, 0xF0,
	0xFF, 0xFF, 0xFF, 0xFF, 0xF0, 0xF0, 0xF0, 0xF0, 0xFF, 0xFF, 0xFF, 0xFF,
	0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0x00, 0x00, 0x00, 0x00,
	0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0x0F, 0x0F, 0x0F, 0x0F,
	0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xF0, 0xF0, 0xF0, 0xF0,
	0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,

	/* Block Graphics (Semigraphics 4 Graphics ) */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xF0, 0xF0, 0xF0, 0xF0, 0xF0, 0xF0,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
	0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F,
	0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0xF0, 0xF0, 0xF0, 0xF0, 0xF0, 0xF0,
	0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
	0xF0, 0xF0, 0xF0, 0xF0, 0xF0, 0xF0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0xF0, 0xF0, 0xF0, 0xF0, 0xF0, 0xF0, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F,
	0xF0, 0xF0, 0xF0, 0xF0, 0xF0, 0xF0, 0xF0, 0xF0, 0xF0, 0xF0, 0xF0, 0xF0,
	0xF0, 0xF0, 0xF0, 0xF0, 0xF0, 0xF0, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
	0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F,
	0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xF0, 0xF0, 0xF0, 0xF0, 0xF0, 0xF0,
	0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
};



static const UINT8 ntsc_square_fontdata8x12[] =
{
	0x00, 0x00, 0x00, 0x1C, 0x22, 0x02, 0x1A, 0x2A, 0x2A, 0x1C, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x08, 0x14, 0x22, 0x22, 0x3E, 0x22, 0x22, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x3C, 0x12, 0x12, 0x1C, 0x12, 0x12, 0x3C, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x1C, 0x22, 0x20, 0x20, 0x20, 0x22, 0x1C, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x3C, 0x12, 0x12, 0x12, 0x12, 0x12, 0x3C, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x3E, 0x20, 0x20, 0x38, 0x20, 0x20, 0x3E, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x3E, 0x20, 0x20, 0x38, 0x20, 0x20, 0x20, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x1E, 0x20, 0x20, 0x26, 0x22, 0x22, 0x1E, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x22, 0x22, 0x22, 0x3E, 0x22, 0x22, 0x22, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x1C, 0x08, 0x08, 0x08, 0x08, 0x08, 0x1C, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x02, 0x02, 0x02, 0x02, 0x22, 0x22, 0x1C, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x22, 0x24, 0x28, 0x30, 0x28, 0x24, 0x22, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x3E, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x22, 0x36, 0x2A, 0x2A, 0x22, 0x22, 0x22, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x22, 0x32, 0x2A, 0x26, 0x22, 0x22, 0x22, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x3E, 0x22, 0x22, 0x22, 0x22, 0x22, 0x3E, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x3C, 0x22, 0x22, 0x3C, 0x20, 0x20, 0x20, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x1C, 0x22, 0x22, 0x22, 0x2A, 0x24, 0x1A, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x3C, 0x22, 0x22, 0x3C, 0x28, 0x24, 0x22, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x1C, 0x22, 0x10, 0x08, 0x04, 0x22, 0x1C, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x3E, 0x08, 0x08, 0x08, 0x08, 0x08, 0x08, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x22, 0x22, 0x22, 0x22, 0x22, 0x22, 0x1C, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x22, 0x22, 0x22, 0x14, 0x14, 0x08, 0x08, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x22, 0x22, 0x22, 0x2A, 0x2A, 0x36, 0x22, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x22, 0x22, 0x14, 0x08, 0x14, 0x22, 0x22, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x22, 0x22, 0x14, 0x08, 0x08, 0x08, 0x08, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x3E, 0x02, 0x04, 0x08, 0x10, 0x20, 0x3E, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x38, 0x20, 0x20, 0x20, 0x20, 0x20, 0x38, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x20, 0x20, 0x10, 0x08, 0x04, 0x02, 0x02, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x0E, 0x02, 0x02, 0x02, 0x02, 0x02, 0x0E, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x08, 0x1C, 0x2A, 0x08, 0x08, 0x08, 0x08, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x08, 0x10, 0x3E, 0x10, 0x08, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x08, 0x08, 0x08, 0x08, 0x08, 0x00, 0x08, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x14, 0x14, 0x14, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x14, 0x14, 0x36, 0x00, 0x36, 0x14, 0x14, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x08, 0x1E, 0x20, 0x1C, 0x02, 0x3C, 0x08, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x32, 0x32, 0x04, 0x08, 0x10, 0x26, 0x26, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x10, 0x28, 0x28, 0x10, 0x2A, 0x24, 0x1A, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x18, 0x18, 0x18, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x08, 0x10, 0x20, 0x20, 0x20, 0x10, 0x08, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x08, 0x04, 0x02, 0x02, 0x02, 0x04, 0x08, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x08, 0x1C, 0x3E, 0x1C, 0x08, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x08, 0x08, 0x3E, 0x08, 0x08, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x30, 0x30, 0x10, 0x20, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x3E, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x30, 0x30, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x02, 0x02, 0x04, 0x08, 0x10, 0x20, 0x20, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x18, 0x24, 0x24, 0x24, 0x24, 0x24, 0x18, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x08, 0x18, 0x08, 0x08, 0x08, 0x08, 0x1C, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x1C, 0x22, 0x02, 0x1C, 0x20, 0x20, 0x3E, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x1C, 0x22, 0x02, 0x04, 0x02, 0x22, 0x1C, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x04, 0x0C, 0x14, 0x3E, 0x04, 0x04, 0x04, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x3E, 0x20, 0x3C, 0x02, 0x02, 0x22, 0x1C, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x1C, 0x20, 0x20, 0x3C, 0x22, 0x22, 0x1C, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x3E, 0x02, 0x04, 0x08, 0x10, 0x20, 0x20, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x1C, 0x22, 0x22, 0x1C, 0x22, 0x22, 0x1C, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x1C, 0x22, 0x22, 0x1E, 0x02, 0x02, 0x1C, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x18, 0x18, 0x00, 0x18, 0x18, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x18, 0x18, 0x00, 0x18, 0x18, 0x08, 0x10, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x04, 0x08, 0x10, 0x20, 0x10, 0x08, 0x04, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x3E, 0x00, 0x3E, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x10, 0x08, 0x04, 0x02, 0x04, 0x08, 0x10, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x18, 0x24, 0x04, 0x08, 0x08, 0x00, 0x08, 0x00, 0x00,

	/* Lower case */
	0x00, 0x00, 0x00, 0x0C, 0x12, 0x10, 0x38, 0x10, 0x12, 0x3C, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x1C, 0x02, 0x1E, 0x22, 0x1E, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x20, 0x20, 0x3C, 0x22, 0x22, 0x22, 0x3C, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x1C, 0x20, 0x20, 0x20, 0x1C, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x02, 0x02, 0x1E, 0x22, 0x22, 0x22, 0x1E, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x1C, 0x22, 0x3E, 0x20, 0x1C, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x0C, 0x12, 0x10, 0x38, 0x10, 0x10, 0x10, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x1E, 0x22, 0x22, 0x22, 0x1E, 0x02, 0x1C,
	0x00, 0x00, 0x00, 0x20, 0x20, 0x3C, 0x22, 0x22, 0x22, 0x22, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x08, 0x00, 0x18, 0x08, 0x08, 0x08, 0x1C, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x04, 0x00, 0x0C, 0x04, 0x04, 0x04, 0x04, 0x24, 0x18,
	0x00, 0x00, 0x00, 0x20, 0x20, 0x24, 0x28, 0x38, 0x24, 0x22, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x18, 0x08, 0x08, 0x08, 0x08, 0x08, 0x1C, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x3C, 0x2A, 0x2A, 0x2A, 0x2A, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x2C, 0x32, 0x22, 0x22, 0x22, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x1C, 0x22, 0x22, 0x22, 0x1C, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x3C, 0x22, 0x22, 0x22, 0x3C, 0x20, 0x20,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x1E, 0x22, 0x22, 0x22, 0x1E, 0x02, 0x03,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x2C, 0x32, 0x20, 0x20, 0x20, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x1E, 0x20, 0x1C, 0x02, 0x3C, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x10, 0x3C, 0x10, 0x10, 0x10, 0x12, 0x0C, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x22, 0x22, 0x22, 0x26, 0x1A, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x22, 0x22, 0x14, 0x14, 0x08, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x22, 0x2A, 0x2A, 0x1C, 0x14, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x22, 0x14, 0x08, 0x14, 0x22, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x22, 0x22, 0x22, 0x22, 0x1E, 0x02, 0x1C,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x3E, 0x04, 0x08, 0x10, 0x3E, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x08, 0x10, 0x10, 0x20, 0x10, 0x10, 0x08, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x08, 0x08, 0x08, 0x00, 0x08, 0x08, 0x08, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x08, 0x04, 0x04, 0x02, 0x04, 0x04, 0x08, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x08, 0x08, 0x08, 0x08, 0x2A, 0x1C, 0x08, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x08, 0x04, 0x3E, 0x04, 0x08, 0x00, 0x00, 0x00, 0x00,

	/* Semigraphics 6 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x0f, 0x0f, 0x0f, 0x0f,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xf0, 0xf0, 0xf0, 0xf0,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xff, 0xff, 0xff, 0xff,
	0x00, 0x00, 0x00, 0x00, 0x0f, 0x0f, 0x0f, 0x0f, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x0f, 0x0f, 0x0f, 0x0f, 0x0f, 0x0f, 0x0f, 0x0f,
	0x00, 0x00, 0x00, 0x00, 0x0f, 0x0f, 0x0f, 0x0f, 0xf0, 0xf0, 0xf0, 0xf0,
	0x00, 0x00, 0x00, 0x00, 0x0f, 0x0f, 0x0f, 0x0f, 0xff, 0xff, 0xff, 0xff,
	0x00, 0x00, 0x00, 0x00, 0xf0, 0xf0, 0xf0, 0xf0, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0xf0, 0xf0, 0xf0, 0xf0, 0x0f, 0x0f, 0x0f, 0x0f,
	0x00, 0x00, 0x00, 0x00, 0xf0, 0xf0, 0xf0, 0xf0, 0xf0, 0xf0, 0xf0, 0xf0,
	0x00, 0x00, 0x00, 0x00, 0xf0, 0xf0, 0xf0, 0xf0, 0xff, 0xff, 0xff, 0xff,
	0x00, 0x00, 0x00, 0x00, 0xff, 0xff, 0xff, 0xff, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0xff, 0xff, 0xff, 0xff, 0x0f, 0x0f, 0x0f, 0x0f,
	0x00, 0x00, 0x00, 0x00, 0xff, 0xff, 0xff, 0xff, 0xf0, 0xf0, 0xf0, 0xf0,
	0x00, 0x00, 0x00, 0x00, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
	0x0f, 0x0f, 0x0f, 0x0f, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x0f, 0x0f, 0x0f, 0x0f, 0x00, 0x00, 0x00, 0x00, 0x0f, 0x0f, 0x0f, 0x0f,
	0x0f, 0x0f, 0x0f, 0x0f, 0x00, 0x00, 0x00, 0x00, 0xf0, 0xf0, 0xf0, 0xf0,
	0x0f, 0x0f, 0x0f, 0x0f, 0x00, 0x00, 0x00, 0x00, 0xff, 0xff, 0xff, 0xff,
	0x0f, 0x0f, 0x0f, 0x0f, 0x0f, 0x0f, 0x0f, 0x0f, 0x00, 0x00, 0x00, 0x00,
	0x0f, 0x0f, 0x0f, 0x0f, 0x0f, 0x0f, 0x0f, 0x0f, 0x0f, 0x0f, 0x0f, 0x0f,
	0x0f, 0x0f, 0x0f, 0x0f, 0x0f, 0x0f, 0x0f, 0x0f, 0xf0, 0xf0, 0xf0, 0xf0,
	0x0f, 0x0f, 0x0f, 0x0f, 0x0f, 0x0f, 0x0f, 0x0f, 0xff, 0xff, 0xff, 0xff,
	0x0f, 0x0f, 0x0f, 0x0f, 0xf0, 0xf0, 0xf0, 0xf0, 0x00, 0x00, 0x00, 0x00,
	0x0f, 0x0f, 0x0f, 0x0f, 0xf0, 0xf0, 0xf0, 0xf0, 0x0f, 0x0f, 0x0f, 0x0f,
	0x0f, 0x0f, 0x0f, 0x0f, 0xf0, 0xf0, 0xf0, 0xf0, 0xf0, 0xf0, 0xf0, 0xf0,
	0x0f, 0x0f, 0x0f, 0x0f, 0xf0, 0xf0, 0xf0, 0xf0, 0xff, 0xff, 0xff, 0xff,
	0x0f, 0x0f, 0x0f, 0x0f, 0xff, 0xff, 0xff, 0xff, 0x00, 0x00, 0x00, 0x00,
	0x0f, 0x0f, 0x0f, 0x0f, 0xff, 0xff, 0xff, 0xff, 0x0f, 0x0f, 0x0f, 0x0f,
	0x0f, 0x0f, 0x0f, 0x0f, 0xff, 0xff, 0xff, 0xff, 0xf0, 0xf0, 0xf0, 0xf0,
	0x0f, 0x0f, 0x0f, 0x0f, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
	0xf0, 0xf0, 0xf0, 0xf0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0xf0, 0xf0, 0xf0, 0xf0, 0x00, 0x00, 0x00, 0x00, 0x0f, 0x0f, 0x0f, 0x0f,
	0xf0, 0xf0, 0xf0, 0xf0, 0x00, 0x00, 0x00, 0x00, 0xf0, 0xf0, 0xf0, 0xf0,
	0xf0, 0xf0, 0xf0, 0xf0, 0x00, 0x00, 0x00, 0x00, 0xff, 0xff, 0xff, 0xff,
	0xf0, 0xf0, 0xf0, 0xf0, 0x0f, 0x0f, 0x0f, 0x0f, 0x00, 0x00, 0x00, 0x00,
	0xf0, 0xf0, 0xf0, 0xf0, 0x0f, 0x0f, 0x0f, 0x0f, 0x0f, 0x0f, 0x0f, 0x0f,
	0xf0, 0xf0, 0xf0, 0xf0, 0x0f, 0x0f, 0x0f, 0x0f, 0xf0, 0xf0, 0xf0, 0xf0,
	0xf0, 0xf0, 0xf0, 0xf0, 0x0f, 0x0f, 0x0f, 0x0f, 0xff, 0xff, 0xff, 0xff,
	0xf0, 0xf0, 0xf0, 0xf0, 0xf0, 0xf0, 0xf0, 0xf0, 0x00, 0x00, 0x00, 0x00,
	0xf0, 0xf0, 0xf0, 0xf0, 0xf0, 0xf0, 0xf0, 0xf0, 0x0f, 0x0f, 0x0f, 0x0f,
	0xf0, 0xf0, 0xf0, 0xf0, 0xf0, 0xf0, 0xf0, 0xf0, 0xf0, 0xf0, 0xf0, 0xf0,
	0xf0, 0xf0, 0xf0, 0xf0, 0xf0, 0xf0, 0xf0, 0xf0, 0xff, 0xff, 0xff, 0xff,
	0xf0, 0xf0, 0xf0, 0xf0, 0xff, 0xff, 0xff, 0xff, 0x00, 0x00, 0x00, 0x00,
	0xf0, 0xf0, 0xf0, 0xf0, 0xff, 0xff, 0xff, 0xff, 0x0f, 0x0f, 0x0f, 0x0f,
	0xf0, 0xf0, 0xf0, 0xf0, 0xff, 0xff, 0xff, 0xff, 0xf0, 0xf0, 0xf0, 0xf0,
	0xf0, 0xf0, 0xf0, 0xf0, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
	0xff, 0xff, 0xff, 0xff, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0xff, 0xff, 0xff, 0xff, 0x00, 0x00, 0x00, 0x00, 0x0f, 0x0f, 0x0f, 0x0f,
	0xff, 0xff, 0xff, 0xff, 0x00, 0x00, 0x00, 0x00, 0xf0, 0xf0, 0xf0, 0xf0,
	0xff, 0xff, 0xff, 0xff, 0x00, 0x00, 0x00, 0x00, 0xff, 0xff, 0xff, 0xff,
	0xff, 0xff, 0xff, 0xff, 0x0f, 0x0f, 0x0f, 0x0f, 0x00, 0x00, 0x00, 0x00,
	0xff, 0xff, 0xff, 0xff, 0x0f, 0x0f, 0x0f, 0x0f, 0x0f, 0x0f, 0x0f, 0x0f,
	0xff, 0xff, 0xff, 0xff, 0x0f, 0x0f, 0x0f, 0x0f, 0xf0, 0xf0, 0xf0, 0xf0,
	0xff, 0xff, 0xff, 0xff, 0x0f, 0x0f, 0x0f, 0x0f, 0xff, 0xff, 0xff, 0xff,
	0xff, 0xff, 0xff, 0xff, 0xf0, 0xf0, 0xf0, 0xf0, 0x00, 0x00, 0x00, 0x00,
	0xff, 0xff, 0xff, 0xff, 0xf0, 0xf0, 0xf0, 0xf0, 0x0f, 0x0f, 0x0f, 0x0f,
	0xff, 0xff, 0xff, 0xff, 0xf0, 0xf0, 0xf0, 0xf0, 0xf0, 0xf0, 0xf0, 0xf0,
	0xff, 0xff, 0xff, 0xff, 0xf0, 0xf0, 0xf0, 0xf0, 0xff, 0xff, 0xff, 0xff,
	0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x00, 0x00, 0x00, 0x00,
	0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x0f, 0x0f, 0x0f, 0x0f,
	0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xf0, 0xf0, 0xf0, 0xf0,
	0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,

	/* Block Graphics (Semigraphics 4 Graphics ) */
	0x00,0x00,0x00,0x00,0x00,0x00, 0x00,0x00,0x00,0x00,0x00,0x00,
	0x00,0x00,0x00,0x00,0x00,0x00, 0x0f,0x0f,0x0f,0x0f,0x0f,0x0f,
	0x00,0x00,0x00,0x00,0x00,0x00, 0xf0,0xf0,0xf0,0xf0,0xf0,0xf0,
	0x00,0x00,0x00,0x00,0x00,0x00, 0xff,0xff,0xff,0xff,0xff,0xff,
	0x0f,0x0f,0x0f,0x0f,0x0f,0x0f, 0x00,0x00,0x00,0x00,0x00,0x00,
	0x0f,0x0f,0x0f,0x0f,0x0f,0x0f, 0x0f,0x0f,0x0f,0x0f,0x0f,0x0f,
	0x0f,0x0f,0x0f,0x0f,0x0f,0x0f, 0xf0,0xf0,0xf0,0xf0,0xf0,0xf0,
	0x0f,0x0f,0x0f,0x0f,0x0f,0x0f, 0xff,0xff,0xff,0xff,0xff,0xff,
	0xf0,0xf0,0xf0,0xf0,0xf0,0xf0, 0x00,0x00,0x00,0x00,0x00,0x00,
	0xf0,0xf0,0xf0,0xf0,0xf0,0xf0, 0x0f,0x0f,0x0f,0x0f,0x0f,0x0f,
	0xf0,0xf0,0xf0,0xf0,0xf0,0xf0, 0xf0,0xf0,0xf0,0xf0,0xf0,0xf0,
	0xf0,0xf0,0xf0,0xf0,0xf0,0xf0, 0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff, 0x00,0x00,0x00,0x00,0x00,0x00,
	0xff,0xff,0xff,0xff,0xff,0xff, 0x0f,0x0f,0x0f,0x0f,0x0f,0x0f,
	0xff,0xff,0xff,0xff,0xff,0xff, 0xf0,0xf0,0xf0,0xf0,0xf0,0xf0,
	0xff,0xff,0xff,0xff,0xff,0xff, 0xff,0xff,0xff,0xff,0xff,0xff
};



enum
{
	GREEN, YELLOW, BLUE, RED,		/*  0 -  3 */
	BUFF, CYAN, MAGENTA, ORANGE,	/*  4 -  7 */

	BLACK = 10,						/* 10 */

	DKGREEN = 12, LTGREEN = 13,		/* 12 - 13 */
	DKORANGE = 14, LTORANGE = 15,	/* 14 - 15 */
};




#define ATTOTIME_STRING_PRECISION	9

/* MC6847 video output pins: Y, phiA, phiB (Y'UV color space).
 *
 * Y voltage levels:
 * VB ("Black") = 0.72V; VWL ("White Low") = 0.65V; VWM ("White Medium") = 0.54V;
 * VWH ("White High") = 0.42V.
 *
 * phiA and phiB voltage levels:
 * VOL ("Output Low") = 1.0V; VR ("Reference"?) = 1.5V; VIH ("Input High") = 2.0V.
 *
 * Color composition:
 *
 * COLOR       |  Y  | phiA | phiB
 * ------------+-----+------+------
 * GREEN       | VWM | VOL  | VOL
 * YELLOW      | VWH | VOL  | VR
 * BLUE        | VWL | VIH  | VR
 * RED         | VWL | VR   | VIH
 * WHITE       | VWH | VR   | VR
 * CYAN        | VWM | VR   | VOL
 * MAGENTA     | VWM | VIH  | VIH
 * ORANGE      | VWM | VOL  | VIH
 * BLACK       | VB  | VR   | VR
 * DARK GREEN  | VB  | VOL  | VOL
 * DARK ORANGE | VB  | VOL  | VIH
 *
 *
 * Mapping voltages to a 0~255 range:
 *
 * Y values:
 * VB = 0; VWL = 59; VWM = 153; VWH = 255.
 *
 * phiA and phiB values:
 * VOL = -128; VR = 0; VIH = +128.
 *
 *
 * Y'UV to RGB convertion:
 * R = Y' + 1.13983 x U
 * G = Y' - 0.39465 x U - 0.58060 x V
 * B = Y' + 2.03211 x V
 *
 * So we get:
 * GREEN = Y'UV(153,-128,-128) = RGB(7,277,-107) --> #07ff00
 * YELLOW = Y'UV(255,-128,0) = RGB(255,305,-5) --> #ffff00
 * BLUE = Y'UV(59,+128,0) = RGB(59,8,319) --> #3b08ff
 * RED: Y'UV(59,0,+128) = RGB(204,-15,59) --> #cc003b
 * WHITE: Y'UV(255,0,0) = RGB(255,255,255) --> #ffffff
 * CYAN: Y'UV(153,0,-128) = RGB(7,227,153) --> #07e399
 * MAGENTA: Y'UV(153,+128,+128) = RGB(298,28,413) --> #ff1cff
 * ORANGE: Y'UV(153,-128,+128) = RGB(298,129,-107) --> #ff8100
 * BLACK: Y'UV(0,0,0) = RGB(0,0,0) --> #000000
 * DARK GREEN: Y'UV(0,-128,-128) = RGB(-145,124,-260) --> #007c00
 * DARK ORANGE: Y'UV(0,-128,+128) = RGB(145,-23,-260) --> #910000
 */

#define M6847_RGB(r,g,b)	((r << 16) | (g << 8) | (b << 0))


static const UINT32 default_palette[] =
{
	M6847_RGB(0x07, 0xff, 0x00),	/* GREEN */
	M6847_RGB(0xff, 0xff, 0x00),	/* YELLOW */
	M6847_RGB(0x3b, 0x08, 0xff),	/* BLUE */
	M6847_RGB(0xcc, 0x00, 0x3b),	/* RED */
	M6847_RGB(0xff, 0xff, 0xff),	/* BUFF */
	M6847_RGB(0x07, 0xe3, 0x99),	/* CYAN */
	M6847_RGB(0xff, 0x1c, 0xff),	/* MAGENTA */
	M6847_RGB(0xff, 0x81, 0x00),	/* ORANGE */

	M6847_RGB(0x00, 0x7c, 0x00),	/* DARK GREEN */
	M6847_RGB(0x07, 0xff, 0x00),	/* GREEN */
	M6847_RGB(0x00, 0x00, 0x00),	/* BLACK */
	M6847_RGB(0xff, 0xff, 0xff),	/* BUFF */

	M6847_RGB(0x00, 0x7c, 0x00),	/* ALPHANUMERIC DARK GREEN */
	M6847_RGB(0x07, 0xff, 0x00),	/* ALPHANUMERIC BRIGHT GREEN */
	M6847_RGB(0x91, 0x00, 0x00),	/* ALPHANUMERIC DARK ORANGE */
	M6847_RGB(0xff, 0x81, 0x00)		/* ALPHANUMERIC BRIGHT ORANGE */
};



#define FACTOR_SCANLINES_PER_FRAME	2
#define FACTOR_CLOCKS_PER_SCANLINE	2

#define GROSS_FACTOR	(FACTOR_SCANLINES_PER_FRAME * FACTOR_CLOCKS_PER_SCANLINE)

static const m6847_variant variants[] =
{
	{
		/* M6847_VERSION_ORIGINAL_NTSC */
		ntsc_square_fontdata8x12,
		FALSE, 0,
		60, 262, 227.5,				/* frames, scanlines, clocks per scanline */
		13, 25, 192, 26, 6,
		13+25+192					/* field sync period */
	},
	{
		/* M6847_VERSION_ORIGINAL_PAL */
		pal_square_fontdata8x12,
		FALSE, 0,
		50, 262, 227.5,				/* frames, scanlines, clocks per scanline */
		13, 25, 192, 26, 6,
		13+25+192					/* field sync period */
	},
	{
		/* M6847_VERSION_M6847Y_NTSC */
		ntsc_square_fontdata8x12,
		FALSE, 0,
		60, 262.5, 227.5,			/* frames, scanlines, clocks per scanline */
		13.5, 25, 192, 26, 6,
		13+25+192					/* field sync period */
	},
	{
		/* M6847_VERSION_M6847Y_PAL */
		pal_square_fontdata8x12,
		FALSE, 0,
		50, 262.5, 227.5,			/* frames, scanlines, clocks per scanline */
		13.5, 25, 192, 26, 6,
		13+25+192					/* field sync period */
	},
	{
		/* M6847_VERSION_M6847T1_NTSC */
		ntsc_round_fontdata8x12,
		TRUE, 0,
		60, 262.5, 227.5,			/* frames, scanlines, clocks per scanline */
		13.5, 25, 192, 26, 6,
		13+25+192					/* field sync period */
	},
	{
		/* M6847_VERSION_M6847T1_PAL */
		pal_round_fontdata8x12,
		TRUE, 0,
		50, 262.5, 227.5,			/* frames, scanlines, clocks per scanline */
		13.5, 25, 192, 26, 6,
		13+25+192					/* field sync period */
	},
	{
		/* M6847_VERSION_GIME_NTSC */
		ntsc_round_fontdata8x12,
		TRUE, 1,
		60, 262.5, 227.5,			/* frames, scanlines, clocks per scanline */
		13.5, 25, 192, 26, 6,
		262.5 - 3					/* field sync period */
	},
	{
		/* M6847_VERSION_GIME_PAL */
		pal_round_fontdata8x12,
		TRUE, 1,
		50, 262.5, 227.5,			/* frames, scanlines, clocks per scanline */
		13.5, 25, 192, 26, 6,
		262.5 - 3					/* field sync period */
	}
};



/*****************************************************************************
    INLINE FUNCTIONS
*****************************************************************************/

INLINE mc6847_state *get_safe_token(device_t *device)
{
	assert(device != NULL);
	assert(device->type() == MC6847);

	return (mc6847_state *)downcast<legacy_device_base *>(device)->token();
}


/*************************************
 *
 *  Utilities
 *
 *************************************/

static UINT32 color(device_t *device, int c)
{
	mc6847_state *mc6847 = get_safe_token(device);
	return mc6847->palette[c];
}

static int attr_index_from_attribute(device_t *device, UINT8 attr)
{
	int result;

	result = ((attr & (M6847_AG | M6847_AS | M6847_INTEXT | M6847_INV | M6847_CSS)) >> 3)
		| ((attr & (M6847_GM1 | M6847_GM0)) << 5);

	/* sanity check */
	assert(result < ARRAY_LENGTH(get_safe_token(device)->fontdata));
	return result;
}

static UINT8 attribute_from_attr_index(device_t *device, int attr_index)
{
	/* sanity check */
	assert(attr_index >= 0);
	assert(attr_index < ARRAY_LENGTH(get_safe_token(device)->fontdata));

	return (UINT8) ((attr_index & 0x1F) << 3)
		| ((attr_index & 0x60) >> 5);
}



/*************************************
 *
 *  Scanline operations
 *
 *************************************/

static int get_scanline(mc6847_state *mc6847)
{
	int result;
	attotime duration;

	/* get the time since last field sync */
	duration = mc6847->hs_rise_timer->start() - mc6847->fs_rise_timer->start();
	assert_always(duration.seconds == 0, "get_scanline(): duration exceeds one second");

	if (duration.attoseconds < mc6847->vblank_period)
	{
		result = -1;	/* vblank */
	}
	else
	{
		result = (duration.attoseconds - mc6847->vblank_period)
			/ mc6847->scanline_period;
	}
	return result;
}



static offs_t scanline_to_address(mc6847_state *mc6847, int scanline)
{
	if (mc6847->ag)
	{
		/* graphics */
		int gm = (mc6847->gm2 << 2) | (mc6847->gm1 << 1) | mc6847->gm0;

		switch (gm)
		{
		case 0: return (scanline / 3) * 0x10;
		case 1: return (scanline / 3) * 0x10;
		case 2: return (scanline / 3) * 0x20;
		case 3: return (scanline / 2) * 0x20;
		case 4: return (scanline / 2) * 0x20;
		case 5: return (scanline / 1) * 0x10;
		case 6: return (scanline / 1) * 0x20;
		case 7: return (scanline / 1) * 0x20;
		}
	}
	else
	{
		/* text, semi-graphics */
		return (scanline / 12) * 0x20;
	}

	/* we never get here */
	return 0;
}


/* TODO: attributes should be read on demand rather then all at once */
static UINT8 update_attributes(mc6847_state *mc6847)
{
	if (!mc6847->in_gm0_func.isnull())
		mc6847->gm0 = mc6847->in_gm0_func();
	if (!mc6847->in_gm1_func.isnull())
		mc6847->gm1 = mc6847->in_gm1_func();
	if (!mc6847->in_gm2_func.isnull())
		mc6847->gm2 = mc6847->in_gm2_func();
	if (!mc6847->in_css_func.isnull())
		mc6847->css = mc6847->in_css_func();
	if (!mc6847->in_inv_func.isnull())
		mc6847->inv = mc6847->in_inv_func();
	if (!mc6847->in_intext_func.isnull())
		mc6847->intext = mc6847->in_intext_func();
	if (!mc6847->in_as_func.isnull())
		mc6847->as = mc6847->in_as_func();
	if (!mc6847->in_ag_func.isnull())
		mc6847->ag = mc6847->in_ag_func();

	return (mc6847->ag << 7) | (mc6847->as << 6) | (mc6847->intext << 5) | (mc6847->inv << 4) |
	       (mc6847->css << 3) | (mc6847->gm2 << 2) | (mc6847->gm1 << 1) | mc6847->gm0;
}


INLINE void prepare_scanline(running_machine &machine, mc6847_state *mc6847, int xpos)
{
	UINT8 attrs, data, attr;
	int scanline;
	int i, border_color;
	int dirty;
	m6847_pixel *scanline_data;

	scanline = get_scanline(mc6847);

	if (LOG_PREPARE)
		logerror("m6847_prepare_scanline(): scanline=%d xpos=%d\n", scanline, xpos);

	if ((scanline >= 0) && (scanline < (mc6847->top_border_scanlines
		+ mc6847->display_scanlines + mc6847->bottom_border_scanlines)))
	{
		if (mc6847->using_custom)
		{
			mc6847->custom_prepare_scanline(scanline);
		}
		else
		{
			/* has the border color changed? */
			attrs = update_attributes(mc6847);

			/* commenting out cause it breaks the coco3 border,
             * the whole 'attr' handling needs to be rewritten anyway
            if (attrs != mc6847->attrs[scanline]) */
			{
				mc6847->dirty = TRUE;
				mc6847->attrs[scanline] = attrs;

				/* choose the border color */
				if (attrs & M6847_AG)
					border_color = (attrs & M6847_CSS) ? BUFF : GREEN;
				else if (mc6847->has_lowercase && (attrs & M6847_GM2))
					border_color = (attrs & M6847_CSS) ? LTORANGE : LTGREEN;
				else
					border_color = BLACK;

				/* need to use palette table directly; border colors are constant */
				mc6847->border[scanline] = mc6847->palette[border_color];
			}

			/* is this a display scanline? */
			scanline -= mc6847->top_border_scanlines;
			if ((scanline >= 0) && (scanline < mc6847->display_scanlines))
			{
				offs_t addr = scanline_to_address(mc6847, scanline);
				dirty = mc6847->dirty;
				scanline_data = mc6847->screendata[scanline];

				for (i = xpos; i < 32; i++)
				{
					data = mc6847->in_dd_func(addr + i);
					attr = update_attributes(mc6847);

					if ((data != scanline_data[i].data)	|| (attr != scanline_data[i].attr))
					{
						dirty = TRUE;
						scanline_data[i].data = data;
						scanline_data[i].attr = attr;
					}
				}
				mc6847->dirty = dirty;
			}
		}
	}
}



void m6847_video_changed(void)
{
	/* NPW 2-May-2006 - Commenting this out until we properly fix bug #878 */
	/* prepare_scanline(get_beamx() / 8); */
}



/*************************************
 *
 *  Field sync/Horizontal sync timers
 *
 *************************************/

static TIMER_CALLBACK( hs_fall )
{
	mc6847_state *mc6847 = (mc6847_state *) ptr;

	if (LOG_HS)
		logerror("hs_fall(): time=%s\n", machine.time().as_string(ATTOTIME_STRING_PRECISION));

	mc6847->hs = ASSERT_LINE;
	mc6847->out_hs_func(mc6847->hs);
}

static TIMER_CALLBACK( hs_rise )
{
	mc6847_state *mc6847 = (mc6847_state *) ptr;

	if (LOG_HS)
		logerror("hs_rise(): time=%s\n", machine.time().as_string(ATTOTIME_STRING_PRECISION));

	mc6847->hs_rise_timer->adjust(
		attotime(0, mc6847->scanline_period), 0);
	mc6847->hs_fall_timer->adjust(
		attotime(0, mc6847->horizontal_sync_period), 0);

	mc6847->hs = CLEAR_LINE;
	mc6847->out_hs_func(mc6847->hs);

	prepare_scanline(machine, mc6847, 0);
}

static TIMER_CALLBACK( fs_fall )
{
	mc6847_state *mc6847 = (mc6847_state *) ptr;

	if (LOG_FS)
		logerror("fs_fall(): time=%s scanline=%d\n", machine.time().as_string(ATTOTIME_STRING_PRECISION), get_scanline(mc6847));

	mc6847->fs = ASSERT_LINE;
	mc6847->out_fs_func(mc6847->fs);
}

static TIMER_CALLBACK( fs_rise )
{
	mc6847_state *mc6847 = (mc6847_state *) ptr;

	if (LOG_FS)
		logerror("fs_rise(): time=%s scanline=%d\n", machine.time().as_string(ATTOTIME_STRING_PRECISION), get_scanline(mc6847));

	/* adjust field sync falling edge timer */
	mc6847->fs_fall_timer->adjust(
		attotime(0, mc6847->field_sync_period), 0);

	/* adjust horizontal sync rising timer */
	mc6847->hs_rise_timer->adjust(attotime::zero);

	/* this is a hook for the CoCo 3 code to extend this stuff */
	if (mc6847->new_frame_callback)
		mc6847->using_custom = !mc6847->new_frame_callback(machine);

	mc6847->fs = CLEAR_LINE;
	mc6847->out_fs_func(mc6847->fs);
}



#if 0
/*************************************
 *
 *  Debugging
 *
 *************************************/


static int get_beamx(mc6847_state *mc6847)
{
	attotime scanline_time;
	int result;

	scanline_time = mc6847->hs_rise_timer->elapsed();
	if (scanline_time.seconds != 0)
		return 0;
	assert(scanline.as_attoseconds() < mc6847->scanline_period);

	if (scanline.as_attoseconds() < (mc6847->clock_period * 42))
	{
		/* hsync */
		result = 0;
	}
	else if (scanline.as_attoseconds() < (mc6847->clock_period * 95 / 2))
	{
		/* left border */
		result = 0;
	}
	else if (scanline.as_attoseconds() < (mc6847->clock_period * 351 / 2))
	{
		/* body */
		result = (scanline.as_attoseconds() - (mc6847->clock_period * 95 / 2))
			/ mc6847->clock_period * 2;
	}
	else
	{
		/* right border */
		result = 256;
	}
	return result;
}


static void execute_m6847_dumpscanline(device_t *device, int ref, int params, const char **param)
{
	mc6847_state *mc6847 = get_safe_token(device);

	int i;
	int beamx = get_beamx(mc6847) / 8;
	int scanline = get_scanline(mc6847);
	const m6847_pixel *pixel = mc6847->screendata[scanline];

	for (i = 0; i < beamx; i++)
	{
		debug_console_printf(device->machine(), "[%02d]: 0x%02X (", i, pixel[i].data);

		if (pixel[i].attr & M6847_AG)		debug_console_printf(device->machine(), " AG");
		if (pixel[i].attr & M6847_AS)		debug_console_printf(device->machine(), " AS");
		if (pixel[i].attr & M6847_INTEXT)	debug_console_printf(device->machine(), " INTEXT");
		if (pixel[i].attr & M6847_INV)		debug_console_printf(device->machine(), " INV");
		if (pixel[i].attr & M6847_CSS)		debug_console_printf(device->machine(), " CSS");
		if (pixel[i].attr & M6847_GM2)		debug_console_printf(device->machine(), " GM2");
		if (pixel[i].attr & M6847_GM1)		debug_console_printf(device->machine(), " GM1");
		if (pixel[i].attr & M6847_GM0)		debug_console_printf(device->machine(), " GM0");

		debug_console_printf(device->machine(), " )\n");
	}
}
#endif



/*************************************
 *
 *  Initialization
 *
 *************************************/

static const UINT8 *find_char(const m6847_variant *v,
	UINT8 byte, UINT8 attr, int *fg, int *bg)
{
	int ch;
	size_t offset = 0;

	if (attr & M6847_AS)
	{
		/* semigraphics */
		*bg = BLACK;

		if (!v->has_lowercase && (attr & M6847_INTEXT))
		{
			/* semigraphics 6 */
			ch = (byte & 0x3F) + 0x60;
			*fg = ((byte >> 6) & 0x03) + ((attr & M6847_CSS) ? BUFF : GREEN);
		}
		else
		{
			/* semigraphics 4 */
			ch = (byte & 0x0F) + 0xA0;
			*fg = ((byte >> 4) & 0x07) + GREEN;
		}
	}
	else
	{
		/* text */
		ch = (byte & 0x3F) + 0x00;

		if (v->has_lowercase)
		{
			if ((ch < 0x20) && (attr & M6847_GM0) && !(attr & M6847_INV))
			{
				/* this is a lowercase character */
				attr |= M6847_INV;
				ch += 0x40;
			}

			if ((ch > 0x20) && (attr & M6847_GM0) && !(attr & M6847_INV)) {
				ch += 0x20;
			}

			if (attr & M6847_GM1)
				attr ^= M6847_INV;

		}

		if (attr & M6847_INV)
		{
			/* dark foreground, light background */
			*fg = (attr & M6847_CSS) ? DKORANGE : DKGREEN;
			*bg = (attr & M6847_CSS) ? LTORANGE : LTGREEN;
		}
		else
		{
			/* light foreground, dark background */
			*fg = (attr & M6847_CSS) ? LTORANGE : LTGREEN;
			*bg = (attr & M6847_CSS) ? DKORANGE : DKGREEN;
		}

		offset = v->text_offset;
	}

	return &v->fontdata[ch * 12] + offset;
}



static void build_fontdata(device_t *device, const m6847_variant *v)
{
	mc6847_state *mc6847 = get_safe_token(device);

	int attr_index, row;
	int fg, bg;
	int byte;
	const UINT8 *char_data;

	for (attr_index = 0; attr_index < ARRAY_LENGTH(mc6847->fontdata); attr_index++)
	{
		UINT8 attr = attribute_from_attr_index(device, attr_index);

		for (byte = 0; byte < 256; byte++)
		{
			char_data = find_char(v, byte, attr, &fg, &bg);

			/* specify colors */
			mc6847->colordata[attr_index][byte / 16][0] = bg;
			mc6847->colordata[attr_index][byte / 16][1] = fg;

			for (row = 0; row < 12; row++)
			{
				mc6847->fontdata[attr_index][byte][row] = char_data[row];
			}
		}
	}
}



/*************************************
 *
 *  Video modes
 *
 *************************************/

static void graphics_color_64(device_t *device, UINT32 *RESTRICT line, const m6847_pixel *RESTRICT video_data)
{
	int x;
	UINT8 byte, attr;

	for (x = 0; x < 16; x++)
	{
		byte = video_data[x].data;
		attr = video_data[x].attr;

		line[ 0] = line[ 1] = line[ 2] = line[ 3] = color(device, ((byte >> 6) & 0x03) + (attr & M6847_CSS ? BUFF : GREEN));
		line[ 4] = line[ 5] = line[ 6] = line[ 7] = color(device, ((byte >> 4) & 0x03) + (attr & M6847_CSS ? BUFF : GREEN));
		line[ 8] = line[ 9] = line[10] = line[11] = color(device, ((byte >> 2) & 0x03) + (attr & M6847_CSS ? BUFF : GREEN));
		line[12] = line[13] = line[14] = line[15] = color(device, ((byte >> 0) & 0x03) + (attr & M6847_CSS ? BUFF : GREEN));
		line += 16;
	}
}



static void graphics_color_128(device_t *device, UINT32 *RESTRICT line, const m6847_pixel *RESTRICT video_data)
{
	int x;
	UINT8 byte, attr;

	for (x = 0; x < 32; x++)
	{
		byte = video_data[x].data;
		attr = video_data[x].attr;

		line[ 0] = line[ 1] = color(device, ((byte >> 6) & 0x03) + (attr & M6847_CSS ? BUFF : GREEN));
		line[ 2] = line[ 3] = color(device, ((byte >> 4) & 0x03) + (attr & M6847_CSS ? BUFF : GREEN));
		line[ 4] = line[ 5] = color(device, ((byte >> 2) & 0x03) + (attr & M6847_CSS ? BUFF : GREEN));
		line[ 6] = line[ 7] = color(device, ((byte >> 0) & 0x03) + (attr & M6847_CSS ? BUFF : GREEN));
		line += 8;
	}
}



static void graphics_bw_128(device_t *device, UINT32 *RESTRICT line, const m6847_pixel *RESTRICT video_data)
{
	int x;
	UINT8 byte, attr;
	UINT32 bg, fg;

	for (x = 0; x < 16; x++)
	{
		byte = video_data[x].data;
		attr = video_data[x].attr;

		bg = color(device, attr & M6847_CSS ? 10 :  8);
		fg = color(device, attr & M6847_CSS ? 11 :  9);

		line[ 0] = line[ 1] = ((byte >> 7) & 0x01) ? fg : bg;
		line[ 2] = line[ 3] = ((byte >> 6) & 0x01) ? fg : bg;
		line[ 4] = line[ 5] = ((byte >> 5) & 0x01) ? fg : bg;
		line[ 6] = line[ 7] = ((byte >> 4) & 0x01) ? fg : bg;
		line[ 8] = line[ 9] = ((byte >> 3) & 0x01) ? fg : bg;
		line[10] = line[11] = ((byte >> 2) & 0x01) ? fg : bg;
		line[12] = line[13] = ((byte >> 1) & 0x01) ? fg : bg;
		line[14] = line[15] = ((byte >> 0) & 0x01) ? fg : bg;
		line += 16;
	}
}



static void graphics_bw_256(device_t *device, UINT32 *RESTRICT line, const m6847_pixel *RESTRICT video_data)
{
	int x;
	UINT8 byte, attr;
	UINT32 bg, fg;

	for (x = 0; x < 32; x++)
	{
		byte = video_data[x].data;
		attr = video_data[x].attr;

		bg = color(device, attr & M6847_CSS ? 10 :  8);
		fg = color(device, attr & M6847_CSS ? 11 :  9);

		line[0] = ((byte >> 7) & 0x01) ? fg : bg;
		line[1] = ((byte >> 6) & 0x01) ? fg : bg;
		line[2] = ((byte >> 5) & 0x01) ? fg : bg;
		line[3] = ((byte >> 4) & 0x01) ? fg : bg;
		line[4] = ((byte >> 3) & 0x01) ? fg : bg;
		line[5] = ((byte >> 2) & 0x01) ? fg : bg;
		line[6] = ((byte >> 1) & 0x01) ? fg : bg;
		line[7] = ((byte >> 0) & 0x01) ? fg : bg;
		line += 8;
	}
}



static void (*const graphics_modes[8])(device_t *device, UINT32 *line, const m6847_pixel *video_data) =
{
	graphics_color_64,	graphics_bw_128,
	graphics_color_128,	graphics_bw_128,
	graphics_color_128,	graphics_bw_128,
	graphics_color_128,	graphics_bw_256
};



static void text_mode(device_t *device, int scanline, UINT32 *RESTRICT line, const m6847_pixel *RESTRICT video_data)
{
	mc6847_state *mc6847 = get_safe_token(device);
	int x;
	UINT8 byte, attr;
	int attr_index;
	UINT32 bg_color, fg_color;
	UINT8 char_data;

	for (x = 0; x < 32; x++)
	{
		byte = video_data[x].data;
		attr = video_data[x].attr;

		attr_index = attr_index_from_attribute(device, attr);

		bg_color = color(device, mc6847->colordata[attr_index][byte / 16][0]);
		fg_color = color(device, mc6847->colordata[attr_index][byte / 16][1]);

		if( (!mc6847->has_lowercase) && (attr & M6847_INTEXT) && !(attr & M6847_AS)) {
			if (mc6847->get_char_rom) {
				char_data = mc6847->get_char_rom(device->machine(),byte,scanline % 12);
			} else {
				char_data = 0xff;
			}
		} else {
			if ((attr & M6847_INTEXT) && !(attr & M6847_AS) && mc6847->get_char_rom)
				char_data = mc6847->get_char_rom(device->machine(),byte,scanline % 12);
			else
				char_data = mc6847->fontdata[attr_index][byte][scanline % 12];
		}

		line[x*8+0] = (char_data & 0x80) ? fg_color : bg_color;
		line[x*8+1] = (char_data & 0x40) ? fg_color : bg_color;
		line[x*8+2] = (char_data & 0x20) ? fg_color : bg_color;
		line[x*8+3] = (char_data & 0x10) ? fg_color : bg_color;
		line[x*8+4] = (char_data & 0x08) ? fg_color : bg_color;
		line[x*8+5] = (char_data & 0x04) ? fg_color : bg_color;
		line[x*8+6] = (char_data & 0x02) ? fg_color : bg_color;
		line[x*8+7] = (char_data & 0x01) ? fg_color : bg_color;
	}
}



static void render_scanline(device_t *device, bitmap_t *bitmap, int scanline)
{
	mc6847_state *mc6847 = get_safe_token(device);
	UINT32 border_color;
	UINT32 *line;
	UINT8 attrs;
	int mode, x;
	const m6847_pixel *video_data;

	/* get the scanline */
	line = BITMAP_ADDR32(bitmap, scanline, 0);

	/* choose the border color */
	border_color = mc6847->border[scanline];
	attrs = mc6847->attrs[scanline];

	/* is this a border scanline? */
	scanline -= mc6847->top_border_scanlines;
	if ((scanline >= 0) && (scanline < mc6847->display_scanlines))
	{
		video_data = mc6847->screendata[scanline];

		/* left border */
		for (x = 0; x < 32; x++)
			line[x] = border_color;

		if (attrs & M6847_AG)
		{
			/* graphics */
			mode = attrs & (M6847_GM2|M6847_GM1|M6847_GM0);
			graphics_modes[mode](device, line + 32, video_data);
		}
		else
		{
			/* text/semigraphics */
			text_mode(device, scanline, line + 32, video_data);
		}

		/* right border */
		for (x = 288; x < 320; x++)
			line[x] = border_color;

		/* special case for artifacting */
		if ((attrs & (M6847_AG|M6847_GM2|M6847_GM1|M6847_GM0))
			== (M6847_AG|M6847_GM2|M6847_GM1|M6847_GM0))
		{
			apply_artifacts(device, line + 32);
		}
	}
	else
	{
		/* we are drawing a border */
		for (x = 0; x < bitmap->width; x++)
			line[x] = border_color;
	}
}



static void set_dirty(mc6847_state *mc6847)
{
	mc6847->dirty = TRUE;
}



/*****************************************************************************
    DEVICE INTERFACE
*****************************************************************************/

static void mc6847_postload(mc6847_state *state)
{
	set_dirty(state);
}

static DEVICE_START( mc6847 )
{
	mc6847_state *mc6847 = get_safe_token(device);
	const mc6847_interface *intf = (mc6847_interface *)device->static_config();
	const mc6847_config *cfg = (mc6847_config *)downcast<const legacy_device_base *>(device)->inline_config();

	const m6847_variant *v;
	UINT32 frequency;
	attoseconds_t period, frame_period;
	double total_scanlines;

	/* validate some basic stuff */
	assert(device->static_config() != NULL);
	assert(downcast<const legacy_device_base *>(device)->inline_config() != NULL);

	/* identify proper M6847 variant */
	assert(cfg->type < ARRAY_LENGTH(variants));
	v = &variants[cfg->type];

	/* allocate instance */
	set_dirty(mc6847);

	/* copy configuration */
	mc6847->new_frame_callback = cfg->new_frame_callback;
	mc6847->custom_prepare_scanline = cfg->custom_prepare_scanline;
	mc6847->has_lowercase = v->has_lowercase;
	mc6847->get_char_rom = cfg->get_char_rom;

	/* assert our assumptions */
	assert((v->scanlines_per_frame * FACTOR_SCANLINES_PER_FRAME)
		== (UINT32) (v->scanlines_per_frame * FACTOR_SCANLINES_PER_FRAME));
	assert((v->clocks_per_scanline * FACTOR_CLOCKS_PER_SCANLINE)
		== (UINT32) (v->clocks_per_scanline * FACTOR_CLOCKS_PER_SCANLINE));
	assert((v->scanlines_per_frame * v->clocks_per_scanline * GROSS_FACTOR)
		== (UINT32) (v->scanlines_per_frame * v->clocks_per_scanline * GROSS_FACTOR));

	/* choose palette */
	if (cfg->custom_palette)
	{
		mc6847->has_custom_palette = 1;
		mc6847->palette = cfg->custom_palette;
	}
	else
	{
		mc6847->palette = default_palette;
	}

	/* allocate timers */
	mc6847->fs_rise_timer = device->machine().scheduler().timer_alloc(FUNC(fs_rise), mc6847);
	mc6847->fs_fall_timer = device->machine().scheduler().timer_alloc(FUNC(fs_fall), mc6847);
	mc6847->hs_rise_timer = device->machine().scheduler().timer_alloc(FUNC(hs_rise), mc6847);
	mc6847->hs_fall_timer = device->machine().scheduler().timer_alloc(FUNC(hs_fall), mc6847);

	/* setup dimensions */
	mc6847->top_border_scanlines = v->top_border_scanlines;
	mc6847->display_scanlines = v->display_scanlines;
	mc6847->bottom_border_scanlines = v->bottom_border_scanlines;

	/* compute frequency and period */
	frequency = v->frames_per_second
		* ((UINT32) (v->scanlines_per_frame * FACTOR_SCANLINES_PER_FRAME))
		* ((UINT32) (v->clocks_per_scanline * FACTOR_CLOCKS_PER_SCANLINE));
	period = ATTOSECONDS_PER_SECOND / frequency;

	/* calculate timing */
	total_scanlines = v->vblank_scanlines
		+ v->top_border_scanlines
		+ v->display_scanlines
		+ v->bottom_border_scanlines
		+ v->vretrace_scanlines;
	mc6847->clock_period = period * GROSS_FACTOR;
	mc6847->scanline_period = period * (UINT32) (v->clocks_per_scanline * GROSS_FACTOR);
	mc6847->field_sync_period = period * (UINT32) (v->clocks_per_scanline * v->field_sync_scanlines * GROSS_FACTOR);
	mc6847->horizontal_sync_period = period * GROSS_FACTOR * 33 / 2;
	mc6847->vblank_period = period * (UINT32) (v->clocks_per_scanline * v->vblank_scanlines * GROSS_FACTOR);

	/* setup timing */
	frame_period = period *
		(UINT32) (v->clocks_per_scanline * total_scanlines * GROSS_FACTOR);
	mc6847->fs_rise_timer->adjust(attotime::zero, 0, attotime(0, frame_period));

	/* build font */
	build_fontdata(device, v);

	/* dump stats */
	if (LOG_STATS)
	{
		logerror("m6847_init():\n");
		logerror("\tclock:      %30s sec\n", attotime(0, period * GROSS_FACTOR).as_string(ATTOTIME_STRING_PRECISION));
		logerror("\tscanline:   %30s sec\n", attotime(0, mc6847->scanline_period).as_string(ATTOTIME_STRING_PRECISION));
		logerror("\tfield sync: %30s sec\n", attotime(0, mc6847->field_sync_period).as_string(ATTOTIME_STRING_PRECISION));
		logerror("\thorz sync:  %30s sec\n", attotime(0, mc6847->horizontal_sync_period).as_string(ATTOTIME_STRING_PRECISION));
		logerror("\tvblank:     %30s sec\n", attotime(0, mc6847->vblank_period).as_string(ATTOTIME_STRING_PRECISION));
		logerror("\tframe:      %30s sec\n", attotime(0, frame_period).as_string(ATTOTIME_STRING_PRECISION));
		logerror("\n");
	}

#if 0
	/* setup debug commands */
	if (device->machine().debug_flags & DEBUG_FLAG_ENABLED)
		debug_console_register_command(device->machine(), "m6847_dumpscanline", CMDFLAG_NONE, 0, 0, 0, execute_m6847_dumpscanline);
#endif


	/* resolve callbacks */
	mc6847->in_dd_func.resolve(intf->in_dd_func, *device);
	mc6847->in_gm2_func.resolve(intf->in_gm2_func, *device);
	mc6847->in_gm1_func.resolve(intf->in_gm1_func, *device);
	mc6847->in_gm0_func.resolve(intf->in_gm0_func, *device);
	mc6847->in_intext_func.resolve(intf->in_intext_func, *device);
	mc6847->in_inv_func.resolve(intf->in_inv_func, *device);
	mc6847->in_as_func.resolve(intf->in_as_func, *device);
	mc6847->in_ag_func.resolve(intf->in_ag_func, *device);
	mc6847->out_fs_func.resolve(intf->out_fs_func, *device);
	mc6847->out_hs_func.resolve(intf->out_hs_func, *device);
	mc6847->in_css_func.resolve(intf->in_css_func, *device);

	/* setup save states */
	device->machine().save().register_postload(save_prepost_delegate(FUNC(mc6847_postload), mc6847));
}

static DEVICE_RESET( mc6847 )
{
	mc6847_state *mc6847 = get_safe_token(device);

	mc6847->gm2 = 0;
	mc6847->gm1 = 0;
	mc6847->gm0 = 0;
	mc6847->intext = 0;
	mc6847->inv = 0;
	mc6847->as = 0;
	mc6847->ag = 0;
	mc6847->css = 0;
}

DEVICE_GET_INFO( mc6847 )
{
	switch (state)
	{
		/* --- the following bits of info are returned as 64-bit signed integers --- */
		case DEVINFO_INT_TOKEN_BYTES:			info->i = sizeof(mc6847_state);					break;
		case DEVINFO_INT_INLINE_CONFIG_BYTES:	info->i = sizeof(mc6847_config);				break;

		/* --- the following bits of info are returned as pointers to data or functions --- */
		case DEVINFO_FCT_START:					info->start = DEVICE_START_NAME(mc6847);		break;
		case DEVINFO_FCT_STOP:					/* Nothing */									break;
		case DEVINFO_FCT_RESET:					info->reset = DEVICE_RESET_NAME(mc6847);		break;

		/* --- the following bits of info are returned as NULL-terminated strings --- */
		case DEVINFO_STR_NAME:					strcpy(info->s, "Motorola 6847");				break;
		case DEVINFO_STR_FAMILY:				strcpy(info->s, "MC6847 VDG");					break;
		case DEVINFO_STR_VERSION:				strcpy(info->s, "1.0");							break;
		case DEVINFO_STR_SOURCE_FILE:			strcpy(info->s, __FILE__);						break;
		case DEVINFO_STR_CREDITS:				strcpy(info->s, "Copyright MESS Team");			break;
	}
}


/***************************************************************************
    IMPLEMENTATION
***************************************************************************/

WRITE_LINE_DEVICE_HANDLER( mc6847_gm2_w )
{
	mc6847_state *mc6847 = get_safe_token(device);
	mc6847->gm2 = state;
}

WRITE_LINE_DEVICE_HANDLER( mc6847_gm1_w )
{
	mc6847_state *mc6847 = get_safe_token(device);
	mc6847->gm1 = state;
}

WRITE_LINE_DEVICE_HANDLER( mc6847_gm0_w )
{
	mc6847_state *mc6847 = get_safe_token(device);
	mc6847->gm0 = state;
}

WRITE_LINE_DEVICE_HANDLER( mc6847_intext_w )
{
	mc6847_state *mc6847 = get_safe_token(device);
	mc6847->intext = state;
}

WRITE_LINE_DEVICE_HANDLER( mc6847_inv_w )
{
	mc6847_state *mc6847 = get_safe_token(device);
	mc6847->inv = state;
}

WRITE_LINE_DEVICE_HANDLER( mc6847_as_w )
{
	mc6847_state *mc6847 = get_safe_token(device);
	mc6847->as = state;
}

WRITE_LINE_DEVICE_HANDLER( mc6847_ag_w )
{
	mc6847_state *mc6847 = get_safe_token(device);
	mc6847->ag = state;
}

WRITE_LINE_DEVICE_HANDLER( mc6847_css_w )
{
	mc6847_state *mc6847 = get_safe_token(device);
	mc6847->css = state;
}

READ_LINE_DEVICE_HANDLER( mc6847_fs_r )
{
	mc6847_state *mc6847 = get_safe_token(device);
	return mc6847->fs;
}

READ_LINE_DEVICE_HANDLER( mc6847_hs_r )
{
	mc6847_state *mc6847 = get_safe_token(device);
	return mc6847->hs;
}


void mc6847_set_palette(device_t *device, UINT32 *palette)
{
	mc6847_state *mc6847 = get_safe_token(device);
	mc6847->has_custom_palette = 1;
	mc6847->palette = palette;
}


/*************************************
 *
 *  Artifacting hack; should be in core
 *
 *************************************/

static UINT32 mix_color(double factor, UINT8 c0, UINT8 c1)
{
	return (UINT32) (UINT8) ((c0 * (1.0 - factor)) + (c1 * (0.0 + factor)) + 0.5);
}



static void apply_artifacts(device_t *device, UINT32 *line)
{
	/* Boy this code sucks; this code was adapted from the old M6847
     * artifacting implmentation.  The only reason that it didn't look as
     * horrible was because the code around it sucked as well.  Now that I
     * have cleaned everything up, the ugliness is much more prominent.
     *
     * Hopefully we will have a generic artifacting algorithm that plugs into
     * the MESS/MAME core directly so we can chuck this hack */
	static const double artifact_colors[14*3] =
	{
		0.157, 0.000, 0.157, /* [ 1] - dk purple   (reverse  2) */
		0.000, 0.157, 0.000, /* [ 2] - dk green    (reverse  1) */
		1.000, 0.824, 1.000, /* [ 3] - lt purple   (reverse  4) */
		0.824, 1.000, 0.824, /* [ 4] - lt green    (reverse  3) */
		0.706, 0.236, 0.118, /* [ 5] - dk blue     (reverse  6) */
		0.000, 0.197, 0.471, /* [ 6] - dk red      (reverse  5) */
		1.000, 0.550, 0.393, /* [ 7] - lt blue     (reverse  8) */
		0.275, 0.785, 1.000, /* [ 8] - lt red      (reverse  7) */
		0.000, 0.500, 1.000, /* [ 9] - red         (reverse 10) */
		1.000, 0.500, 0.000, /* [10] - blue        (reverse  9) */
		1.000, 0.942, 0.785, /* [11] - cyan        (reverse 12) */
		0.393, 0.942, 1.000, /* [12] - yellow      (reverse 11) */
		0.236, 0.000, 0.000, /* [13] - black-blue  (reverse 14) */
		0.000, 0.000, 0.236  /* [14] - black-red   (reverse 13) */
	};

	static const UINT8 artifactcorrection[128] =
	{
		0,  0,		 0,	 0,		 0,  6,		 0,	 2,
		5,  7,		 5,	 7,		 1,  3,		 1, 11,
		8,  6,		 8, 14,		 8,  9,		 8,	 9,
		4,  4,		 4, 15,		12, 12,		12, 15,

		5, 13,		 5, 13,		13,  0,		13,	 2,
		10, 10,		10, 10,		10, 15,		10, 11,
		3,  1,		 3,	 1,		15,  9,		15,	 9,
		11, 11,		11, 11,		15, 15,		15, 15,

		14,  0,		14,	 0,		14,  6,		14,	 2,
		0,  7,		 0,	 7,		 1,  3,		 1, 11,
		9,  6,		 9, 14,		 9,  9,		 9,	 9,
		15,  4,		15, 15,		12, 12,		12, 15,

		2, 13,		 2, 13,		 2,  0,		 2,	 2,
		10, 10,		10, 10,		10, 15,		10, 11,
		12,  1,		12,	 1,		12,  9,		12,	 9,
		15, 11,		15, 11,		15, 15,		15, 15
	};

	mc6847_state *mc6847 = get_safe_token(device);
	UINT32 artifacting, c0, c1;
	UINT32 colors[16];
	const double *factors;
	UINT8 val;
	UINT32 new_line[256];
	int i;

	/* are we artifacting? */
	artifacting = input_port_read_safe(device->machine(), "artifacting", 0x00) & 0x03;
	if (artifacting == 0x00)
		return;
	artifacting &= 0x01;

	/* determine the base artifact colors */
	c1 = line[-1];
	for (i = 0; i < 256; i++)
	{
		if (line[i] != c1)
			break;
	}
	if (i >= 256)
		return;	/* if everything is the same color as the border, no need for anything */
	c0 = line[i];

	/* do we need to update our artifact colors table? */
	if ((artifacting != mc6847->saved_artifacting) || (c0 != mc6847->saved_c0) || (c1 != mc6847->saved_c1))
	{
		mc6847->saved_artifacting = artifacting;
		mc6847->saved_c0 = colors[0] = c0;
		mc6847->saved_c1 = colors[15] = c1;

		/* mix the other colors */
		for (i = 1; i <= 14; i++)
		{
			factors = &artifact_colors[((i - 1) ^ artifacting) * 3];

			colors[i] =	(mix_color(factors[0], c0 >> 16, c1 >> 16) << 16)
					|	(mix_color(factors[1], c0 >>  8, c1 >>  8) <<  8)
					|	(mix_color(factors[2], c0 >>  0, c1 >>  0) <<  0);
		}
		for (i = 0; i < 128; i++)
			mc6847->expanded_colors[i] = colors[artifactcorrection[i]];
	}

	/* artifact the line */
	for (i = 0; i < 256; i += 2)
	{
		val =	((line[i - 2] == c1) ? 0x20 : 0x00)
			|	((line[i - 1] == c1) ? 0x10 : 0x00)
			|	((line[i + 0] == c1) ? 0x08 : 0x00)
			|	((line[i + 1] == c1) ? 0x04 : 0x00)
			|	((line[i + 2] == c1) ? 0x02 : 0x00)
			|	((line[i + 3] == c1) ? 0x01 : 0x00);

		new_line[i + 0] = mc6847->expanded_colors[val * 2 + 0];
		new_line[i + 1] = mc6847->expanded_colors[val * 2 + 1];
	}

	/* and copy the results back */
	memcpy(line, new_line, sizeof(*line) * 256);
}



static INPUT_CHANGED( artifacting_changed )
{
	/* TODO: not sure how to do this with a device... */
//  set_dirty();
}



INPUT_PORTS_START( m6847_artifacting )
	PORT_START("artifacting")
	PORT_CONFNAME( 0x03, 0x01, "Artifacting" ) PORT_CHANGED(artifacting_changed, NULL)
	PORT_CONFSETTING(    0x00, DEF_STR( Off ) )
	PORT_CONFSETTING(    0x01, DEF_STR( Standard ) )
	PORT_CONFSETTING(    0x02, DEF_STR( Reverse ) )
INPUT_PORTS_END




UINT32 mc6847_update(device_t *device, bitmap_t *bitmap, const rectangle *cliprect)
{
	mc6847_state *mc6847 = get_safe_token(device);
	int row, i;
	UINT32 rc = 0;

	/* if we have a custom palette, check to see if it has changed */
	if (!mc6847->dirty && mc6847->has_custom_palette)
	{
		for (i = 0; i < 16; i++)
		{
			if (mc6847->palette[i] != mc6847->saved_palette[i])
			{
				mc6847->dirty = TRUE;
				break;
			}
		}
	}

	if (mc6847->dirty)
	{
		/* this frame is dirty; render it */
		mc6847->dirty = FALSE;

		/* copy palette if we have a custom palete */
		if (mc6847->has_custom_palette)
		{
			for (i = 0; i < 16; i++)
				mc6847->saved_palette[i] = mc6847->palette[i];
		}

		/* the video RAM has been dirtied; need to draw */
		for (row = cliprect->min_y; row <= cliprect->max_y; row++)
			render_scanline(device, bitmap, row);
	}
	else
	{
		/* skip this frame */
		rc = UPDATE_HAS_NOT_CHANGED;
	}
	return rc;
}

DEFINE_LEGACY_DEVICE(MC6847, mc6847);
