/*
 * Copyright (c) 2005-2008 Laf-Widget Kirill Grouchnikov. All Rights Reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 *  o Redistributions of source code must retain the above copyright notice,
 *    this list of conditions and the following disclaimer.
 *
 *  o Redistributions in binary form must reproduce the above copyright notice,
 *    this list of conditions and the following disclaimer in the documentation
 *    and/or other materials provided with the distribution.
 *
 *  o Neither the name of Laf-Widget Kirill Grouchnikov nor the names of
 *    its contributors may be used to endorse or promote products derived
 *    from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO,
 * THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR
 * CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
 * EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
 * PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS;
 * OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
 * WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE
 * OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE,
 * EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */
package org.jvnet.lafwidget.combo;

import java.awt.Component;
import java.awt.event.*;
import java.beans.PropertyChangeEvent;
import java.beans.PropertyChangeListener;

import javax.swing.*;
import javax.swing.text.*;

import org.jvnet.lafwidget.*;

/**
 * Adds auto-completion on editable combo boxes.
 * 
 * @author Kirill Grouchnikov
 * @author Thomas Bierhance http://www.orbital-computer.de/JComboBox/
 * @author inostock
 * @author Daniel Kjellin http://www.daik.se/
 */
public class ComboboxAutoCompletionWidget extends LafWidgetAdapter {
	protected JComboBox comboBox;

	/**
	 * Property change handler on <code>enabled</code> property.
	 */
	protected ComboBoxPropertyChangeHandler changeHandler;

	protected ComboBoxModel model;

	protected Component editor;

	// flag to indicate if setSelectedItem has been called
	// subsequent calls to remove/insertString should be ignored
	protected boolean selecting = false;

	protected boolean hidePopupOnFocusLoss;

	protected boolean hitBackspace = false;

	protected boolean hitBackspaceOnSelection;

	protected ActionListener completionActionListener;

	protected PropertyChangeListener completionPropertyListener;

	protected KeyListener editorKeyListener;

	protected FocusListener editorFocusListener;

	protected CompletionPlainDocument completionDocument;

	protected Document originalDocument;

	protected ActionMap oldActionMap;

	/**
	 * Code contributed by Thomas Bierhance from
	 * http://www.orbital-computer.de/JComboBox/
	 */
	protected class CompletionPlainDocument extends PlainDocument {
		protected JComboBox comboBox;

		protected ComboBoxModel model;

		public CompletionPlainDocument(JComboBox combo) {
			super();
			comboBox = combo;
			model = comboBox.getModel();
		}

		@Override
		public void remove(int offs, int len) throws BadLocationException {
			// return immediately when selecting an item
			if (selecting)
				return;

			if (hitBackspace) {
				// user hit backspace => move the selection backwards
				// old item keeps being selected
				if (offs > 0) {
					if (hitBackspaceOnSelection)
						offs--;
				} else {
					// User hit backspace with the cursor positioned on the
					// start => beep
					comboBox.getToolkit().beep(); // when available use:
					// UIManager.getLookAndFeel().provideErrorFeedback(comboBox);
				}
				// highlight when auto-completion is required
				if (LafWidgetUtilities.hasAutoCompletion(comboBox))
					highlightCompletedText(offs);
			} else {
				super.remove(offs, len);
			}
		}

		@Override
		public void insertString(int offs, String str, AttributeSet a)
				throws BadLocationException {
			// return immediately when selecting an item
			if (selecting)
				return;

			// insert the string into the document
			super.insertString(offs, str, a);

			// return immediately when no auto-completion is required
			if (!LafWidgetUtilities.hasAutoCompletion(comboBox))
				return;

			// lookup and select a matching item
			Object item = lookupItem(this.getText(0, getLength()));
			int mIndex = -1;
			for (int i = 0; i < comboBox.getModel().getSize(); i++) {
				if (comboBox.getModel().getElementAt(i).equals(item)) {
					mIndex = i;
					break;
				}
			}
			if (LafWidgetUtilities.hasUseModelOnlyProperty(comboBox)) {
				if (item != null) {
					setSelectedItem(item);
				} else {
					// keep old item selected if there is no match
					item = comboBox.getSelectedItem();
					// imitate no insert (later on offs will be incremented by
					// str.length(): selection won't move forward)
					offs = offs - str.length();
					// provide feedback to the user that his input has been
					// received but can not be accepted
					comboBox.getToolkit().beep();
				}
				// if the model element (item) is not string or doesn't have
				// the matching toString() implementation, try to get the
				// renderer for that item and get the string from there.
				Component renderer = comboBox.getRenderer()
						.getListCellRendererComponent(
								new JList(comboBox.getModel()), item, mIndex,
								false, false);
				if (renderer instanceof JLabel) {
					setText(((JLabel) renderer).getText());
				} else {
					setText(item.toString());
				}
				// select the completed part
				highlightCompletedText(offs + str.length());
			} else {
				if (item != null) {
					setSelectedItem(item);
					// if the model element (item) is not string or doesn't have
					// the matching toString() implementation, try to get the
					// renderer for that item and get the string from there.
					Component renderer = comboBox.getRenderer()
							.getListCellRendererComponent(
									new JList(comboBox.getModel()), item,
									mIndex, false, false);
					if (renderer instanceof JLabel) {
						setText(((JLabel) renderer).getText());
					} else {
						setText(item.toString());
					}
					// select the completed part
					highlightCompletedText(offs + str.length());
				} else {
					offs = offs - str.length();
				}
			}
		}

		private void setText(String text) {
			try {
				// remove all text and insert the completed string
				super.remove(0, getLength());
				super.insertString(0, text, null);
			} catch (BadLocationException e) {
				throw new RuntimeException(e.toString());
			}
		}

		private void highlightCompletedText(int start) {
			if (editor instanceof JTextComponent) {
				JTextComponent textEditor = (JTextComponent) editor;
				// Fix for defect 2 (defect 151 on Substance) by Daniel Kjellin
				textEditor.setCaretPosition(textEditor.getDocument()
						.getLength());
				textEditor.moveCaretPosition(start);
			}
		}

		private void setSelectedItem(Object item) {
			selecting = true;
			model.setSelectedItem(item);
			selecting = false;
		}

		private Object lookupItem(String pattern) {
			AutoCompletionMatcher matcher = LafWidgetUtilities2
					.getAutoCompletionMatcher(comboBox);
			return matcher.getFirstMatching(model, pattern);
			// Object selectedItem = this.model.getSelectedItem();
			// // only search for a different item if the currently selected
			// does
			// // not match
			// if ((selectedItem != null)
			// && this.startsWithIgnoreCase(selectedItem.toString(),
			// pattern)) {
			// return selectedItem;
			// } else {
			// // iterate over all items
			// for (int i = 0, n = this.model.getSize(); i < n; i++) {
			// Object currentItem = this.model.getElementAt(i);
			// // current item starts with the pattern?
			// if ((currentItem != null)
			// && this.startsWithIgnoreCase(
			// currentItem.toString(), pattern)) {
			// return currentItem;
			// }
			// }
			// }
			// // no item starts with the pattern => return null
			// return null;
		}

		// // checks if str1 starts with str2 - ignores case
		// private boolean startsWithIgnoreCase(String str1, String str2) {
		// return str1.toUpperCase().startsWith(str2.toUpperCase());
		// }
	}

	public boolean isSimple() {
		return true;
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.jvnet.lafwidget.LafWidgetAdapter#setComponent(javax.swing.JComponent)
	 */
	@Override
	public void setComponent(JComponent jcomp) {
		super.setComponent(jcomp);
		comboBox = (JComboBox) jcomp;
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.jvnet.lafwidget.LafWidgetAdapter#installUI()
	 */
	@Override
	public void installUI() {
		ComboBoxEditor cbe = comboBox.getEditor();
		Component cbc = cbe.getEditorComponent();
		if (cbc instanceof JTextComponent) {
			installTextEditor((JTextComponent) cbc);
		} else {
			installEditor(cbc);
		}
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.jvnet.lafwidget.LafWidgetAdapter#installListeners()
	 */
	@Override
	public void installListeners() {
		changeHandler = new ComboBoxPropertyChangeHandler();
		comboBox.addPropertyChangeListener(changeHandler);
	}

	protected void installTextEditor(final JTextComponent c) {
		// Code contributed by Thomas Bierhance from
		// http://www.orbital-computer.de/JComboBox/
		if (comboBox.isEditable()) {
			originalDocument = c.getDocument();
			completionDocument = new CompletionPlainDocument(comboBox);
			c.setDocument(completionDocument);
			model = comboBox.getModel();
			completionActionListener = new ActionListener() {
				public void actionPerformed(ActionEvent e) {
					if ((!selecting) && (completionDocument != null))
						completionDocument.highlightCompletedText(0);
				}
			};
			comboBox.addActionListener(completionActionListener);

			completionPropertyListener = new PropertyChangeListener() {
				public void propertyChange(PropertyChangeEvent e) {
					// if (e.getPropertyName().equals("editor"))
					// configureEditor((ComboBoxEditor) e.getNewValue());
					if (e.getPropertyName().equals("model"))
						model = (ComboBoxModel) e.getNewValue();
				}
			};
			comboBox.addPropertyChangeListener(completionPropertyListener);

			editorKeyListener = new KeyAdapter() {
				@Override
				public void keyPressed(KeyEvent e) {
					if (!LafWidgetUtilities.hasAutoCompletion(comboBox))
						return;
					// if (e.isActionKey())
					// return;
					if (comboBox.isDisplayable()
							&& (e.getKeyCode() != KeyEvent.VK_ENTER)
							&& (e.getKeyChar() != KeyEvent.VK_ESCAPE)) {
						comboBox.setPopupVisible(true);
					}
					hitBackspace = false;
					switch (e.getKeyCode()) {
					// determine if the pressed key is backspace (needed by the
					// remove method)
					case KeyEvent.VK_BACK_SPACE:
						hitBackspace = true;
						hitBackspaceOnSelection = ((JTextField) editor)
								.getSelectionStart() != ((JTextField) editor)
								.getSelectionEnd();
						break;
					case KeyEvent.VK_DELETE:
						if (LafWidgetUtilities
								.hasUseModelOnlyProperty(comboBox)) {
							// ignore delete key on model-only combos
							e.consume();
							comboBox.getToolkit().beep();
						} else {
							((JTextField) editor).replaceSelection("");
						}
						break;
					case KeyEvent.VK_ESCAPE:
						// forward to the parent - allows closing dialogs
						// with editable combos having focus.
						comboBox.getParent().dispatchEvent(e);
						break;
					}
				}
			};
			// Bug 5100422 on Java 1.5: Editable JComboBox won't hide popup when
			// tabbing out
			hidePopupOnFocusLoss = System.getProperty("java.version")
					.startsWith("1.5");
			// Highlight whole text when gaining focus
			editorFocusListener = new FocusAdapter() {
				@Override
				public void focusGained(FocusEvent e) {
					if (completionDocument != null)
						completionDocument.highlightCompletedText(0);
				}

				@Override
				public void focusLost(FocusEvent e) {
					// Workaround for Bug 5100422 - Hide Popup on focus loss
					if (hidePopupOnFocusLoss && (comboBox != null))
						comboBox.setPopupVisible(false);
				}
			};
			// configureEditor(comboBox.getEditor());
			installEditor(c);
			// Handle initially selected object
			Object selected = comboBox.getSelectedItem();
			if (completionDocument != null) {
				if (selected != null) {
					// if the model element (item) is not string or doesn't have
					// the matching toString() implementation, try to get the
					// renderer for that item and get the string from there.
					Component renderer = comboBox.getRenderer()
							.getListCellRendererComponent(
									new JList(comboBox.getModel()), selected,
									comboBox.getSelectedIndex(), false, false);
					if (renderer instanceof JLabel) {
						completionDocument.setText(((JLabel) renderer)
								.getText());
					} else {
						completionDocument.setText(selected.toString());
					}
				}
				completionDocument.highlightCompletedText(0);
			}
		}
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.jvnet.lafwidget.LafWidgetAdapter#uninstallListeners()
	 */
	@Override
	public void uninstallListeners() {
		comboBox.removePropertyChangeListener(changeHandler);
		changeHandler = null;

		if (comboBox.isEditable()) {
			ComboBoxEditor cbe = comboBox.getEditor();
			Component cbc = cbe.getEditorComponent();
			// uninstallTextEditor(null);
			if (cbc instanceof JTextComponent)
				uninstallTextEditor((JTextComponent) cbc);
			else
				uninstallTextEditor(null);
		}

		super.uninstallListeners();
	}

	protected void uninstallTextEditor(final JTextComponent e) {
		if ((e != null) && (originalDocument != null)) {
			if ((completionDocument != null)
					&& (originalDocument instanceof PlainDocument)) {
				try {
					// try updating the contents of the original document
					String currText = completionDocument.getText(0,
							completionDocument.getLength());
					originalDocument.remove(0, originalDocument.getLength());
					originalDocument.insertString(0, currText, null);
				} catch (BadLocationException ble) {
					// fail silently
				}
			}
			// restore the original document
			e.setDocument(originalDocument);
		}
		completionDocument = null;
		comboBox.removeActionListener(completionActionListener);
		completionActionListener = null;
		comboBox.removePropertyChangeListener(completionPropertyListener);
		completionPropertyListener = null;
		if (e == null)
			return;
		if (editorKeyListener != null) {
			e.removeKeyListener(editorKeyListener);
			editorKeyListener = null;
		}
		if (editorFocusListener != null) {
			e.removeFocusListener(editorFocusListener);
			editorFocusListener = null;
		}
		if (oldActionMap != null) {
			comboBox.setActionMap(oldActionMap);
			oldActionMap = null;
		}

	}

	protected void installEditor(final Component c) {
		if ((c == null) || (editor == c))
			return;

		Component last = editor;
		if (last != null) {
			last.removeKeyListener(editorKeyListener);
			last.removeFocusListener(editorFocusListener);
		}

		editor = c;
		editor.addKeyListener(editorKeyListener);
		editor.addFocusListener(editorFocusListener);

		if (oldActionMap == null) {
			// due to the implementation in BasicComboBoxUI (the
			// same action map for all combos) we need to
			// create a new action map
			oldActionMap = comboBox.getActionMap();
			ActionMap newActionMap = new ActionMap();
			Object[] keys = oldActionMap.allKeys();
			for (int i = 0; i < keys.length; i++) {
				if ("enterPressed".equals(keys[i]))
					continue;
				newActionMap.put(keys[i], oldActionMap.get(keys[i]));
			}
			comboBox.setActionMap(newActionMap);
		}
	}

	public class ComboBoxPropertyChangeHandler implements
			PropertyChangeListener {
		public void propertyChange(PropertyChangeEvent e) {
			String propertyName = e.getPropertyName();

			if (propertyName.equals("editable")) {
				boolean oldValue = ((Boolean) e.getOldValue()).booleanValue();
				boolean newValue = ((Boolean) e.getNewValue()).booleanValue();
				ComboBoxEditor cbe = comboBox.getEditor();
				Component cbc = cbe.getEditorComponent();
				if (!oldValue && newValue
						&& LafWidgetUtilities.hasAutoCompletion(comboBox)) {
					if (cbc instanceof JTextComponent) {
						installTextEditor((JTextComponent) cbc);
					} else {
						installEditor(cbc);
					}
				} else if (oldValue && !newValue) {
					if (cbc instanceof JTextComponent)
						uninstallTextEditor((JTextComponent) cbc);
					else
						uninstallTextEditor(null);
				}
			}

			// fix for issue 179 on Substance - allowing no auto-completion
			// mode on editable comboboxes.
			if (propertyName.equals(LafWidget.COMBO_BOX_NO_AUTOCOMPLETION)
					|| propertyName.equals("JComboBox.isTableCellEditor")) {
				ComboBoxEditor cbe = comboBox.getEditor();
				Component cbc = cbe.getEditorComponent();
				if (LafWidgetUtilities.hasAutoCompletion(comboBox)) {
					if (cbc instanceof JTextComponent) {
						installTextEditor((JTextComponent) cbc);
					} else {
						installEditor(cbc);
					}
				} else {
					if (cbc instanceof JTextComponent)
						uninstallTextEditor((JTextComponent) cbc);
					else
						uninstallTextEditor(null);
				}
			}

			// fix for defect 131 in 2.2_01
			if (propertyName.equals("editor")) {
				ComboBoxEditor oldValue = (ComboBoxEditor) e.getOldValue();
				ComboBoxEditor newValue = (ComboBoxEditor) e.getNewValue();
				if ((newValue != null) && (newValue != oldValue)
						&& LafWidgetUtilities.hasAutoCompletion(comboBox)) {
					Component old = (oldValue != null) ? oldValue
							.getEditorComponent() : null;
					if (old instanceof JTextComponent) {
						uninstallTextEditor((JTextComponent) old);
					}
					Component pending = newValue.getEditorComponent();
					if (pending instanceof JTextComponent) {
						installTextEditor((JTextComponent) pending);
					} else {
						installEditor(pending);
					}
					SwingUtilities.invokeLater(new Runnable() {
						public void run() {
							comboBox.doLayout();
						}
					});
				}
			}

			// fix for defect 6 - changing model on editable combo doesn't
			// track changes to the model
			if (propertyName.equals("model")) {
				if (LafWidgetUtilities.hasAutoCompletion(comboBox)) {
					uninstallTextEditor(null);
					ComboBoxEditor cbe = comboBox.getEditor();
					Component cbc = cbe.getEditorComponent();
					if (cbc instanceof JTextComponent) {
						installTextEditor((JTextComponent) cbc);
					} else {
						installEditor(cbc);
					}
				}
			}

			// Do not call super - fix for bug 63
		}
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.jvnet.lafwidget.LafWidget#requiresCustomLafSupport()
	 */
	public boolean requiresCustomLafSupport() {
		return false;
	}
}
