############################################################################
##
## Copyright (c) 2000, 2001, 2002, 2003, 2004, 2005 BalaBit IT Ltd, Budapest, Hungary
##
## This program is free software; you can redistribute it and/or modify
## it under the terms of the GNU General Public License as published by
## the Free Software Foundation; either version 2 of the License, or
## (at your option) any later version.
##
## This program is distributed in the hope that it will be useful,
## but WITHOUT ANY WARRANTY; without even the implied warranty of
## MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
## GNU General Public License for more details.
##
## You should have received a copy of the GNU General Public License
## along with this program; if not, write to the Free Software
## Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
##
##
## $Id: Router.py,v 1.27 2004/02/04 11:54:44 sasa Exp $
##
## Author  : Bazsi
## Auditor :
## Last audited version:
## Notes:
##
############################################################################

"""Module defining the Router interface for directing connections to the appropriate servers.

This module defines the 'AbstractRouter' interface and derived classes,
which are used when determining the server address to connect to.
"""

from Zorp import *
from Zorp import setupDirectedRouter
from SockAddr import SockAddrInet

class AbstractRouter:
	"""Abstract class encapsulating the Router interface.
	
	This abstract class encapsulates the Router interface, which is used
	to determine the destination address of a given service. Different
	implementations of this interface perform Transparent routing
	(directing the client to its original destination), and Directed
	routing (directing the client to a given destination).
	
	The result of this class can be overridden by the proxy using
	the 'setServerAddress' method.

		  forge_addr -- [BOOLEAN] set to true if the client's source address is to be forged on the server side

		  forge_port -- [FORGE_PORT] one of the values
				Z_PORT_ANY,Z_PORT_GROUP,Z_PORT_EXACT or an
				integer between [1..65535] to specify port
				number


	"""
	def __init__(self, forge_addr, forge_port):
		self.forge_addr = forge_addr
		self.forge_port = forge_port

	def routeConnection(self, session):
		"""Function called to perform connection routing.
		
		This function is called to determine the destination address
		of this session, and place it in session.server_address
		
		Arguments
		
		  self -- this instance
		  
		  session -- session we belong to
		"""
		session.server_local_loose = FALSE
		if self.forge_port == Z_PORT_ANY:
			local_port = 0
		elif self.forge_port == Z_PORT_GROUP:
			local_port = session.client_address.port
			session.server_local_loose = TRUE
		elif self.forge_port == Z_PORT_EXACT:
			local_port = session.client_address.port
		elif self.forge_port >= 1 and self.forge_port <= 65535:
			local_port = self.forge_port
		else:
			raise ValueError, "Invalid forge_port value (%d)" % self.forge_port
		
		session.server_tos = session.client_tos

		if self.forge_addr or session.service.snat or session.service.snat_policy:
			local_addr = SockAddrInet(session.client_address.ip_s, local_port)
		else:
			if local_port != 0:
				local_addr = SockAddrInet('0.0.0.0', local_port)
			else:
				local_addr = None
		session.server_local = local_addr

	def setupFastpath(self, proxy):
		"""Function called to setup fast path handlers.

		This function is called to provide the C-level fast path
		handlers to the proxy, which are called when subsequent
		connections are added to the proxy (currently only used
		for UDP proxies)
		"""

		pass
		
	
class TransparentRouter(AbstractRouter):
	"""Class implementing transparent routing.
	
	This class implements transparent routing, which means it directs
	the proxy to connect to the original destination of the client.
	
	Attributes
	
	  forced_port -- set the destination port to this value if non-zero

	  forge_addr -- set to true if the client's source address is to be forged on the server side
	"""
	def __init__(self, forced_port=0, forge_addr=FALSE, overrideable=FALSE, forge_port=Z_PORT_ANY):
		"""Constructor to initialize a TransparentRouter instance.
		
		This constructor initializes a TransparentRouter instance
		by storing arguments as attributes.
		
		Arguments
		
		  self -- this instance
		  
		  forced_port -- [INTEGER] set the destination port to this value if non-zero

		  forge_addr -- [BOOLEAN] set to true if the client's source address is to be forged on the server side

		  forge_port -- [FORGE_PORT] one of the values
				Z_PORT_ANY,Z_PORT_GROUP,Z_PORT_EXACT or an
				integer between [1..65535] to specify port
				number

		  overrideable -- [BOOLEAN] True if proxy may override the
		                  setted value. Needed with SideStackChainer
		"""
		AbstractRouter.__init__(self, forge_addr, forge_port)
		self.forced_port = forced_port
		self.overrideable = overrideable

	def setupFastpath(self, proxy):
	        setupTransparentRouter(proxy, self.forced_port, self.forge_addr)

	def routeConnection(self, session):
		"""Overridden function to perform routing.
		
		This function sets 'session.server_address' to the transparent
		destination as stored in 'session.client_local'.
		
		Arguments
		
		  self -- this instance
		  
		  session -- session we belong to
		"""
		AbstractRouter.routeConnection(self, session)

		if self.forced_port:
			addr = SockAddrInet(session.client_local.ip_s, self.forced_port)
		else:
			addr = SockAddrInet(session.client_local.ip_s, session.client_local.port)
		session.server_address_inband = self.overrideable
		session.setServer(addr)

class DirectedRouter(AbstractRouter):
	"""Class implementing directed routing.
	
	This class imlements directed routing, which means it connects
	each session to a fixed address.
	
	Attributes
	
	  dest_addr -- fixed destination address to connect to
	
	"""
	def __init__(self, dest_addr, forge_addr=FALSE, overrideable=FALSE, forge_port=Z_PORT_ANY):
		"""Constructor to initialize a DirectedRouter.
		
		This constructor initializes a DirectedRouter instance by
		storing arguments as attributes.
		
		Arguments
		
		  self -- this instance
		  
		  dest_addr -- [SOCKADDR] fixed destination address to connect to

		  forge_addr -- [BOOLEAN] set to true if the client's source address is to be forged on the server side

		  overrideable -- [BOOLEAN] True if proxy may override the
		                  setted value. Needed with SideStackChainer

		  forge_port  -- [FORGE_PORT] one of the values
				 Z_PORT_ANY,Z_PORT_GROUP,Z_PORT_EXACT or an
				 integer between [1..65535] to specify port
				 number
		"""
		AbstractRouter.__init__(self, forge_addr, forge_port)
		self.dest_addr = dest_addr
		self.overrideable = overrideable

	def setupFastpath(self, proxy):
		setupDirectedRouter(proxy, self.dest_addr, self.forge_addr)
		
	def routeConnection(self, session):
		"""Overridden function to perform routing.
		
		This function simply sets 'session.server_address' to
		'self.dest_addr'.
		
		Arguments
		
		  self -- this instance
		  
		  session -- session we belong to
		"""
		AbstractRouter.routeConnection(self, session)
	        session.setServer(self.dest_addr)
		session.server_address_inband = self.overrideable
		
class InbandRouter(AbstractRouter):
	"""Class implementing Inband routing.
	
	This class implements inband routing, which basically means that the
	destination address is not known at this time, it should be
	determined by the protocol logic. It can be used for example for
	non-transparent HTTP proxies.
	"""
	def __init__(self, forge_addr=FALSE, forge_port=Z_PORT_ANY):
		"""Constructor to initialize a InbandRouter.
		
		This constructor initializes a InbandRouter instance by
		storing arguments as attributes.
		
		Arguments
		
		  self -- this instance
		  
		  forge_addr -- [BOOLEAN] set to true if the client's source
				address is to be forged on the server side

		  forge_port  -- [FORGE_PORT] one of the values
				 Z_PORT_ANY,Z_PORT_GROUP,Z_PORT_EXACT or an
				 integer between [1..65535] to specify port
				 number

		"""
		AbstractRouter.__init__(self, forge_addr, forge_port)

	def routeConnection(self, session):
		"""Overridden function to perform routing.
		
		This function does nothing, it simply let's the protocol
		logic to choose its destination server, if it doesn't an
		exception will be raised at connection time.
		
		Arguments
		
		  self -- this instance
		  
		  session -- session we belong to
		"""
		AbstractRouter.routeConnection(self, session)
		session.server_address_inband = 1
