/*
 * constants.i -
 *	Useful constants for physical computations with Yorick.
 *
 * Copyright (c) 2000-2003, Eric THIEBAUT.
 * 
 * This program is free software; you can redistribute it and/or  modify it
 * under the terms of the GNU General Public License  as  published  by the
 * Free Software Foundation; either version 2 of the License,  or  (at your
 * option) any later version.
 *
 * This program is distributed in the hope  that  it  will  be  useful, but
 * WITHOUT  ANY   WARRANTY;   without   even   the   implied   warranty  of
 * MERCHANTABILITY or  FITNESS  FOR  A  PARTICULAR  PURPOSE.   See  the GNU
 * General Public License for more details (to receive a  copy  of  the GNU
 * General Public License, write to the Free Software Foundation, Inc., 675
 * Mass Ave, Cambridge, MA 02139, USA).
 *
 * History:
 *	$Id: constants.i,v 1.2 2008/02/15 18:55:27 frigaut Exp $
 *	$Log: constants.i,v $
 *	Revision 1.2  2008/02/15 18:55:27  frigaut
 *	fixed UTF-8 encoding problems (crash yorick-doc on amd64)
 *	
 *	Revision 1.1.1.1  2007/12/11 23:55:12  frigaut
 *	Initial Import - yorick-yutils
 *	
 *	Revision 1.4  2003/11/10 12:21:13  eric
 *	 - fixed typo;
 *
 *	Revision 1.3  2003/11/10 12:18:58  eric
 *	 - change name: AstrophysicalUnit -> AstronomicalUnit;
 *	 - change units: VacuumPermeability;
 *	 - add complete documentation in pseudo-symbol "constants";
 *	 - fixed units for EarthRotationalPeriod;
 *	 - avoid inconsistencies (and errors) by explicitely compute constants
 *	   which depend on others;
 *
 *	Revision 1.2  2001/10/09 07:05:45  eric
 *	 - fixed value of VacuumPermeability
 *
 *	Revision 1.1  2001/03/23 17:02:53  eric
 *	Initial revision
 */

local constants;
/* DOCUMENT Physical constants defined in "constants.i":

     Pi ........................ number pi, semi-perimeter of unit circle
     E ......................... number e
     Deg2Rad ................... radians per degree
     Rad2Deg ................... degrees per radian
     ArcSec .................... radians per arcsecond
     EulerConstant ............. Euler's constant

     ElementaryCharge .......... charge of a protron [C]
     GravitationalConstant ..... gravitational constant [m^3/kg/s^2]
     FineStructureConstant ..... fine-structure constant
     LightSpeed ................ speed of light in vacuum [m/s]
     VacuumPermittivity ........ permittivity of vacuum [F/m]
     VacuumPermeability ........ permeability of vacuum [N m^2/C^2]

     PlanckConstant ............ Planck's constant [J s]
     DiracConstant ............. Dirac's constant [J s]
     BoltzmannConstant ......... Boltzmann's Constant [J/K]
     StefanConstant ............ Stefan-Boltzmann's constant [W/m^2/K^4]
     WienConstant .............. Wien's constant [m K]

     BohrMagneton .............. Bohr magneton [A m^2]
     BohrRadius ................ Bohr radius [m]
     RydbergConstant ........... Rydberg's constant [eV]

     PerfectGazConstant ........ perfect gaz constant [J/mol]
     AvogadroConstant .......... Avogadro's constant [1/mol]
     IcePoint .................. absolute temperature of ice [K]

     ElectronMass .............. electron mass [kg]
     ProtonMass ................ protron mass [kg]
     NeutronMass ............... neutron mass [kg]
     ElementaryMassUnit ........ elementary mass unit [kg]

     LightYear ................. light year [m]
     ParSec .................... astronomical unit per arcsecond [m]
     AstronomicalUnit .......... semi-major axis of Earth orbit [m]

     SunLuminosity ............. luminosity of the Sun [W]
     SunMass ................... mass of the Sun [kg]
     SunRadius ................. radius of the Sun [m]
     SunTemperature ............ effective temperature of the Sun [K]
     SunRotationalPeriod ....... rotational period of the Sun [days]

     EarthMass ................. mass of the Earth [kg]
     EarthRadius ............... radius of the Earth [m]
     EarthRotationalPeriod ..... rotational period of the Earth [hours]
     EarthOrbitalPeriod ........ orbital period of the Earth [days]
 */

/*---------------------------------------------------------------------------*/

#if 0
// The following lines were automatically generated by the script:
grep '\.\.' constants.i | sed 's,^ *\([^ ]*\)[ .]*\(.*\),local \1;\n/* DOCUMENT \1 - \2\n   SEE ALSO constants. */,';
#endif

local Pi;
/* DOCUMENT Pi - number pi, semi-perimeter of unit circle
   SEE ALSO constants. */
local E;
/* DOCUMENT E - number e
   SEE ALSO constants. */
local Deg2Rad;
/* DOCUMENT Deg2Rad - radians per degree
   SEE ALSO constants. */
local Rad2Deg;
/* DOCUMENT Rad2Deg - degrees per radian
   SEE ALSO constants. */
local ArcSec;
/* DOCUMENT ArcSec - radians per arcsecond
   SEE ALSO constants. */
local EulerConstant;
/* DOCUMENT EulerConstant - Euler's constant
   SEE ALSO constants. */
local ElementaryCharge;
/* DOCUMENT ElementaryCharge - charge of a protron [C]
   SEE ALSO constants. */
local GravitationalConstant;
/* DOCUMENT GravitationalConstant - gravitational constant [m^3/kg/s^2]
   SEE ALSO constants. */
local FineStructureConstant;
/* DOCUMENT FineStructureConstant - fine-structure constant
   SEE ALSO constants. */
local LightSpeed;
/* DOCUMENT LightSpeed - speed of light in vacuum [m/s]
   SEE ALSO constants. */
local VacuumPermittivity;
/* DOCUMENT VacuumPermittivity - permittivity of vacuum [F/m]
   SEE ALSO constants. */
local VacuumPermeability;
/* DOCUMENT VacuumPermeability - permeability of vacuum [N m^2/C^2]
   SEE ALSO constants. */
local PlanckConstant;
/* DOCUMENT PlanckConstant - Planck's constant [J s]
   SEE ALSO constants. */
local DiracConstant;
/* DOCUMENT DiracConstant - Dirac's constant [J s]
   SEE ALSO constants. */
local BoltzmannConstant;
/* DOCUMENT BoltzmannConstant - Boltzmann's Constant [J/K]
   SEE ALSO constants. */
local StefanConstant;
/* DOCUMENT StefanConstant - Stefan-Boltzmann's constant [W/m^2/K^4]
   SEE ALSO constants. */
local BohrMagneton;
/* DOCUMENT BohrMagneton - Bohr magneton [A m^2]
   SEE ALSO constants. */
local BohrRadius;
/* DOCUMENT BohrRadius - Bohr radius [m]
   SEE ALSO constants. */
local RydbergConstant;
/* DOCUMENT RydbergConstant - Rydberg's constant [eV]
   SEE ALSO constants. */
local PerfectGazConstant;
/* DOCUMENT PerfectGazConstant - perfect gaz constant [J/mol]
   SEE ALSO constants. */
local AvogadroConstant;
/* DOCUMENT AvogadroConstant - Avogadro's constant [1/mol]
   SEE ALSO constants. */
local IcePoint;
/* DOCUMENT IcePoint - absolute temperature of ice [K]
   SEE ALSO constants. */
local ElectronMass;
/* DOCUMENT ElectronMass - electron mass [kg]
   SEE ALSO constants. */
local ProtonMass;
/* DOCUMENT ProtonMass - protron mass [kg]
   SEE ALSO constants. */
local NeutronMass;
/* DOCUMENT NeutronMass - neutron mass [kg]
   SEE ALSO constants. */
local ElementaryMassUnit;
/* DOCUMENT ElementaryMassUnit - elementary mass unit [kg]
   SEE ALSO constants. */
local LightYear;
/* DOCUMENT LightYear - light year [m]
   SEE ALSO constants. */
local ParSec;
/* DOCUMENT ParSec - astronomical unit per arcsecond [m]
   SEE ALSO constants. */
local AstronomicalUnit;
/* DOCUMENT AstronomicalUnit - semi-major axis of Earth orbit [m]
   SEE ALSO constants. */
local SunLuminosity;
/* DOCUMENT SunLuminosity - luminosity of the Sun [W]
   SEE ALSO constants. */
local SunMass;
/* DOCUMENT SunMass - mass of the Sun [kg]
   SEE ALSO constants. */
local SunRadius;
/* DOCUMENT SunRadius - radius of the Sun [m]
   SEE ALSO constants. */
local SunTemperature;
/* DOCUMENT SunTemperature - effective temperature of the Sun [K]
   SEE ALSO constants. */
local SunRotationalPeriod;
/* DOCUMENT SunRotationalPeriod - rotational period of the Sun [days]
   SEE ALSO constants. */
local EarthMass;
/* DOCUMENT EarthMass - mass of the Earth [kg]
   SEE ALSO constants. */
local EarthRadius;
/* DOCUMENT EarthRadius - radius of the Earth [m]
   SEE ALSO constants. */
local EarthRotationalPeriod;
/* DOCUMENT EarthRotationalPeriod - rotational period of the Earth [hours]
   SEE ALSO constants. */
local EarthOrbitalPeriod;
/* DOCUMENT EarthOrbitalPeriod - orbital period of the Earth [days]
   SEE ALSO constants. */

/*---------------------------------------------------------------------------*/
/* MATHEMATICS */

/* 35 significant digits is sufficient for 16-byte IEEE floating point */
Pi = 3.1415926535897932384626433832795029;
E  = 2.7182818284590452353602874713526625;
Deg2Rad = Pi/180.0;
Rad2Deg = 180.0/Pi;
ArcSec = Deg2Rad/3600.0;

/* lim_{n->+infinity} sum_{k=1}{n} 1/k - log(n) */
EulerConstant = 0.5772156649;

/*---------------------------------------------------------------------------*/
/* PHYSICS */

LightSpeed            = 2.99792458e+8; // [m/s]
PlanckConstant        = 6.6260755e-34;
DiracConstant         = PlanckConstant/Pi;
BoltzmannConstant     = 1.380658e-23;
GravitationalConstant = 6.67259e-11;
WienConstant          = 2.8978e-3;
// StefanConstant     = 5.67032e-8;
StefanConstant        = 2.0*Pi^5*BoltzmannConstant^4/(15.0*PlanckConstant^3
                                                      *LightSpeed^2);
PerfectGazConstant    = 8.31441;
AvogadroConstant      = 6.0221367e+23;
IcePoint              = 273.15; // kelvin temperature at 0 celcius degree
WaterTriplePoint      = 273.16;

VacuumPermittivity    = 8.854187e-12;
// VacuumPermeability = 4e-7*Pi;
VacuumPermeability    = 1.0/(4.0*Pi*VacuumPermittivity);

ElectronMass          = 9.1093897e-31;
ProtonMass            = 1.6726231e-27;
NeutronMass           = 1.6749540e-27;
ElementaryMassUnit    = 1.6605656e-27;
UnifiedAtomicMassUnit = 1.6605402e-27; // +/- 0.0000010e-27
ElementaryCharge      = 1.60217733e-19; // +/- 0.00000049e-19 C
FineStructureConstant = ElementaryCharge^2/(2.0*PlanckConstant*LightSpeed
                                            *VacuumPermittivity);
BohrMagneton          = ElementaryCharge*PlanckConstant/(4.0*Pi*ElectronMass);
BohrRadius            = 5.2918e-11; // [m]
RydbergConstant       = 13.595; // [e V]

#if 0
ElectronComptonWavelength = PlanckConstant/(ElectronMass*LightSpeed);
ProtonComptonWavelength = PlanckConstant/(ProtonMass*LightSpeed);
HydrogenReducedMass = 1.0/(1.0/ElectronMass + 1.0/ProtonMass);
HydrogenReducedMass = 9.1045755e-31; // [kg]
#endif

/*---------------------------------------------------------------------------*/
/* ASTROPHYSICS */

SunLuminosity         = 3.826e+26;  // [W]
SunMass               = 1.989e+30;  // [kg]
SunRadius             = 6.9599e+8;  // [m]
SunTemperature        = 5.77025e+3; // [K]
SunRotationalPeriod   = 25.38;      // [hours]

EarthMass             = 5.976e+24;
EarthRadius           = 6.378e+6;
EarthRotationalPeriod = 23.96;        // [hours]
EarthOrbitalPeriod    = 365.24219879; // [days]

AstronomicalUnit      = 1.4959787066e+11;
LightYear             = 24.0*3600.0*EarthOrbitalPeriod*LightSpeed;
ParSec                = AstronomicalUnit/ArcSec;

/*---------------------------------------------------------------------------*/
