// file      : morphing/anonymous/processor.cxx
// author    : Boris Kolpackov <boris@codesynthesis.com>
// copyright : Copyright (c) 2005-2007 Code Synthesis Tools CC
// license   : GNU GPL v2 + exceptions; see accompanying LICENSE file

#include <morphing/anonymous/processor.hxx>

#include <iostream>
#include <sstream>

#include <backend-elements/indentation/clip.hxx>

#include <xsd-frontend/semantic-graph.hxx>
#include <xsd-frontend/traversal.hxx>

#include <backend-elements/regex.hxx>

#include "../../usage.hxx"

namespace Morphing
{
  using std::endl;
  using std::wcerr;

  using namespace Cult;

  namespace SemanticGraph = XSDFrontend::SemanticGraph;
  namespace Traversal = XSDFrontend::Traversal;

  typedef WideString String;


  namespace Anonymous
  {
    namespace CLI
    {
      extern Key preserve_anonymous    = "preserve-anonymous";
      extern Key anonymous_regex       = "anonymous-regex";
      extern Key anonymous_regex_trace = "anonymous-regex-trace";
    }
  }

  Void Anonymous::Processor::
  usage ()
  {
    std::wostream& e (wcerr);
    ::CLI::Indent::Clip< ::CLI::OptionsUsage, WideChar> clip (e);

    e << "--preserve-anonymous" << endl
      << " Preserve anonymous types. By default anonymous\n"
      << " types are automatically named with names derived\n"
      << " from the enclosing elements/attributes."
      << endl;

    e << "--anonymous-regex <regex>" << endl
      << " Add the provided regular expression to the list of\n"
      << " regular expressions used to derive names for\n"
      << " anonymous types from the names of the enclosing\n"
      << " attributes/elements. The first successful\n"
      << " substitution is used. The last provided expression\n"
      << " is considered first. Expression is a perl-like\n"
      << " regex in the form /pattern/replacement/ ."
      << endl;

    e << "--anonymous-regex-trace" << endl
      << " Trace the process of applying regular expressions\n"
      << " specified with the --anonymous-regex option."
      << endl;
  }

  Anonymous::CLI::OptionsSpec Anonymous::Processor::
  options_spec ()
  {
    return CLI::OptionsSpec ();
  }

  namespace Anonymous
  {
    namespace
    {
      class Context
      {
      public:
        typedef Cult::Containers::Vector<String> NameMapping;

        Context (SemanticGraph::Schema& schema_,
                 SemanticGraph::Path const& file,
                 CLI::Options const& ops)
            : schema_path_ (file),
              schema (schema_),
              schema_path (schema_path_),
              options_ (ops),
              name_mapping_ (),
              failed_ (false),
              ns_ (0),
              ns (ns_),
              options (options_),
              name_mapping (name_mapping_),
              failed (failed_)
        {

          for (Containers::Vector<NarrowString>::ConstIterator
                 i (options.value<CLI::anonymous_regex> ().begin ()),
                 e (options.value<CLI::anonymous_regex> ().end ());
               i != e; ++i)
          {
            name_mapping.push_back (*i);
          }

        }

      protected:
        Context (Context& c)
            : schema (c.schema),
              schema_path (c.schema_path),
              ns (c.ns),
              options (c.options),
              name_mapping (c.name_mapping),
              failed (c.failed)
        {
        }

      public:
        String
        xpath (SemanticGraph::Nameable& n)
        {
          if (dynamic_cast<SemanticGraph::Namespace*> (&n) != 0)
            return L"<namespace-level>"; // There is a bug if you see this.

          assert (n.named ());

          SemanticGraph::Scope& scope (n.scope ());

          if (dynamic_cast<SemanticGraph::Namespace*> (&scope) != 0)
            return n.name ();

          return xpath (scope) + L"/" + n.name ();
        }

      private:
        SemanticGraph::Path const schema_path_;
        CLI::Options const options_;
        SemanticGraph::Namespace* ns_;
        NameMapping name_mapping_;
        Boolean failed_;

      public:
        SemanticGraph::Schema& schema;
        SemanticGraph::Path const& schema_path;
        CLI::Options const& options;
        SemanticGraph::Namespace*& ns;
        NameMapping& name_mapping;
        Boolean& failed;
      };


      // Go into implied/included/imported schemas while making sure
      // we don't process the same stuff more than once.
      //
      struct Uses: Traversal::Uses
      {
        virtual Void
        traverse (Type& u)
        {
          SemanticGraph::Schema& s (u.schema ());

          if (!s.context ().count ("morphing-anonymous-processor-seen"))
          {
            s.context ().set ("morphing-anonymous-processor-seen", true);
            Traversal::Uses::traverse (u);
          }
        }
      };

      // Keep track which namespace we are in.
      //
      struct Namespace: Traversal::Namespace
      {
        Namespace (SemanticGraph::Namespace*& ns)
            : ns_ (ns)
        {
        }

        Void
        pre (SemanticGraph::Namespace& ns)
        {
          ns_ = &ns;
        }

        Void
        post (SemanticGraph::Namespace&)
        {
          ns_ = 0;
        }

      private:
        SemanticGraph::Namespace*& ns_;
      };


      //
      //
      struct Member: Traversal::Element,
                     Traversal::Attribute,
                     protected virtual Context
      {
        Member (Context& c)
            : Context (c)
        {
        }

        struct UnstableConflict
        {
          UnstableConflict (SemanticGraph::Type& type)
              : type_ (type)
          {
          }

          SemanticGraph::Type&
          type () const
          {
            return type_;
          }

        private:
          SemanticGraph::Type& type_;
        };


        virtual Void
        traverse (SemanticGraph::Element& e)
        {
          SemanticGraph::Type& t (e.type ());

          //@@ This IDREF stuff is really ugly!
          //
          if (!t.named ()
              && dynamic_cast<SemanticGraph::Fundamental::IdRef*> (&t) == 0
              && dynamic_cast<SemanticGraph::Fundamental::IdRefs*> (&t) == 0)
          {
            try
            {
              traverse_ (e);
            }
            catch (UnstableConflict const& ex)
            {
              SemanticGraph::Type& t (ex.type ());

              wcerr << e.file () << ":" << e.line () << ":" << e.column ()
                    << ": error: element name '" << xpath (e) << "' "
                    << "creates an unstable conflict when used as a type name"
                    << endl;

              wcerr << t.file () << ":" << t.line () << ":" << t.column ()
                    << ": info: conflicting type is defined here" << endl;

              wcerr << e.file () << ":" << e.line () << ":" << e.column ()
                    << ": info: "
                    << "use --anonymous-regex to resolve this conflict"
                    << endl;

              wcerr << e.file () << ":" << e.line () << ":" << e.column ()
                    << ": info: "
                    << "and don't forget to pass the same option when "
                    << "translating '" << e.file ().leaf () << "' and all "
                    << "the schemas that refer to it" << endl;

              failed = true;
            }
          }
        }

        virtual Void
        traverse (SemanticGraph::Attribute& a)
        {
          SemanticGraph::Type& t (a.type ());

          //@@ This IDREF stuff us really ugly!
          //
          if (!t.named ()
              && dynamic_cast<SemanticGraph::Fundamental::IdRef*> (&t) == 0
              && dynamic_cast<SemanticGraph::Fundamental::IdRefs*> (&t) == 0)
          {
            try
            {
              traverse_ (a);
            }
            catch (UnstableConflict const& ex)
            {
              SemanticGraph::Type& t (ex.type ());

              wcerr << a.file () << ":" << a.line () << ":" << a.column ()
                    << ": error: attribute name '" << xpath (a) << "' "
                    << "creates an unstable conflict when used as a type name"
                    << endl;

              wcerr << t.file () << ":" << t.line () << ":" << t.column ()
                    << ": info: conflicting type is defined here" << endl;

              wcerr << a.file () << ":" << a.line () << ":" << a.column ()
                    << ": info: "
                    << "use --anonymous-regex to resolve this conflict"
                    << endl;

              wcerr << a.file () << ":" << a.line () << ":" << a.column ()
                    << ": info: "
                    << "and don't forget to pass the same option when "
                    << "translating '" << a.file ().leaf () << "' and all "
                    << "the schemas that refer to it" << endl;

              failed = true;
            }
          }
        }

        Void
        traverse_ (SemanticGraph::Member& m)
        {
          using SemanticGraph::Type;

          Type& t (m.type ());

          // Normally this will be the member which also "defined" the type.
          // However, in some cases of cyclic schema inclusion, this does
          // not happen. As a result we need an extra check that will make
          // sure we create the Names edge in the same Schema node as the
          // one which contains the member which initially defined this
          // type. See the cyclic-inclusion test for an example.
          //

          // Find the first member that this type classifies.
          //
          for (Type::ClassifiesIterator i (t.classifies_begin ());
               i != t.classifies_end (); ++i)
          {
            SemanticGraph::Instance& inst (i->instance ());

            if (inst.is_a<SemanticGraph::Member> ())
            {
              // If it is the same member as the one we are traversing,
              // then continue.
              //
              if (&inst == &m)
                break;
              else
                return;
            }
          }

          //
          //
          String name (m.name ());

          SemanticGraph::Scope& s = m.scope ();

          // Run name through custom mapping if any.
          //
          SemanticGraph::Path file (path (m));
          String pat (file == schema_path ? "" : file.string ());

          pat += L' ' + ns->name () + L' ' + xpath (m);

          if (options.value<CLI::anonymous_regex_trace> ())
            wcerr << "pat: '" << pat << "'" << endl;

          for (NameMapping::ReverseIterator e (name_mapping.rbegin ());
               e != name_mapping.rend (); ++e)
          {
            String tmp (BackendElements::Regex::perl_s (pat, *e));

            if (options.value<CLI::anonymous_regex_trace> ())
              wcerr << "try: '" << *e << "' : '" << tmp << "' : ";

            if (tmp != pat)
            {
              if (options.value<CLI::anonymous_regex_trace> ())
                wcerr << '+' << endl;

              name = tmp;
              break;
            }

            if (options.value<CLI::anonymous_regex_trace> ())
              wcerr << '-' << endl;
          }

          UnsignedLong n (1);
          String escaped (name);

          while (conflict (escaped))
          {
            std::wostringstream os;
            os << n++;
            escaped = name + os.str ();
          }

          schema.new_edge<SemanticGraph::Names> (*ns, t, escaped);
        }

        Boolean
        conflict (String const& name)
        {
          using SemanticGraph::Type;
          using SemanticGraph::Schema;

          if (Type* t1 = find (schema, name))
          {
            // Check if this is a stable conflict. A conflict is unstable
            // if a conflicting type is visible from the root schema but
            // is not visible from the schema where the conflicting
            // element is defined.
            //
            Schema& s (dynamic_cast<Schema&> (ns->scope ()));

            Type* t2 (find (s, name));

            if (t1 != t2)
              throw UnstableConflict (*t1);

            return true;
          }

          return false;
        }

        SemanticGraph::Type*
        find (SemanticGraph::Schema& schema, String const& name)
        {
          using SemanticGraph::Type;
          using SemanticGraph::Scope;
          using SemanticGraph::Namespace;

          String ns_name (ns->name ());

          // Get all namespaces across include/import hierarchy with
          // our namespace name.
          //
          Scope::NamesIteratorPair nip (schema.find (ns_name));

          for (; nip.first != nip.second; ++nip.first)
          {
            Namespace& ns (dynamic_cast<Namespace&> (nip.first->named ()));

            Scope::NamesIteratorPair types (ns.find (name));

            for (; types.first != types.second; ++types.first)
            {
              if (Type* t = dynamic_cast<Type*> (&types.first->named ()))
              {
                return t;
              }
            }
          }

          return 0;
        }


        SemanticGraph::Path
        path (SemanticGraph::Nameable& n)
        {
          using SemanticGraph::Scope;
          using SemanticGraph::Schema;
          using SemanticGraph::Uses;

          Schema* schema (0);

          for (Scope* s (dynamic_cast<Scope*> (&n)
                         ? dynamic_cast<Scope*> (&n) : &n.scope ());;
               s = &s->scope ())
          {
            if (schema = dynamic_cast<Schema*> (s))
              break;
          }

          if (!schema->used ())
            return schema_path;

          Uses& u (*schema->used_begin ());
          return u.path ();
        }
      };
    }

    Void Processor::
    morph (Anonymous::CLI::Options const& ops,
           SemanticGraph::Schema& tu,
           SemanticGraph::Path const& file)
    {
      // Custom name mappings.
      //
      Context ctx (tu, file, ops);

      Traversal::Schema schema;
      Uses uses;

      schema >> uses >> schema;

      Traversal::Names schema_names;
      Namespace ns (ctx.ns);
      Traversal::Names ns_names;

      schema >> schema_names >> ns >> ns_names;

      Traversal::Scope scope; // goes to both types and groups
      Member member (ctx);

      ns_names >> scope;
      ns_names >> member;

      Traversal::Names names;

      scope >> names >> member;

      // Some twisted schemas do recusive inclusions.
      //
      tu.context ().set ("morphing-anonymous-processor-seen", true);

      schema.dispatch (tu);

      if (ctx.failed)
        throw Failed ();
    }
  }
}
