/*
 *  xfmedia - simple gtk2 media player based on xine
 *
 *  Copyright (c) 2004-2005 Brian Tarricone, <bjt23@cornell.edu>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; version 2 of the License ONLY.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Library General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <stdio.h>

#ifdef HAVE_STDLIB_H
#include <stdlib.h>
#endif

#ifdef HAVE_STRING_H
#include <string.h>
#endif

#ifdef HAVE_SYS_TYPES_H
#include <sys/types.h>
#endif

#ifdef HAVE_SYS_STAT_H
#include <sys/stat.h>
#endif

#ifdef HAVE_UNISTD_H
#include <unistd.h>
#endif

#ifdef HAVE_MMAP
#include <sys/mman.h>
#ifndef MAP_FILE
#define MAP_FILE 0
#endif
#endif

#include <glib.h>
#include <gtk/gtk.h>

#include <libxfce4util/libxfce4util.h>

#define EXO_API_SUBJECT_TO_CHANGE 1
#include <exo/exo.h>

#include "playlist-files.h"
#include "xfmedia-common.h"

static inline gboolean
file_needs_dir(const gchar *filename)
{
    g_return_val_if_fail(filename && *filename, FALSE);
    
    return (!strstr(filename, ":/") && *filename != '/');
}

static inline gchar *
string_strip_newlines(gchar *str)
{
    while(str[strlen(str)-1] == '\n' || str[strlen(str)-1] == '\r')
        str[strlen(str)-1] = 0;
    
    return str;
}


/********** playlist file load functions **********/

static gint
xfmedia_playlists_load_flat_file(FILE *fp, const gchar *basedir,
        XfmediaPlaylist *plist)
{
    gchar buf[PATH_MAX], buf1[PATH_MAX];
    gint nadded = 0;
    gchar *title_utf8;
    
    while(fgets(buf, PATH_MAX, fp)) {
        if(!*buf || *buf == '#' || *buf == '\n' || *buf == '\r')
            continue;
        
        title_utf8 = xfmedia_filename_to_name(string_strip_newlines(buf));
        
        if(file_needs_dir(buf)) {
            g_snprintf(buf1, PATH_MAX, "%s/%s", basedir, buf);
            xfmedia_playlist_append_entry(plist, title_utf8, -1, buf1, FALSE);
        } else
            xfmedia_playlist_append_entry(plist, title_utf8, -1, buf, FALSE);
        
        nadded++;
        g_free(title_utf8);
    }
    
    return nadded;
}

static gint
xfmedia_playlists_load_m3u(FILE *fp, const gchar *basedir,
        XfmediaPlaylist *plist)
{
    gchar buf[PATH_MAX], buf1[PATH_MAX], buf2[PATH_MAX], *p, *q;
    gint song_len, nadded = 0;
    
    /* check header */
    if(fgets(buf, PATH_MAX, fp)) {
        if(g_ascii_strncasecmp(buf, "#EXTM3U", 7)) {
            /* not an M3U file */
            gchar *title_utf8 = xfmedia_filename_to_name(string_strip_newlines(buf));
            
            if(file_needs_dir(buf)) {
                g_snprintf(buf1, PATH_MAX, "%s/%s", basedir, buf);
                xfmedia_playlist_append_entry(plist, title_utf8, -1, buf1, FALSE);
            } else
                xfmedia_playlist_append_entry(plist, title_utf8, -1, buf, FALSE);
            g_free(title_utf8);
            
            /* fall back to flat file loading */
            return xfmedia_playlists_load_flat_file(fp, basedir, plist) + 1;
        }
    }
    
    while(fgets(buf, PATH_MAX, fp)) {
        if(!g_ascii_strncasecmp(buf, "#EXTINF:", 8)) {
            p = buf + 8;
            q = strstr(p, ",");
            if(!p || *q == 0)
                continue;
            *q = 0;
            q++;
            song_len = atoi(p);
            if(song_len < 0 || *q == 0)
                continue;
            if(fgets(buf1, PATH_MAX, fp)) {
                gchar *title_utf8 = xfmedia_brutal_utf8_validate(string_strip_newlines(q));
                if(file_needs_dir(buf1)) {
                    g_snprintf(buf2, PATH_MAX, "%s/%s", basedir, buf1);
                    xfmedia_playlist_append_entry(plist,
                            title_utf8 ? title_utf8 : q, song_len,
                            string_strip_newlines(buf2), TRUE);
                } else {
                    xfmedia_playlist_append_entry(plist,
                            title_utf8 ? title_utf8 : q, song_len,
                            string_strip_newlines(buf1), TRUE);
                }
                if(title_utf8)
                    g_free(title_utf8);
                nadded++;
            }
        } else {
            gchar *title_utf8 = xfmedia_filename_to_name(string_strip_newlines(buf));
            if(file_needs_dir(buf)) {
                g_snprintf(buf1, PATH_MAX, "%s/%s", basedir, buf);
                xfmedia_playlist_append_entry(plist, title_utf8, -1, buf1, FALSE);
            } else
                xfmedia_playlist_append_entry(plist, title_utf8, -1, buf, FALSE);
            nadded++;
            g_free(title_utf8);
        }
    }
    
    return nadded;
}

static gint
xfmedia_playlists_load_pls(const gchar *filename, const gchar *basedir,
        XfmediaPlaylist *plist)
{
    XfceRc *rcfile;
    gint i, nentries, length = 0;
    const gchar *music_file, *title, *length_str;
    gchar key[128], buf[PATH_MAX], *title_utf8 = NULL;
    gint nadded = 0;
    
    rcfile = xfce_rc_simple_open(filename, TRUE);
    if(xfce_rc_has_group(rcfile, "playlist")) {
        xfce_rc_set_group(rcfile, "playlist");
        
        nentries = xfce_rc_read_int_entry(rcfile, "numberofentries", 0);
        for(i = 1; i <= nentries; i++) {
            g_snprintf(key, 128, "File%d", i);
            music_file = xfce_rc_read_entry(rcfile, key, NULL);
            if(!music_file)
                continue;
            
            g_snprintf(key, 128, "Title%d", i);
            title = xfce_rc_read_entry(rcfile, key, NULL);
            g_snprintf(key, 128, "Length%d", i);
            length_str = xfce_rc_read_entry(rcfile, key, NULL);
            if(length_str)
                length = atoi(length_str);
            
            if(file_needs_dir(music_file))
                g_snprintf(buf, PATH_MAX, "%s/%s", basedir, music_file);
            else
                *buf = 0;
            
            /* set song data directly, and only set info_loaded to TRUE if
             * we got both a length and title from the file */
            if(title)
                title_utf8 = xfmedia_brutal_utf8_validate(title);
            else
                title_utf8 = xfmedia_filename_to_name(*buf ? buf : music_file);
            xfmedia_playlist_append_entry(plist,
                    title_utf8 ? title_utf8 : title,
                    length_str ? length : -1, *buf ? buf : music_file,
                    ((title || title_utf8) && length_str) ? TRUE : FALSE);
            nadded++;
            
            if(title_utf8) {
                g_free(title_utf8);
                title_utf8 = NULL;
            }
        }
    }
    xfce_rc_close(rcfile);
    
    return nadded;
}


/***** begin ASX parser ****/

typedef struct
{
    XfmediaPlaylist *plist;
    const gchar *basedir;
    gint nadded;
    gboolean started;
    gchar *title;
    gchar *uri;
} ASXParserState;

static void
asx_xml_start(GMarkupParseContext *context, const gchar *element_name,
        const gchar **attribute_names, const gchar **attribute_values,
        gpointer user_data, GError **error)
{
    ASXParserState *state = user_data;
    
    if(!state->started) {
        if(!g_ascii_strcasecmp(element_name, "asx"))
            state->started = TRUE;
        else
            return;
    }
    
    if(!g_ascii_strcasecmp(element_name, "ref")) {
        gint i;
        
        if(state->uri) {
            g_free(state->uri);
            state->uri = NULL;
        }
        for(i = 0; attribute_names[i]; i++) {
            if(!g_ascii_strcasecmp(attribute_names[i], "href")) {
                state->uri = g_strdup(attribute_values[i]);
                break;
            }
        }
    }
}

static void
asx_xml_end(GMarkupParseContext *context, const gchar *element_name,
        gpointer user_data, GError **error)
{
    ASXParserState *state = user_data;
    
    if(!g_ascii_strcasecmp(element_name, "ASX"))
        state->started = FALSE;
    if(!state->started)
        return;
    
    if(!g_ascii_strcasecmp(element_name, "entry")) {
        if(state->uri) {
            gchar buf[PATH_MAX], *title_utf8;
            
            if(file_needs_dir(state->uri))
                g_snprintf(buf, PATH_MAX, "%s/%s", state->basedir, state->uri);
            else
                g_strlcpy(buf, state->uri, PATH_MAX);
            
            if(state->title)
                title_utf8 = xfmedia_brutal_utf8_validate(state->title);
            else
                title_utf8 = xfmedia_filename_to_name(state->uri);
            
            xfmedia_playlist_append_entry(state->plist,
                    title_utf8 ? title_utf8 : state->title, -1, buf, FALSE);
            
            state->nadded++;
            
            g_free(title_utf8);
            g_free(state->uri);
            state->uri = NULL;
        }
        if(state->title) {
            g_free(state->title);
            state->title = NULL;
        }
    }
            
}

static void
asx_xml_text(GMarkupParseContext *context, const gchar *text, gsize text_len,  
        gpointer user_data, GError **error)
{
    ASXParserState *state = user_data;
    const gchar *element_name;
    
    if(!state->started)
        return;
    
    element_name = g_markup_parse_context_get_element(context);
    if(!g_ascii_strcasecmp(element_name, "title")) {
        if(state->title) {
            g_free(state->title);
            state->title = NULL;
        }
        if(text_len > 0)
            state->title = g_strdup(text);
    }
}

static gint
xfmedia_playlists_load_asx(FILE *fp, const gchar *basedir,
        XfmediaPlaylist *plist)
{
    struct stat st;
#ifdef HAVE_SYS_MMAN_H
    void *addr = NULL;
#endif
    gchar *file_contents = NULL;
    GMarkupParseContext *pctx;
    GMarkupParser parser = {
        asx_xml_start,
        asx_xml_end,
        asx_xml_text,
        NULL,
        NULL
    };
    ASXParserState state;
    GError *err = NULL;
    
    if(fstat(fileno(fp), &st))
        return 0;
    
#ifdef HAVE_SYS_MMAN_H
    addr = mmap(NULL, st.st_size, PROT_READ, MAP_SHARED|MAP_FILE, fileno(fp), 0);
    if(MAP_FAILED != addr)
        file_contents = addr;
#endif
    
    if(!file_contents) {
        file_contents = g_malloc(st.st_size);
        if(fread(file_contents, 1, st.st_size, fp) != st.st_size) {
            g_free(file_contents);
            return 0;
        }
    }
    
    state.plist = plist;
    state.basedir = basedir;
    state.nadded = 0;
    state.started = FALSE;
    state.title = state.uri = NULL;
    pctx = g_markup_parse_context_new(&parser, 0, &state, NULL);    
    if(!g_markup_parse_context_parse(pctx, file_contents, st.st_size, &err)) {
        if(err) {
            g_critical("Unable to parse ASX playlist file (%d): %s",
                    err->code, err->message);
            g_error_free(err);
        }
    } else {
        if(!g_markup_parse_context_end_parse(pctx, &err)) {
            g_critical("Unable to finish parsing ASX playlist file (%d): %s",
                    err->code, err->message);
            g_error_free(err);
        }
    }
    g_markup_parse_context_free(pctx);
    
#ifdef HAVE_SYS_MMAN_H
    if(MAP_FAILED != addr) {
        munmap(addr, st.st_size);
        file_contents = NULL;
    }
#endif
    
    if(file_contents)
        g_free(file_contents);
    
    return state.nadded;
}

/***** end ASX parser *****/

static gint
xfmedia_playlists_load_wax(FILE *fp, const gchar *basedir,
        XfmediaPlaylist *plist)
{
    gchar buf[4096];
    gint nadded = 0;
    
    while(fgets(buf, 4096, fp) && g_ascii_strncasecmp(buf, "[reference]", 11))
        ;
        
    while(fgets(buf, 4096, fp)) {
        if(!g_ascii_strncasecmp(buf, "ref", 3)) {
            gchar *title_utf8 = xfmedia_filename_to_name(buf);
            xfmedia_playlist_append_entry(plist, title_utf8, -1, buf, FALSE);
            nadded++;
            g_free(title_utf8);
        }
    }
    
    return nadded;
}


/********** playlist file save functions **********/

static void
xfmedia_playlists_save_flat_file(XfmediaPlaylist *plist, FILE *fp)
{
    gchar *filename = NULL;
    gint i, tot;
    
    tot = xfmedia_playlist_get_n_entries(plist);
    for(i = 0; i < tot; i++) {
        if(xfmedia_playlist_get(plist, i, NULL, NULL, &filename) && filename) {
            fprintf(fp, "%s\n", filename);
            g_free(filename);
            filename = NULL;
        }
    }
}

static void
xfmedia_playlists_save_m3u(XfmediaPlaylist *plist, FILE *fp)
{
    gchar *title = NULL, *filename = NULL;
    gint length = -1, i, tot;
    gboolean metadata_loaded = FALSE;
    
    fputs("#EXTM3U\n", fp);
    
    tot = xfmedia_playlist_get_n_entries(plist);
    for(i = 0; i < tot; i++) {
        metadata_loaded = xfmedia_playlist_get_metadata_loaded(plist, i);
        if(metadata_loaded) {
            if(xfmedia_playlist_get(plist, i, &title, &length, &filename)) {
                fprintf(fp, "#EXTINF:%d,%s\n%s\n", length, title ? title : "",
                        filename);
                g_free(title);
                g_free(filename);
            }
        } else {
            if(xfmedia_playlist_get(plist, i, NULL, NULL, &filename)) {
                fprintf(fp, "%s\n", filename);
                g_free(filename);
            }
        }
    }
}

static void
xfmedia_playlists_save_pls(XfmediaPlaylist *plist, FILE *fp)
{
    gchar *title = NULL, *filename = NULL;
    gint length = -1, i, tot;
    gboolean metadata_loaded = FALSE;
    
    tot = xfmedia_playlist_get_n_entries(plist);
    fprintf(fp, "[playlist]\nnumberofentries=%d\n", tot);
    
    for(i = 0; i < tot; i++) {
        metadata_loaded = xfmedia_playlist_get_metadata_loaded(plist, i);
        if(metadata_loaded) {
            if(xfmedia_playlist_get(plist, i, &title, &length, &filename)) {
                fprintf(fp, "File%d=%s\nTitle%d=%s\nLength%d=%d\n",
                        i, filename, i, (title ? title : ""), i, length);
                g_free(title);
                g_free(filename);
            }
        } else {
            if(xfmedia_playlist_get(plist, i, NULL, NULL, &filename)) {
                fprintf(fp, "File%d=%s\n", i,filename ? filename : "");
                g_free(filename);
            }
        }
    }
}


gboolean
xfmedia_playlists_load(XfmediaPlaylist *plist, const gchar *filename)
{
    FILE *fp;
    gchar *p, *basedir;
    gboolean ret = FALSE;
    XfmediaPlaylistFormat format;
    gint nadded = 0;
    
    g_return_val_if_fail(filename && *filename, FALSE);
    
    format = xfmedia_playlists_query_playlist_format(filename);
    if(format == XFMEDIA_PLAYLIST_FORMAT_UNKNOWN)
        format = XFMEDIA_PLAYLIST_FORMAT_FLAT_FILE;
    
    fp = fopen(filename, "r");
    if(fp) {
        p = g_strrstr(filename, "/");
        if(p)
            basedir = g_strndup(filename, strlen(filename)-strlen(p));
        else
            basedir = g_strdup(g_get_current_dir());
        
        switch(format) {
            case XFMEDIA_PLAYLIST_FORMAT_M3U:
                nadded = xfmedia_playlists_load_m3u(fp, basedir, plist);
                break;
            
            case XFMEDIA_PLAYLIST_FORMAT_PLS:
                fclose(fp);
                fp = NULL;
                nadded = xfmedia_playlists_load_pls(filename, basedir, plist);
                break;
            
            case XFMEDIA_PLAYLIST_FORMAT_ASX:
                nadded = xfmedia_playlists_load_asx(fp, basedir, plist);
                break;
            
            case XFMEDIA_PLAYLIST_FORMAT_WAX:
                nadded = xfmedia_playlists_load_wax(fp, basedir, plist);
                break;
            
            default:
                nadded = xfmedia_playlists_load_flat_file(fp, basedir, plist);
                break;
        }
        
        g_free(basedir);
        
        if(fp)
            fclose(fp);
        ret = TRUE;
    }
    
    return ret;
}

gboolean
xfmedia_playlists_save(XfmediaPlaylist *plist, const gchar *filename,
        XfmediaPlaylistFormat format)
{
    FILE *fp;
    gboolean ret = FALSE;
    
    g_return_val_if_fail(filename && *filename, FALSE);
    
    if(format == XFMEDIA_PLAYLIST_FORMAT_UNKNOWN) {
        if(g_str_has_suffix(filename, ".m3u") || g_str_has_suffix(filename, ".M3U"))
            format = XFMEDIA_PLAYLIST_FORMAT_M3U;
        else if(g_str_has_suffix(filename, ".pls") || g_str_has_suffix(filename, ".PLS"))
            format = XFMEDIA_PLAYLIST_FORMAT_PLS;
        else if(g_str_has_suffix(filename, ".asx") || g_str_has_suffix(filename, ".ASX"))
            format = XFMEDIA_PLAYLIST_FORMAT_ASX;
        else if(g_str_has_suffix(filename, ".wax") || g_str_has_suffix(filename, ".WAX"))
            format = XFMEDIA_PLAYLIST_FORMAT_WAX;
        else
            format = XFMEDIA_PLAYLIST_FORMAT_FLAT_FILE;
    }
    
    fp = fopen(filename, "w");
    if(fp) {
        switch(format) {
            case XFMEDIA_PLAYLIST_FORMAT_M3U:
                xfmedia_playlists_save_m3u(plist, fp);
                break;
            case XFMEDIA_PLAYLIST_FORMAT_PLS:
                xfmedia_playlists_save_pls(plist, fp);
                break;
            case XFMEDIA_PLAYLIST_FORMAT_ASX:  /* no write support yet */
            case XFMEDIA_PLAYLIST_FORMAT_WAX:  /* no write support yet */
            case XFMEDIA_PLAYLIST_FORMAT_FLAT_FILE:
            default:
                xfmedia_playlists_save_flat_file(plist, fp);
                break;
        }
        
        fclose(fp);
        ret = TRUE;
    }
    
    return ret;
}

gboolean
xfmedia_playlists_is_playlist_file(const gchar *filename)
{
    return (g_str_has_suffix(filename, ".m3u")
            || g_str_has_suffix(filename, ".M3U")
            || g_str_has_suffix(filename, ".pls")
            || g_str_has_suffix(filename, ".PLS")
            || g_str_has_suffix(filename, ".asx")
            || g_str_has_suffix(filename, ".ASX")
            || g_str_has_suffix(filename, ".wax")
            || g_str_has_suffix(filename, ".WAX"));
}

XfmediaPlaylistFormat
xfmedia_playlists_query_playlist_format(const gchar *filename)
{
    g_return_val_if_fail(filename && *filename, XFMEDIA_PLAYLIST_FORMAT_UNKNOWN);
    
    if(g_str_has_suffix(filename, ".m3u") || g_str_has_suffix(filename, ".M3U"))
        return XFMEDIA_PLAYLIST_FORMAT_M3U;
    else if(g_str_has_suffix(filename, ".pls") || g_str_has_suffix(filename, ".PLS"))
        return XFMEDIA_PLAYLIST_FORMAT_PLS;
    else if(g_str_has_suffix(filename, ".asx") || g_str_has_suffix(filename, ".ASX"))
        return XFMEDIA_PLAYLIST_FORMAT_ASX;
    else if(g_str_has_suffix(filename, ".wax") || g_str_has_suffix(filename, ".WAX"))
        return XFMEDIA_PLAYLIST_FORMAT_WAX;
    
    return XFMEDIA_PLAYLIST_FORMAT_UNKNOWN;
}

gchar *
xfmedia_playlists_get_first_entry(const gchar *filename, gchar **title,
        gint *stream_length)
{
    FILE *fp;
    gchar *entry = NULL, buf[4096];
    
    if(g_str_has_suffix(filename, ".pls") || g_str_has_suffix(filename, ".PLS")) {
        gboolean got_file = FALSE, got_title = FALSE, got_length = FALSE;
        
        fp = fopen(filename, "r");
        if(!fp)
            return NULL;
        
        if(!title)
            got_title = TRUE;
        if(!stream_length)
            got_length = TRUE;
        
        while(fgets(buf, 4096, fp)) {
            if(!got_file && !g_ascii_strncasecmp(buf, "file1=", 6)) {
                while(buf[strlen(buf)-1] == '\n' || buf[strlen(buf)-1] == '\r')
                    buf[strlen(buf)-1] = 0;
                entry = g_strdup(buf+6);
                got_file = TRUE;
                continue;
            }
            
            if(!got_title && !g_ascii_strncasecmp(buf, "title1=", 7)) {
                while(buf[strlen(buf)-1] == '\n' || buf[strlen(buf)-1] == '\r')
                    buf[strlen(buf)-1] = 0;
                *title = g_strdup(buf+7);
                got_title = TRUE;
                continue;
            }
            
            if(!got_length && !g_ascii_strncasecmp(buf, "length1=", 8)) {
                while(buf[strlen(buf)-1] == '\n' || buf[strlen(buf)-1] == '\r')
                    buf[strlen(buf)-1] = 0;
                *stream_length = atoi(buf+8);
                got_length = TRUE;
            }
            
            if(got_file && got_title && got_length)
                break;
        }
        
        fclose(fp);
    } else if(g_str_has_suffix(filename, ".m3u") || g_str_has_suffix(filename, ".M3U")) {
        fp = fopen(filename, "r");
        if(!fp)
            return NULL;
        
        while(fgets(buf, 4096, fp)) {
            if(!*buf || *buf == '\n' || *buf == '\r')
                continue;
            
            if(!g_ascii_strncasecmp(buf, "#EXTM3U", 7))
                continue;
            
            while(buf[strlen(buf)-1] == '\n' || buf[strlen(buf)-1] == '\r')
                buf[strlen(buf)-1] = 0;
            
            if(!g_ascii_strncasecmp(buf, "#EXTINF:", 8)) {
                gchar *p;
                
                if(title) {
                    p = strstr(buf+8, ",");
                    if(p)
                        *title = g_strdup(p+1);
                }
                        
                if(stream_length) {
                    p = strstr(buf+8, ",");
                    if(p) {
                        *p = 0;
                        *stream_length = atoi(buf+8);
                    }
                }
                
                continue;
            }
            
            entry = g_strdup(buf);
            break;
        }
        
        fclose(fp);
    } else if(g_str_has_suffix(filename, ".asx") || g_str_has_suffix(filename, ".ASX")) {
        gboolean got_file = FALSE, got_title = FALSE;
        
        fp = fopen(filename, "r");
        if(!fp)
            return NULL;
        
        if(!title)
            got_title = TRUE;
        if(stream_length)
            *stream_length = -1;
        
        while(fgets(buf, 4096, fp)) {
            gchar *p, *q;
            
            if(!got_file && (p=strstr(buf, "<Ref href"))) {
                p += 9;
                while(*p && *p != '=')
                    p++;
                while(*p && *p != '"')
                    p++;
                p++;
                if(*p) {
                    q = strstr(p, "\"");
                    if(q) {
                        entry = g_strndup(p, q-p);
                        got_file = TRUE;
                        continue;
                    }
                }
            }
            
            if(!got_title && (p=strstr(buf, "<Title>"))) {
                p += 7;
                q = strstr(p, "</Title>");
                if(q) {
                    *title = g_strndup(p, q-p);
                    got_title = TRUE;
                }
            }
            
            if(got_file && got_title)
                break;
        }
        
        fclose(fp);
    } else if(g_str_has_suffix(filename, ".wax") || g_str_has_suffix(filename, ".WAX")) {
        fp = fopen(filename, "r");
        
        while(fgets(buf, 4096, fp) && g_ascii_strncasecmp(buf, "[reference]", 11))
            ;
        
        while(fgets(buf, 4096, fp)) {
            if(!g_ascii_strncasecmp(buf, "ref1=", 5)) {
                while(buf[strlen(buf)-1] == '\n' || buf[strlen(buf)-1] == '\r')
                    buf[strlen(buf)-1] = 0;
                entry = g_strdup(buf+5);
                break;
            }
        }
        
        fclose(fp);
    }
    
    return entry;
}

static const gchar *cache_extensions[] = {
    "asx",
    "m3u",
    "pls",
    "wax",
    NULL
};

/* FIXME: this function causes a double-free() crash */
gchar *
xfmedia_playlists_cache_find(const gchar *original_uri)
{
    gchar *filename, *md5sum, buf[512];
    gint i;
    
    g_return_val_if_fail(original_uri && *original_uri, NULL);
    
    md5sum = exo_str_get_md5_str(original_uri);
    
    for(i = 0; cache_extensions[i]; i++) {
        g_snprintf(buf, sizeof(buf), "xfmedia/cached-playlists/%s.%s", md5sum,
                cache_extensions[i]);
        g_free(md5sum);
        filename = xfce_resource_save_location(XFCE_RESOURCE_CACHE, buf, FALSE);
        if(!filename)
            return NULL;
        
        if(g_file_test(filename, G_FILE_TEST_EXISTS))
            return filename;
        
        g_free(filename);
    }
    
    return NULL;
}

gchar *
xfmedia_playlists_cache_save(const gchar *original_uri, const gchar *data,
        const gchar *extension)
{
    gchar *filename, *md5sum, buf[512];
    FILE *fp;
    
    g_return_val_if_fail(original_uri && *original_uri && data && *data, NULL);
    
    md5sum = exo_str_get_md5_str(original_uri);
    
    g_snprintf(buf, sizeof(buf), "xfmedia/cached-playlists/%s.%s", md5sum,
            extension);
    g_free(md5sum);
    filename = xfce_resource_save_location(XFCE_RESOURCE_CACHE, buf, TRUE);
    if(!filename)
        return NULL;
    
    fp = fopen(filename, "w");
    if(!fp) {
        g_free(filename);
        return NULL;
    }
    fwrite(data, 1, strlen(data), fp);
    fclose(fp);
    
    return filename;
}
