/* Copyright 2001-2005 Matt Flax <flatmax@ieee.org>
   This file is part of MFFM Time Scale Modification for Audio.

   MFFM Time Scale Modification for Audio is free software; you can redistribute
 it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.
   
   MFFM Time Scale Modification for Audio is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.
   
   You have received a copy of the GNU General Public License
   along with MFFM Time Scale Modification for Audio
 */
#ifndef WSOLA4SDL_H
#define WSOLA4SDL_H


#include <SDL/SDL_sound.h>
#include "WSOLA.H"

#define DEF_TAU 1.0
#define FS 44100

  /** This class interfaces WSOLA and Audiere.
      By interfacing WSOLA and Audiere, time scale modificaiton of audio
      (speeding and slowing of audio in real time) is accomplished. Audiere
      is responsible for file reading and decomression as well as sound card
      device operation. WSOLA is responsible for the time scale modification
      in real time.
      
      To use this class one must :
      //Start Audiere as normal ...
      
      //Construct a WSOLA based sound source...
      WSOLA4SDL mWsolaSource = new WSOLA4SDL(mSampleSource, mTau, mFrameStartIndex, mFrameEndIndex);
      
      //create a stream using WSOLA4SDL as the source
      OutputStreamPtr  mOutputStream(OpenSound(mAudioDevice, mWsolaSource , true));
      //play the stream to hear it.
      mOutputStream->play();
      
      //You may now alter the speed of audio playback in real time ...
      mWsolaSource->setTau(float tau); //Where float is the speed of playback where : fast < tau=1.0 < slow
      
      Upon finishing, the callback function is executed so WSOLA4SDL can
      signal the parent application that the stream has finished playing.
  */
  class WSOLA4SDL {
    
    ///This private function reads, scales and writes audio to/from Audiere
    int WSOLARead(int frame_count){
      cout<<"WSOLA4SDL::WSOLARead"<<endl;
      return 0;
      /*
      int lastFrame=0;
      int framesProcessed=(int)((double)windowSize/2.0/(double)channels);
      //exit strategy
      if ((wsola->getSourceIndex()/channels+framesProcessed)>input.getEnd()){
	std::cout<<"last frame : exit as you wish"<<endl;
	lastFrame=1;
	//input-=framesProcessed;
	//setPosition(getLength());
	//		reset();
	//	return 0;
	std::cout<<"input.getEnd() "<<input.getEnd()<<"wsola->getSourceIndex()/channels "<<wsola->getSourceIndex()/channels<<endl;
	framesProcessed=input.getEnd()-wsola->getSourceIndex()/channels;
	std::cout<<"last frame : "<<framesProcessed<<" frames dropped"<<endl;
	if (lastFrameCallback)
	  lastFrameCallback(callBackData);
	//				if (callBackData)
	//					::SendMessage((HWND)callBackData, WM_MESSAGETOAMIS, amis::AUDIO_CLIP_FINISHED, 0);
	return 0;
      }
      
      if (firstFrame){
	std::cout<<"firstFrame"<<endl;
	//load the input data to the buffer and then to wsola
	m_source->setPosition(wsola->getSourceIndex()/channels);//seek to the correct location
	int readCnt=m_source->read(wsola->getSourceLength()/channels, &(*input.window)[0]);
	//std::cout<<"read "<<readCnt<<" frames from the source : wanted "<<wsola->getSourceLength()/channels<<" frames"<<endl;
	wsola->initProcess(&(*input.window)[0], tau);
	//std::cout<<"wsola->getSourceLength() "<<wsola->getSourceLength()<<endl;
	//std::cout<<"readCnt="<<readCnt<<endl;
	firstFrame=0;
      }
      
      //load the input data to the buffer and then to wsola
      m_source->setPosition(wsola->getSourceIndex()/channels);//seek to the correct location
      //std::cout<<"here"<<endl;
      int readCnt;
      readCnt=m_source->read(wsola->getSourceLength()/channels, &(*input.window)[0]);
      //std::cout<<"read "<<readCnt<<" frames from the source : wanted "<<wsola->getSourceLength()/channels<<" frames"<<endl;
      wsola->loadSourceInput(&(*input.window)[0]);
      
      //Load desired input data
      m_source->setPosition(wsola->getDesiredIndex()/channels);//seek to the correct location
      readCnt=m_source->read(wsola->getDesiredLength()/channels, &(*input.window)[0]);
      
      wsola->loadDesiredInput(&(*input.window)[0]);
      
      //Process outputting to the local memory buffer
      if (tau!=1.0){
	int ret=wsola->processFrame(output, tau); //process halfWindow
	if (ret<0){
	  if (ret==WSOLA::INPUT_READ2DF_ERR)
	    std::cout<<"Destination frame read error, most probably using a factor >=1.0 and at the end of the input file"<<endl;
	  else if (ret==WSOLA::INPUT_READ2SF_ERR)
	    std::cout<<"Source frame read error, most probably using a factor <1.0, and at the end of the input file"<<endl;
	  else if (ret==WSOLA::WRITE_ERR)
	    std::cout<<"Is your disk full, we encountered a write error"<<endl;
	  else if (ret==WSOLA::PROCFFT_ERR)
	    std::cout<<"Please contact the author - this error shouldn't occur."<<endl;
	  else if (ret==WSOLA::DEFAULT_ERR)
	    std::cout<<"Un-known default error encountered"<<endl;
	  else //Handle other errors here
	    std::cout<<"Unknown error - unreliable failure"<<endl;
	}
      } else {
	memcpy(output,
	       &(*input.window)[0],framesProcessed*channels*sizeof(ATYPE));
	wsola->shiftOn(tau);
      }
      
      if (lastFrame)
	reset();
      //	return WSOLA::FINISHED_NORMALLY;
      
      //std::cout<<input<<endl;
      return framesProcessed;
      */
    }
  public:
    
    /**The constructor allows one to pass the audiere file (source) and
       specify the initial speed (ta) as well as the start point (startPoint)
       and end point (endPoint). The callback function is operational in non
       WIN32 environments, where it is called upon hitting the last frame
       lastFrameCallback_in(callBackData_in);
    */
    WSOLA4SDL(char *fileName, double ta=DEF_TAU, int startPoint=0, int endPoint=0, void (*lastFrameCallback_in)(void *)=NULL, void *callBackData_in=NULL) {
      SDL_desiredInfo=NULL;
      SDL_desired=SDL_obtained=NULL;

      //set up the various SDL structures
      SDL_desired = new SDL_AudioSpec;
      if (!SDL_desired){
	cerr<<"WSOLA4SDL :: Couldn't malloc the SDL_AudioSpec"<<endl;
	exit(-1);
      }
      SDL_obtained = new SDL_AudioSpec;
      if (!SDL_obtained){
	cerr<<"WSOLA4SDL :: Couldn't malloc the SDL_AudioSpec"<<endl;
	exit(-1);
      }

      SDL_desiredInfo = new Sound_AudioInfo;
      if (!SDL_desiredInfo){
	cerr<<"WSOLA4SDL :: Couldn't malloc the SDL_desiredInfo"<<endl;
	exit(-1);
      }

      //Set up the windowSize
      //default to CD quality audio
      cout<<"HERE"<<endl;
      SDL_desiredInfo->rate=FS;
      SDL_desiredInfo->format=AUDIO_S16;
      SDL_desiredInfo->channels=2;
      //Work out the buffer size
      windowSize =SDL_desiredInfo->channels*HANNING_LENGTH(SDL_desiredInfo->rate);
      //SDL requires a buffer which is a power of 2
      double temp=log((double)windowSize)/log(2.0);
      if (fabs(windowSize-pow(2,floor(temp)))<fabs(windowSize-pow(2,floor(temp))))
	windowSize=(int)pow(2,floor(temp));
      else
	windowSize=(int)pow(2,ceil(temp));

      std::cout<<"windowSize="<<windowSize<<endl;
      // Make sure that the half window size is an integer number of channels
      while (remainder((double)windowSize /2.0/(double)SDL_desiredInfo->channels, floor((double)windowSize /2.0/(double)SDL_desiredInfo->channels)) != 0.0){
	cout<<"adjusting windowSize"<<endl;
	windowSize ++;
      }
      std::cout<<"windowSize="<<windowSize<<endl;

      //Try to open a file ...
      SDL_Source=Sound_NewSampleFromFile(fileName, SDL_desiredInfo, windowSize);
      if (SDL_Source == NULL){
            fprintf(stderr, "Couldn't load \"%s\"!\n"
                            "  reason: [%s].\n",
                            fileName, Sound_GetError());
	    exit(-1);
      } else {
	cout<<"file opened successfully.\nRate: "<<SDL_Source->actual.rate<<" Hz (wanted " << SDL_Source->desired.rate << "Hz)"<<endl;
	cout<<"Format: "<<SDL_Source->actual.format<<" (wanted " << SDL_Source->desired.format << ")"<<endl;
	cout<<"channels: "<<(int)SDL_Source->actual.channels<<" (wanted " << (int)SDL_Source->desired.channels << ")"<<endl;	
      }

      //Setup the SDL library to call the callback and use the same format
      SDL_desired->freq=SDL_desiredInfo->rate;
      SDL_desired->format=SDL_desiredInfo->format;
      SDL_desired->channels=SDL_desiredInfo->channels;
      SDL_desired->samples=windowSize/2/SDL_desiredInfo->channels;
      SDL_desired->callback=read;
      SDL_desired->userdata=NULL;

      //open the audio device
      if ( SDL_OpenAudio(SDL_desired, SDL_obtained) < 0 ){
	fprintf(stderr, "Couldn't open audio: %s\n", SDL_GetError());
	exit(-1);
      }      

      cout<<"the buffer size in bytes is "<<SDL_obtained->size<<endl;

      if (SDL_obtained->freq !=SDL_desired->freq){
	cout<<"WSOLA4SDL :: obtained sample rate error"<<endl;
	exit(-1);
      }

      if (SDL_obtained->channels !=SDL_desired->channels){
	cout<<"WSOLA4SDL :: obtained channel count error"<<endl;
	exit(-1);
      }

      //Set up WSOLA
      tau = ta;
      firstFrame = 1;
      wsola=0;
      wsola = new WSOLA (windowSize, SDL_desiredInfo->rate, SDL_desiredInfo->channels);//Version 2 instantiation;	
      if (!wsola){
	std::cout<<"WSOLA4SDL(SampleSourcePtr source) : couldn't instantiate WSOLA"<<endl;
	exit(-1);
      }
      
      output=NULL; //Null the input / output buffers
      
      //Reserve local memory stores
      
      //set the input time code which holds the length of the audio
      // as well as the beginning, current and end locations
      // as well as a window for the input data
      input.init(startPoint,m_source->getLength()+1); //the point we want to start from
      setPosition(startPoint);
      (*input.window)=wsola->getSourceLength(); //Set the window size
      input.setFinish(m_source->getLength()+1); //Set the absolute length
      //std::cout<<input<<endl;
      if (endPoint!=0)//Guard against dummy entry
	if (endPoint<=input.getFinish()-1)
	  input.setEnd(endPoint); //the point we want to end at
	else
	  input.setEnd(input.getFinish()-1); //Force end at end of stream
      //This is default ...
      std::cout<<"input:\n"<<input<<endl;
      
      //Set up the output array
      output = new ATYPE[windowSize]; //The output store
      if (output==NULL){
	std::cout<<"error defining output memory"<<endl;
	exit(-2);
      }
      
      //set up the callback funciton
      lastFrameCallback=lastFrameCallback_in;
      callBackData=callBackData_in;

      //play sound
      //        SDL_PauseAudio(0);

    }
    
    ///Deconstructor
    ~WSOLA4SDL(){
      if (SDL_desired) delete SDL_desired;
      if (SDL_obtained) delete SDL_obtained;
      if (SDL_desiredInfo) delete SDL_desiredInfo;
      if (wsola) delete wsola; wsola=NULL;
      if (output) delete [] output; output=NULL;
    }

    /*
    ///Used to get various format information like channel count, sample rate and sample format
    void ADR_CALL getFormat(int& channel_count, int& sample_rate, SampleFormat& sample_format){
      m_source->getFormat(channel_count,sample_rate,sample_format);
      std::cout<<"channel_count,sample_rate,sample_format"<<channel_count<<'\t'<<sample_rate<<'\t'<<sample_format<<endl;
      if (GetSampleSize(sample_format)!=wsola->getFrameSize())
	printf("Difference between Audiere (%d bytes) and WSOLA (%d bytes) frame sizes ... should be OK as long as WSOLA > Audiere",GetSampleSize(sample_format),wsola->getFrameSize());
    }

    */    

    ///This overloaded function is called by Audiere to fill the output buffer

    //void my_audio_callback(void *userdata, Uint8 *stream, int len);
    static void read(void* userdata, Uint8* buffer, int frame_count) {
      cout<<"WSOLA4SDL::read(void* userdata="<<userdata<<", void* buffer="<<buffer<<", int frame_count="<<frame_count<<") "<<endl;

      /*
      //std::cout<<"request frame_count "<<frame_count<<endl;
      int framesProcessed=0, realSize=(int)((double)windowSize/2.0/(double)channels);
      //std::cout<<"framesProcessed "<<framesProcessed<<" realSize "<<realSize<<endl;
      while ((framesProcessed+realSize)<frame_count){
	int oneCycleCnt=WSOLARead(realSize);
	if (oneCycleCnt>0){
	  memcpy(&((ATYPE*)buffer)[framesProcessed*channels], output,oneCycleCnt*channels*sizeof(ATYPE));
	  framesProcessed+=oneCycleCnt;
	} else
	  break;
      }
      //std::cout<<"returning "<<framesProcessed<<" wanted "<<frame_count<<endl;
      return framesProcessed;
      */
    }
    
    /*    ///Resets the source and WSOLA
    void ADR_CALL reset() {
      m_source->reset();
      wsola->reset();
    }
    
    ///Checks whether the source is seekable
    bool ADR_CALL isSeekable() {
      return m_source->isSeekable();
    }
    ///Returns the length of the source
    int  ADR_CALL getLength() {
      return m_source->getLength();
    }
    
    ///Set the position of both the source and WSOLA accordingly
    void ADR_CALL setPosition(int position){
      m_source->setPosition(position);
      wsola->setPosition(position*channels);
      input=position;
    }
    
    ///Position is indexed to the original stream index, as tau is variable
    int  ADR_CALL getPosition(){
      std::cout<<"getpos"<<endl;
      //return input.getCount();
      return m_source->getPosition();
    }
    
    ///Tells whether the source is set to repeat
    bool ADR_CALL getRepeat() {
      return m_source->getRepeat();
    }
    ///Sets the state of the source repeat
    void ADR_CALL setRepeat(bool repeat){
      m_source->setRepeat(repeat);
    }
    ///Sets the play back speed
    void  setTau(float ta) {
      std::cout<<"setTau "<<ta<<endl;
      this->tau = ta;
      std::cout<<"tau is now "<<this->tau<<endl;
    }
    ///Get the playback speed
    float getTau(){
      return tau;
    }
    
    ///Conformity with new Audiere API which supports tags : getTagCount
    int getTagCount() {return m_source->getTagCount();}
    ///Conformity with new Audiere API which supports tags : getTagKey
    const char* getTagKey(int i) {return m_source->getTagKey(i);}
    ///Conformity with new Audiere API which supports tags : getTagValue
    const char* getTagValue(int i) {return m_source->getTagValue(i);}
    ///Conformity with new Audiere API which supports tags : getTagType
    const char* getTagType(int i) {return m_source->getTagType(i);}
    
  private:
    ///The input source (file)
    SampleSourcePtr m_source;

    ///The input source (file)
    Sound_Sample SDL_source;
    
    ///The number of channels in the source
    int channels;
    /// The sample rate of the source (Hz)
    int fs;
    /// Audiere related sample format
    SampleFormat mSampleFormat;
    
    
    */
  private:
    Sound_Sample * SDL_Source; //The file being played
    //lib SDL stuff ...
    SDL_AudioSpec *SDL_desired, *SDL_obtained; //The audio desired and callback info.

    //SDL_sound file handling ...
    Sound_AudioInfo * SDL_desiredInfo;// The desired format

    /// The size of the hanning window (in bytes)
    int windowSize;

    ///The time scale modification class
    WSOLA *wsola;
    
    ///Locally stored speed variable
    float tau;
    ///Indication that this is the first frame
    int firstFrame;
    
    /// Input buffer
    TIMECODETYPE_W input;
    /// Output buffer
    ATYPE *output;

  public:
    /** A callback function which notifies a parent app. that the last frame is being processed.
	This function is passed some user data (if necessary)
    */
    void (*lastFrameCallback)(void *);
    ///The user data to pass in the callback function argument
    void *callBackData;

  };
#endif //WSOLA4SDL_H_
