/* Copyright 2001 Matt Flax <flatmax@ieee.org>
   This file is part of MFFM Time Scale Modification for Audio.

   MFFM Time Scale Modification for Audio is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.
   
   MFFM Time Scale Modification for Audio is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.
   
   You have received a copy of the GNU General Public License
   along with MFFM Time Scale Modification for Audio
 */
#ifndef FILEBUFFER_H_
#define FILEBUFFER_H_

/* This class is intended to stream through an audio file and buffer
   the audio data.
   It is designed to NOT seek ... as not all audio file libraries can
   seek. Consequently due to this limitation, it can't rewind nor seek.

   This class will suit a limited number of audio processing applications
   which only process data locally on a number (audio) stream. WSOLA is a
   good example of local signal processing.


   Data is not considered to be framed (as in channel framed), it is rather
   treated as a stream of numbers.
*/

#include <st.h>
#include "sox_routines.h"

#define ATYPE st_sample_t

//extern "C" void open_input(ft_t ft);
//extern "C" void st_fail(const char *fmt, ...);
//extern "C" void st_warn(const char *fmt, ...);

#include "./WSOLA.H"

//#include <values.h>

#define WINDOW_MULTIPLICITY 3 //If this is not even, then check it doesn't ruin anything.

/** This FileBuffer is for streaming forward through audio files.
	It is customised to work with WSOLA and Sox.

	This file buffer removes the requirement of seeking in sound files and thus
	removes CPU load by ensuring decoding is only required once.

	The paradigm works in a similar way to my bit stream class but is also quite different.
	A buffer of memory is reserved. The read cycle is as follows :
	a] The last half of the buffer is copied to the first half of the buffer.
	b] A window of data is read from file to the last half of a buffer.
	
	Once all data requests are local to the last half of the buffer, the read cycle is executed.

	In this way a whole file is streamed.

	This paradigm is not reserved for WSOLA nor SOX, so kindly replace at will.

  */

#define DEBUG_FB
#ifdef DEBUG_FB
#include <fstream>
#endif

class FileBuffer {

  //Declare a structure which is suitable for this application
  // This structure holds time code.
  TIMECODETYPE_W buffer;
  
#ifdef DEBUG_FB
  ofstream *tempOutput;
#endif
  
  //Declare a modulus operator in the form of a counter
  counter windowModHalf;
  
  //In this version sox is used as the file buffer
  ft_t file_desc;
  
  // read from file to buffer, returns 0 on success
  int readHalf(void){
    //cout<<"\n\n enter readHalf \n\n"<<endl;
    
    int readCnt=(*buffer.window).getCount()/2;
    
    //First copy the last half to the first half ...
    memcpy(&(*buffer.window)[0],&(*buffer.window)[readCnt],readCnt*sizeof(ATYPE));
    
    //Now read into the second half
    int readC=(*file_desc->h->read)(file_desc,(st_sample_t*)(&(*buffer.window)[readCnt]), (st_ssize_t)readCnt);
    if (readCnt!=readC){
      std::cout<<"\nread "<<readC<<" samples from the source : wanted "<<readCnt<<" samples"<<endl;
      if (readC>0){
	cout<<"Work out how to pad zeros and exit"<<endl;
	//file_desc->info.channels
	memset((void*)(&(*buffer.window)[readC*file_desc->info.channels]),0,(readCnt-readC)*file_desc->info.channels*sizeof(ATYPE));
	//	    buffer+=readC;
	//	    buffer+=readCnt;
	//		return readCnt;
      } else {
	memset((void*)(&(*buffer.window)[0]),0,readCnt*file_desc->info.channels*sizeof(ATYPE));
	//	    buffer+=readCnt;
	//			return readCnt;
      }
    } else
      ;//cout<<"read "<<readCnt<<" sample OK - continue at will"<<endl;
    
#ifdef DEBUG_FB
    tempOutput->write((char*)(&(*buffer.window)[readCnt]),readCnt*sizeof(ATYPE));
#endif
    
    //increment the current time
    buffer+=readCnt;
    return readCnt;
  }
  
public:
  FileBuffer(const char *fileName, int windowCnt, int channels, int fs){
    cout<<"AYTPE size in bytes = "<<sizeof(ATYPE)<<endl;
    //Prepare the time code ...
    int startPoint=0;

#ifdef DEBUG_FB
	tempOutput=NULL;
	tempOutput=new ofstream("test.raw",ios::binary);
#endif

    //windowModHalf=NULL;
    buffer.init(startPoint,1024); //the point we want to start from
    buffer.setFinish(MAXINT); //Set the absolute length (normally automatic)
    buffer.setEnd(MAXINT-2);
    //(*buffer.window).setFrameSize(channels);
    (*buffer.window)=windowCnt*WINDOW_MULTIPLICITY; //Set the WSOLA window size

    //Do regular sox open and begin read stuff
    file_desc=NULL; //Sox instantiation
    file_desc=st_initformat();
    file_desc->filename= (char*)fileName;
    open_input(file_desc);
    if (!file_desc){
      std::cout<<"WSOLA4SoxNAudiere("<<fileName<<") : file couldn't be opened"<<endl;
		throw -1;
    }
    
    file_desc->info.channels=channels;
    file_desc->info.rate=fs;
    file_desc->filetype="auto";
    if (st_gettype(file_desc)){
      st_fail("Unknown input file format for '%s':  %s", file_desc->filename, file_desc->st_errstr);
		throw -2;
	}

    //Start reading the sox input
    if ((*file_desc->h->startread)(file_desc) != ST_SUCCESS){
	st_fail("Failed reading %s: %s",file_desc->filename,
		file_desc->st_errstr);
		throw -3;
    }
    
    //Have a look at the sox file settings now :
    printf("opened file %s: seekable=%d, length=%d, of type=%s, with comment=%s, rate=%d, channels=%d and word size=%d\n",file_desc->filename,file_desc->seekable,file_desc->length, file_desc->filetype, file_desc->comment, file_desc->info.rate, file_desc->info.channels, file_desc->info.size);
    if (st_checkformat(file_desc) )
      st_fail("bad input format for file %s: %s", file_desc->filename,file_desc->st_errstr);
    printf("Input file %s: using sample rate %lu\n\tsize %s, encoding %s, %d %s",
	   file_desc->filename, file_desc->info.rate,
	   st_sizes_str[(unsigned char)file_desc->info.size],
	   st_encodings_str[(unsigned char)file_desc->info.encoding],
	   file_desc->info.channels,
	   (file_desc->info.channels > 1) ? "channels" : "channel");

    //set up the modulator
    windowModHalf=0;
    windowModHalf.setMaxCount((*buffer.window).getCount()/2);

    if (readHalf()<0){//Fill the first half of the buffer
      cerr<<"FileBuffer: initial reading failed, is the file smaller then "<<(*buffer.window).getCount()*WINDOW_MULTIPLICITY<<"samples ?\n Otherwise, please check why"<<endl;
    }
    if (readHalf()<0){//Fill the second half of the buffer
      cerr<<"FileBuffer: initial reading failed, is the file smaller then "<<(*buffer.window).getCount()*WINDOW_MULTIPLICITY<<"samples ?\n Otherwise, please check why"<<endl;
    }

  }

  //Destructor
  ~FileBuffer(){
    //Close the file ...
    if ((*file_desc->h->stopread)(file_desc) == ST_EOF)
      st_warn(file_desc->st_errstr);
    fclose(file_desc->fp);
    //if (windowModHalf) delete windowModHalf; windowModHalf=NULL;
    //free(file_desc->filename);

#ifdef DEBUG_FB
	if (tempOutput){
		tempOutput->close();
		delete tempOutput; tempOutput=NULL;
	}
#endif
  }

  //getData - returns zero on success
  int getData(int srcLoc, int srcCnt, st_sample_t* source, int desiredLoc, int desiredCnt, st_sample_t* desired){

    //cout<<"srcLoc="<<srcLoc<<" desiredLoc="<<desiredLoc<<endl;

    int loc=min(srcLoc,desiredLoc);

    int res=0;
    //Check to see that we aren't trailing the active portion ...
    while (buffer.getCount()<=(loc+buffer.window->getCount()/2))
      res=readHalf();
    
    //This shouldn't be necessary.
    //loc=min(srcLoc,desiredLoc);
    //Check to see that we aren't in the last portion of the window
    //if (((buffer.getCount()-loc)<(buffer.window->getCount()/2)))
    //  res=readHalf();
    
    //Copy the source data
    //windowModHalf.resetCarry();
    //windowModHalf=srcLoc;
    //memcpy(source, &(*buffer.window)[windowModHalf.getCount()],srcCnt*sizeof(ATYPE));
    int readPoint=buffer.window->getCount()+srcLoc-buffer.getCount();
    if (readPoint<0)
      return 0;
    memcpy(source, &(*buffer.window)[readPoint],srcCnt*sizeof(ATYPE));
    
    //Copy the desired data
    //windowModHalf.resetCarry();
    //windowModHalf=desiredLoc;
    //memcpy(desired, &(*buffer.window)[windowModHalf.getCount()],desiredCnt*sizeof(ATYPE));
    readPoint=buffer.window->getCount()+desiredLoc-buffer.getCount();
    if (readPoint+desiredCnt > buffer.window->getCount())
      readPoint=readPoint;
    memcpy(desired, &(*buffer.window)[readPoint],desiredCnt*sizeof(ATYPE));
    
    if (res<0)
      return res;
    else
      return 0;
  }
  
  // A rough indication or where we are at in a file
  int getPosition(void){
    return buffer.getCount();
  }
};
#endif//FILEBUFFER_H_
