/* nwc_fsentry.cc
 * This file belongs to Worker, a filemanager for UNIX/X11.
 * Copyright (C) 2006 Ralf Hoffmann.
 * You can contact me at: ralf@boomerangsworld.de
 *   or http://www.boomerangsworld.de/worker
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

#include "nwc_fsentry.hh"
#include <string.h>  //memcpy

namespace NWC
{

  FSEntry::FSEntry( const std::string &fullname )
  {
    _fullname = fullname;
    _exists = false;
  
    memset( &_statbuf, 0, sizeof( _statbuf ) );
    memset( &_dest_statbuf, 0, sizeof( _dest_statbuf ) );
    _is_link = false;
    _is_corrupt = false;

    if ( _fullname[0] == '/' ) {
      worker_struct_stat stbuf;
    
      if ( worker_lstat( _fullname.c_str(), &stbuf ) == 0 ) {
        _statbuf.size = stbuf.st_size;
        _statbuf.lastaccess = stbuf.st_atime;
        _statbuf.lastmod = stbuf.st_mtime;
        _statbuf.lastchange = stbuf.st_ctime;
        _statbuf.mode = stbuf.st_mode;
        _statbuf.userid = stbuf.st_uid;
        _statbuf.groupid = stbuf.st_gid;
        _statbuf.inode = stbuf.st_ino;
        _statbuf.nlink = stbuf.st_nlink;
        _statbuf.blocks = stbuf.st_blocks;
        _statbuf.rdev = stbuf.st_rdev;
        _statbuf.dev = stbuf.st_dev;

        _exists = true;
      }

      if ( S_ISLNK( _statbuf.mode ) ) {
        _is_link = true;

        if ( worker_stat( _fullname.c_str(), &stbuf ) == 0 ) {
          _dest_statbuf.size = stbuf.st_size;
          _dest_statbuf.lastaccess = stbuf.st_atime;
          _dest_statbuf.lastmod = stbuf.st_mtime;
          _dest_statbuf.lastchange = stbuf.st_ctime;
          _dest_statbuf.mode = stbuf.st_mode;
          _dest_statbuf.userid = stbuf.st_uid;
          _dest_statbuf.groupid = stbuf.st_gid;
          _dest_statbuf.inode = stbuf.st_ino;
          _dest_statbuf.nlink = stbuf.st_nlink;
          _dest_statbuf.blocks = stbuf.st_blocks;
          _dest_statbuf.rdev = stbuf.st_rdev;
          _dest_statbuf.dev = stbuf.st_dev;
        } else {
          /* corrupt */
          _is_corrupt = true;
          _dest_statbuf.mode = 0;  // to prevent corrupted links treated as dirs
          if ( errno == ENOENT ) {
            // Eventl. fuer spaetere Betrachtungen
          }
        }
      }
    }
  }

  FSEntry::FSEntry( const FSEntry &other )
  {
    _fullname = other._fullname;
    _exists = other._exists;
    memcpy( &_statbuf, &other._statbuf, sizeof( other._statbuf ) );
    memcpy( &_dest_statbuf, &other._dest_statbuf, sizeof( other._dest_statbuf ) );
    _is_link = other._is_link;
    _is_corrupt = other._is_corrupt;
  }

  FSEntry::~FSEntry()
  {
  }

  bool FSEntry::isDir( bool follow_symlinks ) const
  {
    bool is_dir = false;

    if ( _exists == true ) {
      if ( follow_symlinks == true && _is_link == true && _is_corrupt == false ) {
        if ( S_ISDIR( _dest_statbuf.mode ) ) is_dir = true;
      } else if ( S_ISDIR( _statbuf.mode ) ) is_dir = true;
    }
    return is_dir;
  }

  std::string FSEntry::getFullname() const
  {
    return _fullname;
  }

  bool FSEntry::entryExists() const
  {
    return _exists;
  }

  FSEntry *FSEntry::clone() const
  {
    return new FSEntry( *this );
  }

  std::string FSEntry::getBasename() const
  {
    const std::string &fn = getFullname();
  
    std::string::size_type pos = fn.rfind( "/" );
    if ( pos == std::string::npos )
      return fn;

    return std::string( fn, pos + 1 );
  }

  std::string FSEntry::getDirname() const
  {
    const std::string &fn = getFullname();
  
    std::string::size_type pos = fn.rfind( "/" );
    if ( pos == std::string::npos )
      return std::string( "." );

    return std::string( fn, 0, pos );
  }

  bool FSEntry::isReg( bool follow_symlinks ) const
  {
    bool is_reg = false;

    if ( _exists == true ) {
      if ( follow_symlinks == true && _is_link == true && _is_corrupt == false ) {
        if ( S_ISREG( _dest_statbuf.mode ) ) is_reg = true;
      } else if ( S_ISREG( _statbuf.mode ) ) is_reg = true;
    }
    return is_reg;
  }

    bool FSEntry::isLink() const
    {
        return _is_link;
    }

    loff_t FSEntry::stat_size() const
    {
        return _statbuf.size;
    }
    
    loff_t FSEntry::stat_dest_size() const
    {
        return _dest_statbuf.size;
    }

    dev_t FSEntry::stat_dev() const
    {
        return _statbuf.dev;
    }

    dev_t FSEntry::stat_dest_dev() const
    {
        return _dest_statbuf.dev;
    }

    ino_t FSEntry::stat_inode() const
    {
        return _statbuf.inode;
    }

    ino_t FSEntry::stat_dest_inode() const
    {
        return _dest_statbuf.inode;
    }
}
