// WideMargin. Simple fast bible software.
// Copyright (C) 2011  Daniel Hughes
//
// This program is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program.  If not, see <http://www.gnu.org/licenses/>.

using System;
using Gtk;
using System.Linq;
using System.Data.Linq;
using System.Collections.Generic;
using WideMargin.MVCInterfaces;
using System.Text;
using WideMargin.Utilities;

namespace WideMargin.GUI
{
	[System.ComponentModel.ToolboxItem(true)]
	public partial class SearchResultsWidget : Gtk.Bin, ISearchResultsView
	{
		private EventLocker _eventLocker = new EventLocker();
		private EventHandler<LinkClickedEventArgs<IVerseIdentifier>> _verseLinkActivated;
		private EventHandler<PageRequestEventArgs> _pageRequest;
		private PageRanger _pageRanger;
		private TextView _textView;
		
		public SearchResultsWidget()
		{
			this.Build ();
			var scrolledWindow = new GaplessScrolledWindow();
			_textView = new TextView();
			_textView.WrapMode = WrapMode.Word;
			scrolledWindow.ShadowType = ShadowType.In;
			scrolledWindow.Add(_textView);
			Add(scrolledWindow);
			
			_textView.MotionNotifyEvent += HandleTextViewMotionNotifyEvent;
			_pageRanger = new PageRanger();
		}

		/// <summary>
		/// Occurs when the mouse moves over the text
		/// If its over a link show the hand cursor.
		/// </summary>
		/// <param name="o">sender</param>
		/// <param name="args">Movement arguments</param>
		private void HandleTextViewMotionNotifyEvent (object o, MotionNotifyEventArgs args)
		{
			int x;
			int	y;
			_textView.WindowToBufferCoords (Gtk.TextWindowType.Widget,
		                                    (int)args.Event.X,
		                                    (int)args.Event.Y,
		                                    out x,
		                                    out y);

			Gtk.TextIter iter = _textView.GetIterAtLocation (x, y);
			bool foundLink = false;
			foreach(TextTag tag in iter.Tags)
			{
				if(tag is LinkTag<IVerse> || tag is LinkTag<int>)
				{
					foundLink = true;
					break;
				}
			}
			
			Gdk.Window window = _textView.GetWindow (Gtk.TextWindowType.Text);
			if(!foundLink)
			{
				window.Cursor = new Gdk.Cursor (Gdk.CursorType.Xterm);
			}
			else
			{
				window.Cursor = new Gdk.Cursor (Gdk.CursorType.Hand2);
			}
		}
		
		private void ClearTags()
		{
			TextBuffer buffer = _textView.Buffer;
			buffer.RemoveAllTags(buffer.GetIterAtOffset (0),
		   						 buffer.GetIterAtOffset (buffer.Text.Length - 1));
			List<TextTag> tagsToRemove = new List<TextTag>();
			buffer.TagTable.Foreach(tag => tagsToRemove.Add(tag));
			foreach(TextTag tag in tagsToRemove)
			{
				buffer.TagTable.Remove(tag);
			}
			
		}
		
		public void ShowResults(IEnumerable<IVerse> verses, int numberOfPages, int currentPage)
		{
			Application.Invoke((_,__) =>
			{
				//clean up existing tags
				ClearTags();

				//create new text
				List<LinkTag<IVerse>> links = new List<LinkTag<IVerse>>(10);
				StringBuilder builder = new StringBuilder();
				TextBuffer buffer = _textView.Buffer;
				foreach(IVerse verse in verses)
				{
					string linkLine = string.Format("{0} {1}:{2}",
					                                verse.Book,
							                        verse.Chapter,
							                        verse.VerseNumber);
					
					LinkTag<IVerse> linkTag = new LinkTag<IVerse> (linkLine, verse);
					
					linkTag.LinkClicked += HandleLinkTagLinkClicked;
						
					buffer.TagTable.Add(linkTag);

					links.Add(linkTag);
					
					builder.AppendLine(linkLine);
					builder.AppendLine(verse.Contents);
					builder.AppendLine();
				}
				
				//create page link line
				var pageLinks = new List<LinkTag<int>>();
				LinkTag<int> backLink = null;
				LinkTag<int> forwardLink = null;
				
				int firstPage;
				int lastPage;
				_pageRanger.GetPageRange(numberOfPages,currentPage, out firstPage, out lastPage);
				
				if (numberOfPages >  0 && currentPage > 0)
				{
					builder.Append("<  ");
					backLink = new LinkTag<int>("pagePrevious", currentPage - 1);
					backLink.LinkClicked += HandleBackLinkLinkClicked;
					buffer.TagTable.Add(backLink);
				}
				for(int index = firstPage; index <= lastPage; index++)
				{
					builder.AppendFormat("{0}  ",index + 1); //make text one based
					if(index != currentPage)
					{
						var pageLink = new LinkTag<int>(string.Format("page{0}",index), index);
						pageLink.LinkClicked += HandleBackLinkLinkClicked;
						pageLinks.Add(pageLink);
						buffer.TagTable.Add(pageLink);
					}
				}
				if (numberOfPages > 0 && currentPage < numberOfPages - 1)
				{
					builder.Append(">");
					forwardLink = new LinkTag<int>("pageNext", currentPage + 1);
					forwardLink.LinkClicked += HandleBackLinkLinkClicked;
					buffer.TagTable.Add(forwardLink);
				}
				
				string text = builder.ToString();
				buffer.Text = text;
				
				//add verse link tags
				foreach(TextTag link in links)
				{
					AddTag(text, link.Name, false, buffer, link);
				}

				//add page link tags
				if (numberOfPages >  0 && currentPage > 0)
				{
					AddTag(text, "<", true, buffer, backLink);
				}
				foreach(LinkTag<int> link in pageLinks)
				{
					string linkTarget = (link.LinkTarget + 1).ToString() + " ";
					AddTag(text, linkTarget, true, buffer, link, 1);
				}
				if(numberOfPages > 0 && currentPage < numberOfPages - 1)
				{
					AddTag(text, ">", true, buffer, forwardLink);
				}
				
				//centre the last line
				TextTag centerTag = new TextTag("Centre");
				buffer.TagTable.Add(centerTag);
				centerTag.Justification = Justification.Center;
				centerTag.SizePoints = 14;
				string lastLine = text.Substring(text.LastIndexOf(Environment.NewLine) + 1);
				AddTag(text, lastLine, true, buffer, centerTag);				
			});
		}
		
		
		public void GetPageRange(int numberOfPages, int currentPage, out int firstPage, out int lastPage)
		{
			//figure out which end of the range we are closer to
			int distanceToEnd = numberOfPages - currentPage;
			int distanceToStart = currentPage;
			
			if(distanceToEnd < distanceToStart)
			{
				if(distanceToEnd > 5)
				{
					lastPage = currentPage + 4;
					firstPage = lastPage - 10;

				}
				lastPage = numberOfPages - 1;
				firstPage = lastPage - 10;
				if(firstPage < 0)
				{
					firstPage = 0;	
				}
				return;
			}

			if(distanceToStart > 5)
			{
				firstPage = currentPage - 4;
				lastPage = firstPage + 10;
			}
			firstPage = 0;
			lastPage = 9;
			if(lastPage > numberOfPages - 1)
			{
				lastPage = numberOfPages - 1;	
			}
		}

		private void HandleBackLinkLinkClicked (object sender, LinkClickedEventArgs<int> e)
		{
			_pageRequest.Fire(this, new PageRequestEventArgs(e.LinkTarget, e.Button));
		}
		
		private void AddTag(string text, string linkText, bool lastIndex, TextBuffer textBuffer, TextTag tag)
		{
			AddTag(text, linkText, lastIndex, textBuffer, tag, 0);
		}
		
		private void AddTag(string text, string linkText, bool lastIndex, TextBuffer textBuffer, TextTag tag, int reduceLength)
		{
			int startIndex;
			if(lastIndex)
			{ 
				startIndex = text.LastIndexOf(linkText);
			}
			else
			{
				startIndex = text.IndexOf(linkText);
			}
			TextIter iter1 = textBuffer.GetIterAtOffset (startIndex); 
			TextIter iter2 = textBuffer.GetIterAtOffset (startIndex + linkText.Length - reduceLength); 
			textBuffer.ApplyTag(tag, iter1, iter2);
		}

		private void HandleLinkTagLinkClicked (object sender, LinkClickedEventArgs<IVerse> e)
		{
			_verseLinkActivated.Fire(this, new LinkClickedEventArgs<IVerseIdentifier>(e.LinkTarget.Identifier, e.Button));
		}
		
		public event EventHandler<LinkClickedEventArgs<IVerseIdentifier>> VerseLinkActivated
		{
			add
			{
				_eventLocker.Add(ref _verseLinkActivated, value);	
			}
			remove
			{
				_eventLocker.Remove(ref _verseLinkActivated, value);				
			}
		}
		
		public event EventHandler<PageRequestEventArgs> PageRequest
		{
			add
			{
				_eventLocker.Add(ref _pageRequest, value);
			}
			remove
			{
				_eventLocker.Remove(ref _pageRequest, value);
			}
		}
	}
}

