/*
 * MFVec3f.cpp
 *
 * Copyright (C) 1999 Stephen F. White
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program (see the file "COPYING" for details); if 
 * not, write to the Free Software Foundation, Inc., 675 Mass Ave, 
 * Cambridge, MA 02139, USA.
 */

#include <stdio.h> 
#include "stdafx.h"

#include "MFVec3f.h"
#include "SFVec3f.h"
#include "Vec3f.h"
#include "DuneApp.h"

MFVec3f::MFVec3f() : MFFloat(3)
{
}

FieldValue *MFVec3f::copy()
{ 
    const float *values = ((MFFloat *)MFFloat::copy())->getValues();
    return new MFVec3f((float *) values, getSize());
}

bool
MFVec3f::equals(const FieldValue *value) const
{
    return value->getType() == MFVEC3F && 
           MFFloat::equals((const MFFloat *) value);
}

FieldValue *
MFVec3f::getSFValue(int index) const
{
    return new SFVec3f(getValue(index));
}

void
MFVec3f::setSFValue(int index, FieldValue *value)
{
#ifdef DEBUG
    if (value->getType() != SFVEC3F) {
	assert(0);
	return;
    }
#endif

    setSFValue(index, ((SFVec3f *) value)->getValue());
}

void
MFVec3f::setSFValue(int index, const float *values)
{
    for (int i = 0; i < getStride(); i++)
        _value[index * getStride() + i] = values[i];
}

MyString
MFVec3f::getEcmaScriptComment(MyString name, int flags) const
{
    const char *indent = ((FieldValue *)this)->getEcmaScriptIndent(flags);
    MyString ret;
    ret = "";
    if (TheApp->GetEcmaScriptAddAllowedValues()) {
        ret += indent;
        ret += "// allowed values:\n";

        ret += indent;
        ret += "   // array ([0] [1] [2] [3] ...) of 3D Vectors, 3 floating point numbers\n";
    }
    if (TheApp->GetEcmaScriptAddAllowedComponents()) {
        ret += indent;
        ret += "// allowed components:\n";

        ret += indent;
        ret += "   // x: ";
        ret += name;
        ret += "[???].x or ";
        ret += name;
        ret += "[???][0]\n";

        ret += indent;
        ret += "   // y: ";
        ret += name;
        ret += "[???].y or ";
        ret += name;
        ret += "[???][1]\n";

        ret += indent;
        ret += "   // z: ";
        ret += name;
        ret += "[???].z or ";
        ret += name;
        ret += "[???][2]\n";
    }
    if (TheApp->GetEcmaScriptAddAvailableFunctions()) {
        ret += indent;
        ret += "// available functions:\n";
        if (flags != EL_EVENT_IN) {
            ret += indent;
            ret += "   // ";
            ret += name;
            ret += " = new MFVec3f(sfvec3f_v1, sfvec3f_v2, ...);\n";
        }
        if (flags != EL_EVENT_OUT) {
            ret += indent;
            ret += "   // int_i = ";
            ret += name;
            ret += ".length();\n";

            ret += indent;
            ret += "   // string_str = ";
            ret += name;
            ret += ".toString();\n";
       }
    }
    if (TheApp->GetEcmaScriptAddExampleUsage()) {
        ret += indent;
        ret += "// example usage:\n";
        if (flags != EL_EVENT_IN) {
             ret += indent;
             ret += "   // ";
             ret += name;
             ret += " = new MFVec3f(SFVec3f(3, 5.2, 1), sfvec3f_v);\n";

             ret += indent;
             ret += "   // ";
             ret += name;
             ret += "[0].z = 0.5;\n";
        } 
        if (flags != EL_EVENT_OUT) {
             ret += indent;
             ret += "   // float_z =";
             ret += name;
             ret += "[0].z;\n";
        }
        if (flags == EL_FIELD_DEF) {
             ret += indent;
             ret += "   // ";
             ret += name;
             ret += "[0] = ";
             ret += name;
             ret += "[1].normalize();\n";
        } 
    }
    return ret;
}

void 
MFVec3f::insertSFValue(int index, FieldValue *value)
{
    insertSFValue(index, ((SFVec3f *)value)->getValue()); 
}

void 
MFVec3f::insertSFValue(int index, const float *values)
{
    for (int i = 0; i < getStride(); i++)
        _value.insert(values[i], index * getStride() + i);
}

Vec3f
MFVec3f::getMinBoundingBox(void)
{
    Vec3f retMin(0, 0, 0);
    const float *values = getValues();
    if (getSFSize() > 0) {
       retMin[0] = values[0];
       retMin[1] = values[1];
       retMin[2] = values[2];
       for (int j = 1; j < getSFSize(); j++)
           for (int i = 0; i < 3; i++)
               if (values[j * 3 + i] < retMin[i])
                   retMin[i] = values[j * 3 + i];
    }
    return retMin;
}

Vec3f
MFVec3f::getMaxBoundingBox(void)
{
    Vec3f retMax(0, 0, 0);
    const float *values = getValues();
    if (getSFSize() > 0) {
       retMax[0] = values[0];
       retMax[1] = values[1];
       retMax[2] = values[2];
       for (int j = 1; j < getSFSize(); j++)
           for (int i = 0; i < 3; i++)
               if (values[j * 3 + i] > retMax[i])
                   retMax[i] = values[j * 3 + i];
    }
    return retMax;
}

void
MFVec3f::flip(int index)
{
    for (int i = 0; i < getSFSize(); i++)
        _value[i * 3 + index] *= -1.0;
}

