/**************************************************************************
* This file is part of the WebIssues program
* Copyright (C) 2006 Michał Męciński
* Copyright (C) 2007-2008 WebIssues Team
*
* This program is free software; you can redistribute it and/or modify
* it under the terms of the GNU General Public License as published by
* the Free Software Foundation; either version 2 of the License, or
* (at your option) any later version.
**************************************************************************/

#ifndef DEFINITIONINFO_H
#define DEFINITIONINFO_H

#include <QVariant>
#include <QMap>

namespace WebIssues
{

/**
* Structure storing an attribute definition.
*
* The attribute definition consists of a type and metadata.
*
* Standard metadata for all attribute types is the required flag and the default value.
*
* Refer to the WebIssues protocol for information about additional metadata available for
* each type of attribute.
*/
class DefinitionInfo
{
public:
    /**
    * Type of an attribute.
    */
    enum Type
    {
        /** Invalid attribute definition. */
        Invalid,
        /** A <tt>TEXT</tt> attribute type. */
        Text,
        /** An <tt>ENUM</tt> attribute type. */
        Enum,
        /** A <tt>NUMERIC</tt> attribute type. */
        Numeric,
        /** A <tt>DATETIME</tt> attribute type. */
        DateTime,
        /** A <tt>USER</tt> attribute type. */
        User,

        First = Text,
        Last = User
    };

public:
    /**
    * Default constructor.
    *
    * Construct an invalid attribute definition.
    */
    DefinitionInfo();

    /**
    * Destructor.
    */
    ~DefinitionInfo();

public:
    /**
    * Check if the definition is valid.
    *
    * @return @c true if the definition is valid.
    */
    bool isValid() const { return m_type != Invalid; }

    /**
    * Set the attribute type.
    */
    void setType( Type type ) { m_type = type; }

    /**
    * Return the attribute type.
    */
    Type type() const { return m_type; }

    /**
    * Set the required flag of the attribute.
    */
    void setRequired( bool required ) { m_required = required; }

    /**
    * Return the required flag of the attribute.
    */
    bool required() const { return m_required; }

    /**
    * Set the default value of the attribute.
    */
    void setDefaultValue( const QString& value ) { m_default = value; }

    /**
    * Return the default value of the attribute.
    */
    QString defaultValue() const { return m_default; }

    /**
    * Set the metadata of the attribute.
    *
    * @param key Name of the metadata to set. It can consist of lowercase letters and hyphens.
    *
    * @param value Value of the metadata. It can be an integer, string or an array of strings.
    *              If it's a null value, the metadata is cleared.
    */
    void setMetadata( const QString& key, const QVariant& value );

    /**
    * Return the metadata of the attribute.
    *
    * @param key Name of the metadata to return.
    *
    * @return Value of the metadata or a null value if the metadata is not defined.
    */
    QVariant metadata( const QString& key ) const;

    /**
    * Return a list of names of defined metadata.
    */
    QStringList metadataKeys() const;

private:
    Type m_type;

    bool m_required;
    QString m_default;

    QMap<QString, QVariant> m_metadata;
};

}

#endif
