//<copyright>
//
// Copyright (c) 1996
// Institute for Information Processing and Computer Supported New Media (IICM),
// Graz University of Technology, Austria.
//
// This file is part of VRweb.
//
// VRweb is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2, or (at your option)
// any later version.
//
// VRweb is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with VRweb; see the file LICENCE. If not, write to the
// Free Software Foundation, Inc., 59 Temple Place - Suite 330,
// Boston, MA 02111-1307, USA.
//
// Note that the GNU General Public License does not permit incorporating
// the Software into proprietary or commercial programs. Such usage
// requires a separate license from IICM.
//
//</copyright>

//<file>
//
// Name:        clipping.C
//
// Purpose:     Interface to clipping algorithms
//
// Created:      8 Feb 1996  Georg Meszaros
//
// Modified:     
//
//
// Description:
//
// 
// 
//
//</file>


//<function>
//
// Prototype: Face* clip(
//              Face* face, 
//              vector3D* normal
//            );    
//
// Purpose: clip a polygon against a plane
//
//
// Description:
//
//
//
//
//</function>

#include "clipping.h"

#include <vrml/QvMFLong.h>
#include <ge3d/vectors.h>
#include <ge3d/color.h>
#include <hyperg/utils/types.h>

#include "polygon.h"
#include "faceattr.h"

#include <iostream.h>
#include <math.h>

// decision if a point is on a plane - error of calculation
const float margin_ = 0.01;



void Clipping::print3D(const point3D& p)
{
  cerr << "(" << p.x << "," << p.y << "," << p.z << ") \n";
}


void Clipping::printRGB(const colorRGB& c)
{
  cerr << "(" << c.R << "," << c.G << "," << c.B << ") \n";
}


int Clipping::inFront(const point3D& vertex, const vector3D& normal, float plane)
{
  return ( dot3D(normal, vertex) > plane ); 
}

int Clipping::inBack(const point3D& vertex, const vector3D& normal, float plane)
{
  return ( dot3D(normal, vertex) < plane ); 
}

int Clipping::onPlane(const point3D& vertex, const vector3D& normal, float plane)
{
  float dotproduct = dot3D(normal, vertex);
  if (dotproduct == plane)
    return 1;  
  if (dotproduct < plane)
    return ( (plane - dotproduct) < margin_ ); 
  // (dotproduct > plane)
    return ( (dotproduct - plane) < margin_ );
}


point3D* Clipping::intersection(const point3D& vertex_1, const point3D& vertex_2,
                                const vector3D& normal, float plane,
                                const vector3D& normal_1, const vector3D& normal_2,
                                vector3D& new_normal)
{
// Calculates the intersection point of the ray from vertex_1 to
// vertex_2 with the plane described by the face normal vector and 
// d of a plane equation ax + by + cz = d = plane.
//
// Also the interpolated normal vector for the newly created vertex is
// calculated.

  // calculate the normal vector for the new vertex 
  double norm;
  point3D* new_point;
  
  vector3D diff_1_new;
  vector3D diff_new_2;
  
  double distance_1_new;
  double distance_new_2;

  vector3D weighted_1;
  vector3D weighted_2;

//cerr << "before add 1 and 2\n";
//print3D(normal_1);
//print3D(normal_2);
//print3D(new_normal);

  new_point = intersection(vertex_1, vertex_2, normal, plane);

  if (new_point)
  {
    sub3D(normal_1, *new_point, diff_1_new);
    sub3D(*new_point, normal_2, diff_new_2);
    distance_1_new = norm3D(diff_1_new);
    distance_new_2 = norm3D(diff_new_2);
   
    if (distance_1_new < 0) distance_1_new*=-1;
    if (distance_new_2 < 0) distance_new_2*=-1;
    
    weighted_1.x = normal_1.x;
    weighted_1.y = normal_1.y; 
    weighted_1.z = normal_1.z;
    weighted_2.x = normal_2.x;
    weighted_2.y = normal_2.y; 
    weighted_2.z = normal_2.z;
     
    scl3D(weighted_1, distance_new_2);
    scl3D(weighted_2, distance_1_new);
    
    add3D(weighted_1, weighted_2, new_normal);

    norm = norm3D(new_normal);

    if (norm) scl3D(new_normal, 1/norm); 
  }
  return new_point;
}


point3D* Clipping::intersection(const point3D& vertex_1, const point3D& vertex_2,
                                const vector3D& normal, float plane)
{
// Calculates the intersection point of the ray from vertex_1 to
// vertex_2 with the plane described by the face normal vector and 
// d of a plane equation ax + by + cz = d = plane.
// If the intersection can not be calculated (ray lies parallel or
// on plane) this method returns a nil pointer. 
// There is an intersection, even when both vertices lie on the same
// side of the polygon.
// Allocates new memory for return value


  point3D* point = nil;

  float t;
  vector3D diff;
  sub3D(vertex_2, vertex_1, diff); 
  float scalar = dot3D(normal, diff); 
 
  // line is parallel to plane
  if (!scalar) return nil;
  if ((scalar > 0) && (scalar < margin_)) return nil;
  if ((scalar < 0) && (scalar > -margin_)) return nil;
  
  t = (plane - dot3D(normal, vertex_1)) / scalar; 
  point = new point3D;

  scl3D(diff, t);
  add3D(vertex_1, diff, *point);

  return point;
}


void Clipping::clip(Face& face, const vector3D& normal, float plane, 
                    Face*& front_face, Face*& back_face, SplitType& split_type)


{
//cerr << "*Clipping::clip\n";
// trivial accept, reject before starting to split

  point3D* vertex_list = face.faceAttr()->vertexList();
  int* vertex_index_list = face.vertexIndexList();
  unsigned vertex_num = face.vertexNumber();

  // normal_index_list is the same as the vertex_index_list !!! 
  //vector3D* normal_list = face.faceAttr()->normalList();
 
  point3D current_vertex;
  point3D next_vertex;
  int current_index = vertex_index_list[0];
  int next_index = vertex_index_list[1];
  
  int current_infront = 0;
  int next_infront = 0;
  int current_onplane = 0;
  int next_onplane = 0;
  int current_inback = 0;
  int next_inback = 0;
  
  enum Trivial{all_in_front, all_in_back, mixed, either};
  Trivial last_status;
  Trivial new_status;
 
  // at least 3 vertices necessary, a normal and the plane

  current_vertex = vertex_list[current_index];
  next_vertex = vertex_list[next_index];
   
  current_inback = inBack(current_vertex, normal, plane);      
  current_infront = inFront(current_vertex, normal, plane);
  current_onplane = onPlane(current_vertex, normal, plane);
  next_inback = inBack(next_vertex, normal, plane);
  next_infront = inFront(next_vertex, normal, plane);
  next_onplane = onPlane(next_vertex, normal, plane);

  if (current_onplane && next_onplane) new_status = either;  
  else if (current_onplane && next_infront) new_status = all_in_front;
  else if (current_onplane && next_inback) new_status = all_in_back;
  else if (current_infront && next_onplane) new_status = all_in_front;
  else if (current_inback && next_onplane) new_status = all_in_back;
  else if (current_infront && next_infront) new_status = all_in_front;
  else if (current_inback && next_inback) new_status = all_in_back;
  else if (current_infront && next_inback) new_status = mixed;
  else if (current_inback && next_infront) new_status = mixed;
  

  for (int i=1; (i < vertex_num - 1) && (new_status != mixed); i++)
  { 
    last_status = new_status;
    current_index = next_index;
    next_index = vertex_index_list[i+1];

    current_vertex = next_vertex;
    next_vertex = vertex_list[next_index];
      
    current_inback = next_inback;      
    current_infront = next_infront;
    current_onplane = next_onplane;
    next_inback = inBack(next_vertex, normal, plane);
    next_infront = inFront(next_vertex, normal, plane);
    next_onplane = onPlane(next_vertex, normal, plane);
              
    if (current_onplane && next_onplane) new_status = either;  
    else if (current_onplane && next_infront) new_status = all_in_front;
    else if (current_onplane && next_inback) new_status = all_in_back;
    else if (current_infront && next_onplane) new_status = all_in_front;
    else if (current_inback && next_onplane) new_status = all_in_back;
    else if (current_infront && next_infront) new_status = all_in_front;
    else if (current_inback && next_inback) new_status = all_in_back;
    else if (current_infront && next_inback) new_status = mixed;
    else if (current_inback && next_infront) new_status = mixed;
     
    // last == new, nothing to do
    // last == either, new stays unchanged
    if ((new_status != last_status) && (last_status != either))
       if (new_status == either) new_status = last_status;
       else new_status = mixed;
       
  }


  // 'either' says that the face lies on the plane
 
  if (new_status == either)
  {
    split_type = SAME_PLANE;
    return;
  } 

  if (new_status == all_in_back)
  {
    split_type = ONLY_BACK; 
    back_face = &face;
    return;
  }   
  
  if (new_status == all_in_front) 
  {
    split_type = ONLY_FRONT;
    front_face = &face;
    return;
  }


  // ***********************************  
  // not trivial, splitting is necessary

  // these two faces are allocated in split

  front_face = split(face, normal, plane, FRONT);
  back_face = split(face, normal, plane, BACK);

  split_type = SPLIT;
  return;

}



Face* Clipping::split(Face& face, 
                      const vector3D& normal, float plane,
                      PolygonSide whichside)

{
//cerr << "*Clipping::split\n";

// Sutherland Hodgman
// allocates memory for a new face and for new indexlists on the free store
// if splitting is necessary
// A value of -1 indicates the end of an vertex index list

// When Clipping::intersection is called in this function then there is always
// an intersection between the 2 points and the plane, the security
// check on nil as return value could be rewritten if the function 
// intersection was only called here.

  point3D* vertex_list = face.faceAttr()->vertexList();
  vector3D* normal_list;
  int* vertex_index_list = face.vertexIndexList();
  unsigned vertex_num = face.vertexNumber();
  int vertex_count=0;
  
  point3D current_vertex;
  point3D next_vertex;
  int current_index = vertex_index_list[0];
  int next_index = vertex_index_list[1];

  int current_inback = 0;
  int next_inback = 0;
  
  // for complex polygons at most vertex_num+1 vertices are needed
  // one more is needed for the terminating -1 value in the index lists

  // this array is going to be deleted after copying indices to an array of the correct size
  QvMFLong* new_face_indices = new QvMFLong;
  new_face_indices->allocValues(vertex_num+2); // 
  //cerr << "original array lenght: " << vertex_num << "\n"; 

  vector3D normal_vector = face.normal();
 
  // point3D new_vertex;   // unused
  vector3D new_normal;
  init3D(new_normal,0,0,0);

  point3D* intersec; // to calculate the intersection
 
  next_index = vertex_index_list[0];
  next_vertex = vertex_list[next_index];      
  next_inback = inBack(next_vertex, normal, plane);


  for (int j=0; j < vertex_num; j++)
  { 
       // we need to reload the lists because it might have been
       // moved by a reallocation process - extendVertices(...)
       vertex_list = face.faceAttr()->vertexList();
       normal_list = face.faceAttr()->normalList(); 

       init3D(new_normal,0,0,0);       

       current_index = next_index;
       if (j == (vertex_num - 1)) next_index = vertex_index_list[0];
       else next_index = vertex_index_list[j+1];

       current_vertex = next_vertex;
       next_vertex = vertex_list[next_index];

       current_inback = next_inback;      
       next_inback = inBack(next_vertex, normal, plane);
      

       // =======================  
       if ( whichside == FRONT ) 
       {
         if (!current_inback && !next_inback) 
         {
           if (vertex_count >= new_face_indices->num) 
              new_face_indices->allocValues(vertex_count+1);
           new_face_indices->values[vertex_count++] = next_index;         
         }
         else
         if  (!current_inback && next_inback)
         {
           if (!onPlane(current_vertex, normal, plane))
           {
             if (face.normalNumber()) 
                intersec = intersection(current_vertex,  next_vertex, normal, plane,
                                        normal_list[current_index], normal_list[next_index],
                                        new_normal);
             else
             {
               intersec = intersection(current_vertex, next_vertex, normal, plane);
             }

             if (intersec)
             {
               if (vertex_count >= new_face_indices->num) 
                   new_face_indices->allocValues(vertex_count+1);
               new_face_indices->values[vertex_count++] = 
                 face.faceAttr()->extendVertices(*intersec, new_normal);
               delete intersec;
             }
           }  
         }
         else
         if (current_inback && !next_inback)
         {
           if (onPlane(next_vertex, normal, plane))
           {
             if (vertex_count >= new_face_indices->num) 
                   new_face_indices->allocValues(vertex_count+1);
             new_face_indices->values[vertex_count++] = next_index;
           }
           else  
           {
             if (face.normalNumber())            
                intersec = intersection(current_vertex, next_vertex, normal, plane,
                                        normal_list[current_index], normal_list[next_index],
                                        new_normal);
             else
             {
               intersec = intersection(current_vertex, next_vertex, normal, plane);
             }

             if (intersec)
             {
               if (vertex_count+1 >= new_face_indices->num) 
                   new_face_indices->allocValues(vertex_count+2);
               new_face_indices->values[vertex_count++] = 
                 face.faceAttr()->extendVertices(*intersec, new_normal);                 
               new_face_indices->values[vertex_count++] = next_index;
               delete intersec;             
             }  
           }
         }
 
       } // if whichside == front


       // ==========================
       if ( whichside == BACK )  
       {
        
         if (current_inback && next_inback)
         {
           if (vertex_count >= new_face_indices->num) 
              new_face_indices->allocValues(vertex_count+1);
           new_face_indices->values[vertex_count++] = next_index;         
         }
         else
         if  (current_inback && !next_inback)
         {
           if (onPlane(next_vertex, normal, plane))
           {
             if (vertex_count >= new_face_indices->num) 
                new_face_indices->allocValues(vertex_count+1);
             new_face_indices->values[vertex_count++] = next_index;
           }
           else
           {
             if (face.normalNumber())
               intersec = intersection(current_vertex,  next_vertex, normal, plane,
                                       normal_list[current_index], normal_list[next_index],
                                       new_normal);
             else 
             {
               intersec = intersection(current_vertex, next_vertex, normal, plane);
             }

             if (intersec)
             {
               if (vertex_count >= new_face_indices->num) 
                   new_face_indices->allocValues(vertex_count+1);
               new_face_indices->values[vertex_count++] = 
                 face.faceAttr()->extendVertices(*intersec, new_normal);
               delete intersec;
             }
           }  
         }
         else
         if (!current_inback && next_inback)
         {
           if (onPlane(current_vertex, normal, plane))
           {
             if (vertex_count+1 >= new_face_indices->num) 
                new_face_indices->allocValues(vertex_count+2);
             new_face_indices->values[vertex_count++] = current_index;
             new_face_indices->values[vertex_count++] = next_index;
           }
           else
           {
             if (face.normalNumber())
               intersec = intersection(current_vertex,  next_vertex, normal, plane,
                                      normal_list[current_index], normal_list[next_index],
                                      new_normal);
             else
             {
               intersec = intersection(current_vertex, next_vertex, normal, plane);
             }
              
             if (intersec)
             {  
               if (vertex_count+1 >= new_face_indices->num) 
                   new_face_indices->allocValues(vertex_count+2);
               new_face_indices->values[vertex_count++] = 
                  face.faceAttr()->extendVertices(*intersec, new_normal);                 
               new_face_indices->values[vertex_count++] = next_index;
             }  
           }
         }
 
       } // if whichside == back

  } // for all vertices


  // all index lists end with -1
  if (vertex_count >= new_face_indices->num) 
     new_face_indices->allocValues(vertex_count+1);
  new_face_indices->values[vertex_count] = -1;   // terminator

  int* cut_new_face_indices = new int[vertex_count + 1];
  for (int i=0; i <= vertex_count; i++)
  {
      cut_new_face_indices[i] = new_face_indices->values[i];
      //cout << cut_new_face_indices[i] << " ";
  }
  //cerr << "new vertices: " << vertex_count << "\n";
  delete new_face_indices;

  
  // create the new face
 
  Face* result_face;

  if (face.normalNumber())
  {
     //cout << "with normals\n";
     result_face = new Face(vertex_count, cut_new_face_indices, vertex_count,
                            &normal_vector, face.faceAttr() );
  }
  else
  {
     //cout << "without normals\n";
     result_face = new Face(vertex_count, cut_new_face_indices, 0,
                            &normal_vector, face.faceAttr() );
  }
  
  // to make it easier (and usefull)
  // normal indices are the same like
  // vertex indices -- entries are stored in the same order 


// should never happen
if (vertex_count < 3)
{ 
cout << "************** vertex_count < 3 *****************\n";
face.print();
cout << "clip with plane:\n";
Clipping::print3D(normal);
cout << "d: " << plane << "\n";
if (whichside == 0) cout << "*Front face:\n";
if (whichside == 1) cout << "*Back face:\n";
result_face->print();
cout << "*************************************************\n\n";
return nil;
}

  return result_face;
}    
