/*
   Miscellaneous functions
   --------------------------------------------------------------------
   VCHE - Virtual Console Hex Editor

   Copyright (C) 1998, 1999 Diego Javier Grigna <diego@grigna.com>

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include "common.h"

struct fkey_text fkeys[ 3][ 11] = {
{ 
  { "HELP"  , 4}, { "EDIT"  , 4}, { "HEX  " , 5}, { "SEARCH", 6},
  { "SAGAIN", 6}, { "GOTO"  , 4}, { "FILES" , 5}, { "ATABLE", 6},
  { "GNUGPL", 6}, { "QUIT"  , 4}, { ""      , 0},
},
{ 
  { "HELP"  , 4}, { "EDIT"  , 4}, { "NOHEX" , 5}, { "SEARCH", 6},
  { "SAGAIN", 6}, { "GOTO"  , 4}, { "FILES" , 5}, { "ATABLE", 6},
  { "GNUGPL", 6}, { "QUIT"  , 4}, { ""      , 0},
},
{
  { " "     , 1}, { "SAVE"  , 4}, { "EDMASK", 6}, { "XOR"   , 3},
  { "OR"    , 2}, { "AND"   , 3}, { "NEG"   , 3}, { "ADD"   , 3},
  { "SUB"   , 3}, { "CANCEL", 6}, { ""      , 0},
},
};

char *nfk[] = { " 1", " 2", " 3", " 4", " 5", " 6", " 7", " 8", " 9", "10", 0};

/* This variabled will be allocated (during initialization) with the size
   of the screen columns, and will be filled with spaces ' ' */
char *space_buffer;

static void print_long( long value, int prec);

struct vc_flags flags;
char *trans_buffer;

void do_exit( int status)
{
#if defined(VCHE_VC)
 term_color( VC_COLOR_OFF);
 term_unhide_cursor();
 term_noraw();
 term_goto( 0, vc_lines);
#elif defined(VCHE_RAW)
 term_color( VC_COLOR_OFF);
 term_disable_graf_map();
 term_unhide_cursor();
 term_noraw();
 /* fix submitted by Ellen Geertsema (ellen@euronet.nl), 07-nov-1999 */
 term_goto( 0, vc_lines);
#elif defined(VCHE_NC)
 term_unhide_cursor();
 endwin();
#endif
 /* next line added by Ellen Geertsema (ellen@euronet.nl), 07-nov-1999 */
 putchar('\n');
 exit( status);
}

void draw_filename( char *filename)
{
 char *ptr;
 int len;
 int mlen;

 ptr = rindex( filename, '/');

 if( ptr == NULL)
     ptr = filename;
 else
     ptr++;

 len  = strlen( ptr);

 mlen = vc_cols - 26;

 lib_cputs( 0, 0, mlen, space_buffer, VC_COLOR_BACKCYAN);

 if( len > mlen) {
#if defined(VCHE_VC)
     vc_color_putc( mlen - 1, 0, '\x1A', VC_COLOR_ATTENTION);
#elif defined(VCHE_RAW)
     term_color( VC_COLOR_ATTENTION);
     term_enable_graf_map();
     raw_putc( mlen - 1, 0, RAW_RARROW);
     term_disable_graf_map();
     term_color( VC_COLOR_BACKCYAN);
#elif defined(VCHE_NC)
     nc_color_putc( mlen - 1, 0, ACS_RARROW, VC_COLOR_ATTENTION);
#endif
     len = mlen - 1;
 }

 lib_cputs( 0, 0, len, ptr, VC_COLOR_BACKCYAN);
}

void draw_main_screen( void)
{
#if defined(VCHE_VC)
 vc_fill_screen( VC_COLOR_BACKBLUE);
 vc_color_puts( 0, 0, vc_cols, space_buffer, VC_COLOR_BACKCYAN);
 vc_puts( vc_cols - 26, 0, 8, "WXMEB0");
 vc_putc( vc_cols - 5 , 0, '');
 vc_putc( vc_cols - 1 , 0, '%');
#elif defined(VCHE_RAW)
 term_color( VC_COLOR_BACKCYAN);
 raw_puts( 0, 0, vc_cols, space_buffer);
 term_enable_graf_map();
 raw_putc( vc_cols - 26, 0, RAW_LINE_VERT);
 term_disable_graf_map();
 raw_puts( vc_cols - 25, 0, 6, "WXMEB0"  );
 term_enable_graf_map();
 raw_putc( vc_cols - 19, 0, RAW_LINE_VERT);
 raw_putc( vc_cols - 5 , 0, RAW_LINE_VERT);
 term_disable_graf_map();
 raw_putc( vc_cols - 1 , 0, '%');
#elif defined(VCHE_NC)
 nc_fill_screen( VC_COLOR_CYANBLUE);
 nc_color_puts( 0, 0, vc_cols, space_buffer, VC_COLOR_BACKCYAN);
 nc_color_putc( vc_cols - 26, 0, ACS_VLINE, VC_COLOR_BACKCYAN);
 nc_color_puts( vc_cols - 25, 0, 6, "WXMEB0", VC_COLOR_BACKCYAN);
 nc_color_putc( vc_cols - 19, 0, ACS_VLINE, VC_COLOR_BACKCYAN);
 nc_color_putc( vc_cols - 5 , 0, ACS_VLINE, VC_COLOR_BACKCYAN);
 nc_color_putc( vc_cols - 1 , 0, '%', VC_COLOR_BACKCYAN);
#endif

 draw_fkeys( fkeys[ flags.nohex]);
}

void draw_fkeys( struct fkey_text fkt[])
{
 int sp, x, y, i;

 /*
  * It also works on different resolutions, 80x25, 132x60 and
  * any 80xANY modes.
  */

 sp = ( vc_cols / 10) - 8;
 y  = vc_lines - 1;

 lib_cputs( 0, vc_lines - 1, vc_cols, space_buffer, VC_COLOR_BACKCYAN);

 i = 0;
 x = 2;

 do {
     lib_cputs( x, y, fkt[ i].size, fkt[ i].text, VC_COLOR_BACKCYAN); 
     x += 8 + sp;
     i++;
 } while( fkt[ i].size != 0);

 i = 0;
 x = 0;
 do {
     lib_cputs( x, y, 2, nfk[ i], VC_COLOR_LIGHTGRAY);
     x += 8 + sp;
     i++;
 } while( nfk[ i]);
}

void draw_offset_n_perc( long offset)
{
 double d_offset;
 double d_filesize;
 double d_percent;

 /* I need to use "double" because the multiplication
    could overflow the "long" arguments if the file
    is too big. */

 d_offset   = offset;
 d_filesize = vc_filesize;

 if( vc_filesize == 0)
     d_percent = 0;
 else
     d_percent  = ( d_offset * 100) / d_filesize;

 print_long( ( long) d_percent, 3);
 lib_cputs( vc_cols -  4, 0,  3, trans_buffer, VC_COLOR_BACKCYAN);

 print_long( offset, 13);
 lib_cputs( vc_cols - 18, 0, 13, trans_buffer, VC_COLOR_BACKCYAN);
}

void print_long( long value, int prec)
{
 register char *bp = trans_buffer + prec;

 do {
     *--bp = hex_trans[ value % 10];
 } while( (value /= 10) != 0);

 /* space padding */
 while( bp != trans_buffer)
        *--bp = ' ';
}


void set_flag( int flag)
{
 int xsflag = vc_cols - 25;

 switch( flag) {
#if defined(VCHE_VC)
         case FLAG_ZERO: if( flags.zero)
                             lib_cputc( xsflag + 5, 0, '0' , VC_COLOR_ATTENTION);
                         else
                             lib_cputc( xsflag + 5, 0, '0' , VC_COLOR_BACKCYAN);
                         break;
         case FLAG_MASK: if( flags.mask)
                             lib_cputc( xsflag + 2, 0, 'M' , VC_COLOR_ATTENTION);
                         else
                             lib_cputc( xsflag + 2, 0, 'M' , VC_COLOR_BACKCYAN);
                         break;
#endif
         case FLAG_BELL: if( flags.bell)
                             lib_cputc( xsflag + 4, 0, 'B' , VC_COLOR_ATTENTION);
                         else
                             lib_cputc( xsflag + 4, 0, 'B' , VC_COLOR_BACKCYAN);
                         break;
         case FLAG_EDIT: if( flags.edit)
                             lib_cputc( xsflag + 3, 0, 'E' , VC_COLOR_ATTENTION);
                         else
                             lib_cputc( xsflag + 3, 0, 'E' , VC_COLOR_BACKCYAN);
                         break;
         case FLAG_NOHEX:
                         if( flags.nohex)
                             lib_cputc( xsflag + 1, 0, 'X' , VC_COLOR_ATTENTION);
                         else
                             lib_cputc( xsflag + 1, 0, 'X' , VC_COLOR_BACKCYAN);
                         break;
         case FLAG_READONLY:
                         if( flags.readonly)
                             lib_cputc( xsflag, 0, 'W' , VC_COLOR_ATTENTION);
                         else
                             lib_cputc( xsflag, 0, 'W' , VC_COLOR_BACKCYAN);
                         break;
 }

}

void *allocate_mem( int size)
{
 void *voidptr = NULL;

 voidptr = calloc( size + 2, sizeof( char));

 if( voidptr == NULL) {
     fprintf( stderr, "\nNot enough memory\n\n");
     do_exit( 1);
 }

 return voidptr;
}

char *trim_spaces( char *s)
{
 static char a[ 8192];
 char *save = s;
 char *start, *end;

 start = s;
 end = strchr( s, 0);

 if( *end == '\n' || *end == '\r') *end = 0;

 while( end >= start && *end <= ' ') end--;
 *++end = 0;

 while( end >= start && *start <= ' ') start++;
 save = start;

 memset( &a, 0, sizeof( a));
 strcpy( a, save);

 return a;
}

void set_hexmode_variables( void)
{
 bytes_per_row = vc_cols - 12;

 if( !flags.nohex) {
     bytes_per_row     = bytes_per_row * 4 / 17;
     bytes_per_row_hex = bytes_per_row * 13 / 4;
 }

 bytes_to_read = bytes_per_row * (vc_lines - 2);
}

int check_readonly( void)
{
 int i;

 if( !flags.readonly) return 1;

 term_beep();

 i = lib_ask( "We are in READ ONLY mode do you want to set READ/WRITE mode?", " Yes ", " No ", " Cancel ", 1);

 term_hide_cursor();

 if( i == 0) return 1;

 return -1;
}

#if defined(VCHE_NC)

void translate_buffer_colors( int what_attr, attr_t *attr, short *color)
{
 switch( what_attr) {
         case BUF_BBLUE : *attr  = A_BOLD;
                          *color = 25;
                          break;
         case BUF_BLGRAY: *attr  = A_NORMAL;
                          *color = 26;
                          break;
         case BUF_WHITE : *attr  = A_BOLD;
                          *color = 21;
                          break;
 }
}

#endif

