"""
This contains the main class which is instantiated to handle
options.
"""

import copy

class Choices(object):
    def __init__(self):
        self.loaders = []
        self.options = {}

    def define(self, name, **kwargs):
        """
        Defines an option. Options can take various additional arguments
        to define their behavior as well as potential help strings and
        so on. The exact options available depends on the option class
        used. By default, this is :py:class:`Option <choices.choices.Option>`.
        """
        self.options[name] = Option(name, **kwargs)

    def add_loader(self, instance):
        """
        Adds a loader to the sequence of loaders. The loaders are responsible
        for loading in configuration. Multiple loaders can be added and
        they will be called in sequence. Loaders added later will overwrite
        settings set earlier. This allows you to easily set precedence of
        configuration sources. For example: defaults => file => command line,
        where settings defined on the command line override file options which
        then override the default settings.
        """
        self.loaders.append(instance)

    def load(self, **kwargs):
        """
        Loads the configured option from the various loaders. This will
        return an options dictionary. Any additional keyword arguments
        will be passed into the ``load`` method for the various loaders.
        """
        settings = {}

        # Load all settings from the loaders
        for loader in self.loaders:
            for (key, value) in loader.load(self.options, dict(settings), **kwargs).iteritems():
                settings[key] = value

        for (key, option) in self.options.iteritems():
            if key in settings:
                value = settings[key]
            else:
                value = Option.UNSET_VALUE

            # Ask the option to filter the value. This allows options
            # to do post-processing just before the load is complete.
            settings[key] = option.value(value)

        return ChoiceResult(settings)

class Option(object):
    UNSET_VALUE = object()

    def __init__(self, name, type=None, help=None, default=None):
        """
        Represents a single option. This class is usually instantiated by
        calling :py:func:`define <choices.choices.Choices.define>`, and
        shouldn't be instantiated directly.

        :Parameters:
          - `name`: The name of the option.
          - `option_type` (optional): The type to coerce this option into.
          - `help` (optional): A help string to associate with the option.
          - `default` (optional): A default value to be associated with this
            option.

        """
        self.name = name
        self.option_type = type
        self.help = help
        self.default = default

    def value(self, value):
        """
        This is called to allow options to post-process values just
        before they're returned from a load.
        """
        if value is self.UNSET_VALUE:
            # The value was not specified and we have a default,
            # so we use it.
            value = copy.deepcopy(self.default)
        elif self.option_type is not None and value is not None:
            if self.option_type == bool and isinstance(value, str):
                # Handle booleans in a special way so that certain strings
                # will properly parse to false/true
                truthy_values = ["1", "true", "yes"]
                return value.lower() in truthy_values
            else:
                value = self.option_type(value)

        return value

class ChoiceResult(dict):
    """
    This is a custom dict subclass that overrides ``__getattribute__``
    so that options can be accessed using simple dot-syntax:

        options.foo
        options.bar
    """

    def __getattribute__(self, name):
        if name in self:
            return self[name]

        return super(ChoiceResult, self).__getattribute__(name)
