/*   EXTRAITS DE LA LICENCE
	Copyright CEA, contributeurs : Luc BILLARD et Damien
	CALISTE, laboratoire L_Sim, (2001-2005)
  
	Adresse ml :
	BILLARD, non joignable par ml ;
	CALISTE, damien P caliste AT cea P fr.

	Ce logiciel est un programme informatique servant  visualiser des
	structures atomiques dans un rendu pseudo-3D. 

	Ce logiciel est rgi par la licence CeCILL soumise au droit franais et
	respectant les principes de diffusion des logiciels libres. Vous pouvez
	utiliser, modifier et/ou redistribuer ce programme sous les conditions
	de la licence CeCILL telle que diffuse par le CEA, le CNRS et l'INRIA 
	sur le site "http://www.cecill.info".

	Le fait que vous puissiez accder  cet en-tte signifie que vous avez 
	pris connaissance de la licence CeCILL, et que vous en avez accept les
	termes (cf. le fichier Documentation/licence.fr.txt fourni avec ce logiciel).
*/

/*   LICENCE SUM UP
	Copyright CEA, contributors : Luc BILLARD et Damien
	CALISTE, laboratoire L_Sim, (2001-2005)

	E-mail address:
	BILLARD, not reachable any more ;
	CALISTE, damien P caliste AT cea P fr.

	This software is a computer program whose purpose is to visualize atomic
	configurations in 3D.

	This software is governed by the CeCILL  license under French law and
	abiding by the rules of distribution of free software.  You can  use, 
	modify and/ or redistribute the software under the terms of the CeCILL
	license as circulated by CEA, CNRS and INRIA at the following URL
	"http://www.cecill.info". 

	The fact that you are presently reading this means that you have had
	knowledge of the CeCILL license and that you accept its terms. You can
	find a copy of this licence shipped with this software at Documentation/licence.en.txt.
*/

#include "view.h"

#include <GL/gl.h>
#include <GL/glu.h> 

#include <math.h>

#include <visu_tools.h>
#include <visu_configFile.h>
#include <visu_data.h>
#include <coreTools/toolMatrix.h>
#include <coreTools/toolConfigFile.h>

/**
 * SECTION:view
 * @short_description: Defines all necessary informations for the
 * rendering of a view.
 *
 * <para>The #VisuOpenGLView stores three basic informations: one for the
 * position and orientation of the camera (#VisuOpenGLCamera), one for the
 * description of the bounding box in the OpenGL coordinates
 * (#OpenGLBox, should be moved elsewhere later) and one last for the
 * definition of the viewing window (#OpenGLWindow, including volumic
 * informations).</para>
 *
 * <para>One resource is used by this part, defining the precision
 * desired by the user when drawing OpenGL objects. This precision can
 * be changed using VisuOpenGLViewSet_precision() and all V_Sim part
 * drawing something should use VisuOpenGLCameraGet_numberOfFacettes() to
 * know the size of the vertices to be drawn depending on this
 * precision and the level of zoom.</para>
 *
 * <para>The rendering is done in an OpenGl viewport whose size is
 * given by the bounding box (plus 10%). The camera can be positionned
 * with three angles (theta, phi and omega) and has a zoom factor
 * (gross) and a perspective value (d_red). The angle theta is around
 * the z axis (box coordinates), phi is around the new x axis (after
 * the theta rotation) and omega is a rotation around the axis which
 * goes from the observer to the center of the bounding box. By
 * default the camera looks at the center of the bounding box but this
 * can be changed with the Xs and Ys parameters. These values are
 * stored and are readable through the #VisuOpenGLCamera structure. They
 * must be changed with the following methods :
 * openGLViewSet_thetaPhiOmega(), openGLViewSet_gross(),
 * openGLViewSet_persp() and openGLViewSet_XsYs().</para>
 */

/* Global value used for precision. */
static float precisionOfRendering;
/* Global value used as default when creating a new VisuOpenGLView. */
static float anglesDefault[3] = {40., -50., 0.};
static float translatDefault[2] = {0.5, 0.5};
static float grossDefault = 1.;
static float perspDefault = 5.;

/**
 * openGLCameraSet_thetaPhiOmega:
 * @camera: a valid #VisuOpenGLCamera object ;
 * @valueTheta: a floatinf point value in degrees ;
 * @valuePhi: a floating point value in degrees ;
 * @valueOmega: a floating point value in degrees ;
 * @mask: to specified what values will be changed.
 *
 * Change the orientation of the camera to the specified angles.
 *
 * Returns: TRUE if the signal OpenGLAskForReDraw should be emitted.
 */
gboolean openGLCameraSet_thetaPhiOmega(VisuOpenGLCamera *camera, float valueTheta,
				       float valuePhi, float valueOmega, int mask)
{
  float valT, valP, valO;
  int idem;

  g_return_val_if_fail(camera, FALSE);
  
  idem = 1;
  if (mask & VISU_CAMERA_THETA)
    {
      valT = valueTheta;
      while (valT < -180.)
	valT += 360.;
      while (valT > 180.)
	valT -= 360.;

      if (camera->theta != valT)
	{
	  idem = 0;
	  camera->theta = valT;
	}
    }
  if (mask & VISU_CAMERA_PHI)
    {
      valP = valuePhi;
      while (valP < -180.)
	valP += 360.;
      while (valP > 180.)
	valP -= 360.;

      if (camera->phi != valP)
	{
	  idem = 0;
	  camera->phi = valP;
	}
    }
  if (mask & VISU_CAMERA_OMEGA)
    {
      valO = valueOmega;
      while (valO < -180.)
	valO += 360.;
      while (valO > 180.)
	valO -= 360.;

      if (camera->omega != valO)
	{
	  idem = 0;
	  camera->omega = valO;
	}
    }
  if (idem)
    return FALSE;

/*   modelize(view); */
/*   g_signal_emit (visu, VISU_GET_CLASS (visu)->OpenGLThetaPhi_signal_id, */
/* 		 0 , NULL); */

  return TRUE;
}

/**
 * openGLCameraSet_XsYs:
 * @camera: a valid #VisuOpenGLCamera object ;
 * @valueX: a floatinf point value in the bounding box scale
 *          (1 is the size of the bounding box) ;
 * @valueY: a floating point value in bounding box scale ;
 * @mask: to specified what values will be changed.
 *
 * Change the point where the camera is pointed to.
 *
 * Returns: TRUE if the signal OpenGLAskForReDraw should be emitted.
 */
gboolean openGLCameraSet_XsYs(VisuOpenGLCamera *camera,
			      float valueX, float valueY, int mask)
{
  float valX, valY, dX, dY/* , z */;
/*   float O[3]; */
  
  g_return_val_if_fail(camera, FALSE);
  
  dX = 0.f;
  dY = 0.f;
  if (mask & MASK_XS)
    {
      valX = valueX;
      if (valX < -3.)
	valX = -3.;
      if (valX > 3.)
	valX = 3.;

      if (camera->xs != valX)
	{
	  dX = valX - camera->xs;
	  camera->xs = valX;
	}
    }
  if (mask & MASK_YS)
    {
      valY = valueY;
      if (valY < -3.)
	valY = -3.;
      if (valY > 3.)
	valY = 3.;

      if (camera->ys != valY)
	{
	  dY = valY - camera->ys;
	  camera->ys = valY;
	}
    }
  if (dX == 0.f && dY == 0.f)
    return FALSE;

/*   g_signal_emit (visu, VISU_GET_CLASS (visu)->OpenGLXsYs_signal_id, */
/* 		 0 , NULL); */

/*   project(view); */

  /* Change the eyes position. */
/*   O[0] = view->camera->centre[0] + view->box->dxxs2; */
/*   O[1] = view->camera->centre[1] + view->box->dyys2; */
/*   O[2] = view->camera->centre[2] + view->box->dzzs2; */
/*   z = openGLViewGet_zCoordinate(view, O); */
/*   DBG_fprintf(stderr, "OpenGL View: new window coordinates (%f;%f;%f).\n", */
/* 	      0.5f * view->window->width, 0.5f * view->window->height, z); */
/*   openGLViewGet_realCoordinates(view, view->camera->centre, */
/* 				0.5f * view->window->width, */
/* 				0.5f * view->window->height, z, FALSE); */
/*   modelize(view); */

  return TRUE;
}

/**
 * openGLCameraSet_gross:
 * @camera: a valid #VisuOpenGLCamera object ;
 * @value: a positive floating point value.
 *
 * Change the value of the camera zoom value. If the value is higher than 10
 * it is set to 10 and if the value is negative it is set to 0.001.
 *
 * Returns: TRUE if the signal OpenGLAskForReDraw should be emitted.
 */
gboolean openGLCameraSet_gross(VisuOpenGLCamera *camera, float value)
{
  float val;
  
  g_return_val_if_fail(camera, FALSE);
  
  val = value;
  if (val < 0.02)
    val = 0.02;
  else if (val > 999.)
    val = 999.;

  if (camera->gross == val)
    return FALSE;

  camera->gross = val;
/*   g_signal_emit (visu, VISU_GET_CLASS (visu)->OpenGLGross_signal_id, */
/* 		 0 , NULL); */
/*   project(view); */
  
/*   g_signal_emit (visu, VISU_GET_CLASS (visu)->OpenGLFacetteChanged_signal_id, */
/* 		 0 , NULL); */
  return TRUE;
}

/**
 * openGLCameraSet_persp:
 * @camera: a valid #VisuOpenGLCamera object ;
 * @value: a floating point value greater than 1.1.
 *
 * Change the value of the camera perspective value and put it in
 * bounds if needed.
 *
 * Returns: TRUE if the signal OpenGLAskForReDraw should be emitted.
 */
gboolean openGLCameraSet_persp(VisuOpenGLCamera *camera, float value)
{
  g_return_val_if_fail(camera, FALSE);
  
  value = CLAMP(value, 1.1f, 100.f);
  if (camera->d_red == value)
    return 0;

  camera->d_red = value;

/*   project(view); */
/*   modelize(view); */
  return TRUE;
}
/**
 * openGLWindowSet_viewport:
 * @window: a valid #OpenGLWindow object ;
 * @width: the new horizontal size ;
 * @height: the new vertical size.
 *
 * It changes the size of the OpenGl area and reccompute the OpenGL viewport.
 *
 * Returns: TRUE the size of @window is actually changed.
 */
gboolean openGLWindowSet_viewport(OpenGLWindow *window, guint width, guint height)
{
  DBG_fprintf(stderr, "OpenGL Window: set viewport size (%dx%d) for window %p.\n",
	      width, height, (gpointer)window);
  
  g_return_val_if_fail(window, FALSE);

  if (window->width == width && window->height == height)
    return FALSE;

  DBG_fprintf(stderr, " | old values were %dx%d.\n", window->width,
	      window->height);
  window->width = width;
  window->height = height;
  glViewport(0, 0, window->width, window->height);
  return TRUE;
}

/**
 * openGLViewSet_upAxis:
 * @view: a #VisuOpenGLView object.
 * @upAxis: a direction.
 *
 * In constraint observation mode, the "north" direction is a singular
 * one. Define this direction with this routine.
 *
 * Since: 3.6
 */
void openGLViewSet_upAxis(VisuOpenGLView *view, ViewAxis upAxis)
{
  g_return_if_fail(view && view->camera);

  view->camera->upAxis = upAxis;
}

/**
 * openGLCameraSet_refLength:
 * @camera: a #VisuOpenGLCamera object.
 * @value: a new length.
 * @unit: its measurement unit.
 *
 * Change the reference value that is used for the zoom.
 *
 * Since: 3.6
 *
 * Returns: TRUE if the value is indeed changed.
 */
gboolean openGLCameraSet_refLength(VisuOpenGLCamera *camera, float value, ToolUnits unit)
{
  g_return_val_if_fail(camera, FALSE);

  if (camera->length0 == value && camera->unit == unit)
    return FALSE;

  camera->length0 = value;
  camera->unit    = unit;
  return TRUE;
}
/**
 * openGLCameraGet_refLength:
 * @camera: a #VisuOpenGLCamera object.
 * @unit: a location for unit value (can be NULL).
 *
 * The zoom is define from a reference length in given unit. If @unit
 * is provided, the corresponding unit will be set.
 *
 * Since: 3.6
 *
 * Returns: the current reference length.
 */
float openGLCameraGet_refLength(VisuOpenGLCamera *camera, ToolUnits *unit)
{
  g_return_val_if_fail(camera, -1.f);

  if (unit)
    *unit = camera->unit;
  return camera->length0;
}

/**
 * openGLModelize:
 * @camera: a #VisuOpenGLCamera object.
 *
 * Set-up the orientation matrix, depending on the camera definition.
 */
void openGLModelize(VisuOpenGLCamera *camera)
{
  double theta_rad, d_red;
  double phi_rad; 
  double sth, cth, sph, cph, com, som;
  double distance;
  int permut[3][3] = {{1,2,0}, {2,0,1}, {0,1,2}};

  g_return_if_fail(camera);
 
  DBG_fprintf(stderr, "OpenGL Camera: modelize view.\n");
  DBG_fprintf(stderr, "OpenGL Camera: using ref length %g.\n",
	      camera->length0);

  if (camera->d_red > 100.)
    d_red = 100.;
  else
    d_red = camera->d_red;
    theta_rad = camera->theta * PI180;
  phi_rad   = camera->phi   * PI180; 

  distance = d_red * camera->length0;

  sth = sin(theta_rad);
  cth = cos(theta_rad);      
  sph = sin(phi_rad);
  cph = cos(phi_rad);
  com = cos(camera->omega * PI180);
  som = sin(camera->omega * PI180);

  /* La matrice de rotation est la suivante pour passer
     des coordonnes transformes aux coordonnes de l'cran :
     /cph.cth -sph cph.sth\
     |sph.cth  cph sph.sth| (for z as north axis)
     \   -sth   0      cth/

     / cph -sph.sth sph.cth\
     |  0       cth     sth| (for y as north axis)
     \-sph -sth.cph cph.cth/
     Ainsi la camra qui est situ en (0,0,Distance) dans le repre transform
     devient dans le repre de l'cran : */
  camera->eye[permut[camera->upAxis][0]] = distance*sth*cph;
  camera->eye[permut[camera->upAxis][1]] = distance*sth*sph;
  camera->eye[permut[camera->upAxis][2]] = distance*cth;
   
  /* Vecteur donnant la direction verticale.
     Dans le repre transform il est (-1,0,0). */
  camera->up[permut[camera->upAxis][0]] = -cth*cph*com + sph*som;
  camera->up[permut[camera->upAxis][1]] = -cth*sph*com - cph*som;
  camera->up[permut[camera->upAxis][2]] = sth*com;

  glMatrixMode(GL_MODELVIEW); 
  glLoadIdentity();
  gluLookAt(camera->eye[0], camera->eye[1], camera->eye[2],
	    camera->centre[0], camera->centre[1], camera->centre[2],
	    camera->up[0], camera->up[1], camera->up[2]);
}

/**
 * openGLProject:
 * @window: definition of the screen.
 * @camera: position of the camera.
 * @extens: expansion of the object.
 *
 * This method is used to set the projection and the OpenGL viewport.
 */
void openGLProject(OpenGLWindow *window, const VisuOpenGLCamera *camera, float extens)
{ 
  double x, y, xmin, xmax, ymin, ymax, fact, rap;
  double rap_win, d_red;

  g_return_if_fail(camera && extens > 0.f && window);

  DBG_fprintf(stderr, "OpenGL View: project view.\n");

  if (camera->d_red > 100.)
    d_red = 100.;
  else
    d_red = camera->d_red;
  
  fact = d_red * camera->length0;
  window->near = MAX(0.01, fact - extens);
  window->far  = fact + extens;

  fact = window->near / camera->gross / d_red;
  rap = 2. * window->near / (d_red - 1.);
  x = (0.5 - camera->xs) * rap;
  xmin = x - fact;
  xmax = x + fact;
  y = (0.5 - camera->ys) * rap;
  ymin = y - fact;
  ymax = y + fact;
  window->left   = xmin;
  window->bottom = ymin;
    
  rap_win = (1.0*window->height)/window->width;
  if ( 1. > rap_win )
    {
      window->top   = ymax;
      fact                = (ymax - ymin) / rap_win;
      window->left  = 0.5 * (xmin + xmax - fact);
      window->right = 0.5 * (xmin + xmax + fact);
    }
  else if ( 1. < rap_win )
    {
      window->right  = xmax;
      fact                 = (xmax - xmin) * rap_win;
      window->bottom = 0.5 * (ymin + ymax - fact);
      window->top    = 0.5 * (ymin + ymax + fact);
    }
  else
    {
      window->right  = xmax;
      window->top    = ymax;
    }

  glMatrixMode(GL_PROJECTION);
  glLoadIdentity();
  if (d_red == 100.)
    glOrtho(window->left, window->right, window->bottom,
	    window->top, window->near, window->far);
  else
    glFrustum(window->left, window->right, window->bottom,
	      window->top, window->near, window->far);
  glMatrixMode(GL_MODELVIEW);
}

VisuOpenGLView* VisuOpenGLViewNew(void)
{
  VisuOpenGLView *view;

  view         = g_malloc(sizeof(VisuOpenGLView));
  view->camera = g_malloc(sizeof(VisuOpenGLCamera));
  view->window = g_malloc(sizeof(OpenGLWindow));

  view->window->width  = 100.;
  view->window->height = 100.;
  view->window->near   = 0.;
  view->window->far    = 0.;

  view->camera->theta   = anglesDefault[0];
  view->camera->phi     = anglesDefault[1];
  view->camera->omega   = anglesDefault[2];
  view->camera->xs      = translatDefault[0];
  view->camera->ys      = translatDefault[1];
  view->camera->gross   = grossDefault;
  view->camera->d_red   = perspDefault;
  view->camera->length0 = -1.;
  view->camera->unit    = unit_undefined;
  view->camera->upAxis  = VIEW_Z;
  view->camera->centre[0] = 0.f;
  view->camera->centre[1] = 0.f;
  view->camera->centre[2] = 0.f;

  DBG_fprintf(stderr, "OpenGL view : create a new view (%p).\n", (gpointer)view);
  DBG_fprintf(stderr, " | theta = %g\n", view->camera->theta);
  DBG_fprintf(stderr, " | phi   = %g\n", view->camera->phi);
  DBG_fprintf(stderr, " | omega = %g\n", view->camera->omega);
  DBG_fprintf(stderr, " | dx-dy = %g %g\n", view->camera->xs, view->camera->ys);
  DBG_fprintf(stderr, " | gross = %g\n", view->camera->gross);
  DBG_fprintf(stderr, " | persp = %g\n", view->camera->d_red);
  DBG_fprintf(stderr, " | width x height = %d x %d\n", view->window->width, view->window->height);
  return view;
}
void VisuOpenGLViewFree(VisuOpenGLView *view)
{
  g_return_if_fail(view);

  if (view->camera)
    g_free(view->camera);
  if (view->window)
    g_free(view->window);
  g_free(view);
}
VisuOpenGLView* VisuOpenGLViewCopy(VisuOpenGLView *view)
{
  VisuOpenGLView *viewCopy;

  g_return_val_if_fail(view, (VisuOpenGLView*)0);

  viewCopy         = g_malloc(sizeof(VisuOpenGLView));
  viewCopy->camera = g_malloc(sizeof(VisuOpenGLCamera));
  viewCopy->window = g_malloc(sizeof(OpenGLWindow));

  viewCopy->window->width  = view->window->width;
  viewCopy->window->height = view->window->height;
  viewCopy->window->near   = view->window->near;
  viewCopy->window->far    = view->window->far;
  viewCopy->window->left   = view->window->left;
  viewCopy->window->right  = view->window->right;
  viewCopy->window->bottom = view->window->bottom;
  viewCopy->window->top    = view->window->top;

  viewCopy->camera->theta   = view->camera->theta;
  viewCopy->camera->phi     = view->camera->phi;
  viewCopy->camera->omega   = view->camera->omega;
  viewCopy->camera->xs      = view->camera->xs;
  viewCopy->camera->ys      = view->camera->ys;
  viewCopy->camera->gross   = view->camera->gross;
  viewCopy->camera->d_red   = view->camera->d_red;
  viewCopy->camera->length0 = view->camera->length0;
  viewCopy->camera->unit    = view->camera->unit;
  viewCopy->camera->upAxis  = view->camera->upAxis;
  viewCopy->camera->centre[0] = view->camera->centre[0];
  viewCopy->camera->centre[1] = view->camera->centre[1];
  viewCopy->camera->centre[2] = view->camera->centre[2];
  
  return viewCopy;
}

/* This is a function to get the number of facettes advised
   by the server (according to its policy on rendering)
   to draw an object according to a given dimension. */
gint VisuOpenGLCameraGet_numberOfFacettes(VisuOpenGLCamera *camera, guint winSize,
                                          float dimension)
{
  gint rsize;     
  gint nlat;
#define NLAT_MIN 12
#define NLAT_MAX 50
#define RSIZE_MIN  10
#define RSIZE_MAX 250

#define NLAT_V_MIN 0
#define NLAT_V_MAX (NLAT_MIN)
#define RSIZE_V_MIN  0
#define RSIZE_V_MAX (RSIZE_MIN)

#define NLAT_MINI 3
#define NLAT_MAXI 100

  static float fac = -1.0f, fac_v = -1.0f;

  g_return_val_if_fail(camera, -1);

/*   if (!visuCamera->gr || !visuCamera->gross) */
/*     return -1; */

  /* calculate once fac and fac_v!... */
  if(fac < 0.0f) {
    fac = ((float)(NLAT_MAX - NLAT_MIN))/(RSIZE_MAX - RSIZE_MIN);
    fac_v = ((float)(NLAT_V_MAX - NLAT_V_MIN))/(RSIZE_V_MAX - RSIZE_V_MIN);
  }

  rsize = (int)((float)winSize *
		(0.5 * dimension / camera->length0 * camera->gross *
		 camera->d_red / (camera->d_red - 1.)));
  
  if(rsize < RSIZE_MIN) {
    nlat = (int)(NLAT_V_MIN + fac_v * (rsize - RSIZE_V_MIN));
    if(nlat < NLAT_MINI) nlat = NLAT_MINI;
  }
  else if(rsize > RSIZE_MAX) {
    nlat = NLAT_MAX;
  }
  else {
    nlat = (int)(NLAT_MIN + fac * (rsize - RSIZE_MIN));
  }

  nlat = (int)((float)nlat * precisionOfRendering);
  if (nlat < NLAT_MINI)
    nlat = NLAT_MINI;
  if (nlat > NLAT_MAXI)
    nlat = NLAT_MAXI;
   
  return nlat;
}
/* This function change the value of the parameter precisionOfRendering. */
int VisuOpenGLViewSet_precision(float value)
{
  if (value <= 0. || value == precisionOfRendering)
    return 0;
  
  precisionOfRendering = value;

/*   g_signal_emit (visu, VISU_GET_CLASS (visu)->OpenGLFacetteChanged_signal_id, */
/* 		 0 , NULL); */
  return 1;
}
/* This function retrieve the value of the parameter precisionOfRendering. */
float VisuOpenGLViewGet_precision(void)
{
  return precisionOfRendering;
}
float VisuOpenGLViewGet_fileUnitPerPixel(VisuOpenGLView *view)
{
  float deltaH, deltaV;

  g_return_val_if_fail(view, 0.);

  deltaH = view->window->right - view->window->left;
  deltaV = view->window->top - view->window->bottom;
  if (deltaH < deltaV)
    return deltaH / (float)view->window->width;
  else
    return deltaV / (float)view->window->height;
}

void VisuOpenGLViewGet_screenAxes(VisuOpenGLView *view, float xAxis[3], float yAxis[3])
{
  double cth, sth, cph, sph, com, som;
  float matPhi[3][3], matTheta[3][3], matOmega[3][3];
  float matRes[3][3], matRes2[3][3];
  float axis[3];

  g_return_if_fail(view);

  cth = cos(view->camera->theta * PI180);
  sth = sin(view->camera->theta * PI180);
  cph = cos(view->camera->phi * PI180);
  sph = sin(view->camera->phi * PI180);
  com = cos(view->camera->omega * PI180);
  som = sin(view->camera->omega * PI180);

  matPhi[0][0] = cph;
  matPhi[1][0] = sph;
  matPhi[2][0] = 0.;
  matPhi[0][1] = -sph;
  matPhi[1][1] = cph;
  matPhi[2][1] = 0.;
  matPhi[0][2] = 0.;
  matPhi[1][2] = 0.;
  matPhi[2][2] = 1.;

  matTheta[0][0] = cth;
  matTheta[1][0] = 0.;
  matTheta[2][0] = -sth;
  matTheta[0][1] = 0.;
  matTheta[1][1] = 1.;
  matTheta[2][1] = 0.;
  matTheta[0][2] = sth;
  matTheta[1][2] = 0.;
  matTheta[2][2] = cth;

  matOmega[0][0] = com;
  matOmega[1][0] = som;
  matOmega[2][0] = 0.;
  matOmega[0][1] = -som;
  matOmega[1][1] = com;
  matOmega[2][1] = 0.;
  matOmega[0][2] = 0.;
  matOmega[1][2] = 0.;
  matOmega[2][2] = 1.;

  tool_matrix_productMatrix(matRes, matTheta, matOmega);
  tool_matrix_productMatrix(matRes2, matPhi, matRes);

  axis[0] = 0.;
  axis[1] = 1.;
  axis[2] = 0.;
  tool_matrix_productVector(xAxis, matRes2, axis);

  axis[0] = -1.;
  axis[1] = 0.;
  axis[2] = 0.;
  tool_matrix_productVector(yAxis, matRes2, axis);
}
/**
 * openGLViewGet_zCoordinate:
 * @view: a #VisuOpenGLView object.
 * @xyz: a cartesian point.
 *
 * Use this routine to know the Z value of a real point defined by
 * @xyz in caretsian coordinates.
 */
float openGLViewGet_zCoordinate(VisuOpenGLView *view, float xyz[3])
{
  GLdouble model[16], project[16];
  GLint viewport[4];
  GLdouble xyzGL[3], winGL[3];

  g_return_val_if_fail(view, 0.5f);

  glGetDoublev(GL_MODELVIEW_MATRIX, model);
  glGetDoublev(GL_PROJECTION_MATRIX, project);
  glGetIntegerv(GL_VIEWPORT, viewport);
  
  xyzGL[0] = (GLdouble)xyz[0];
  xyzGL[1] = (GLdouble)xyz[1];
  xyzGL[2] = (GLdouble)xyz[2];
  gluProject(xyzGL[0], xyzGL[1], xyzGL[2], model, project, viewport,
	     winGL, winGL + 1, winGL + 2);
  DBG_fprintf(stderr, "OpenGL View: get z coordinates from %gx%gx%g: %g.\n",
	      xyz[0], xyz[1], xyz[2], (float)winGL[2]);
  DBG_fprintf(stderr, " | win coords: %gx%g\n",
	      (float)winGL[0], (float)winGL[1]);
  return (float)winGL[2];
}
/**
 * openGLViewGet_realCoordinates:
 * @view: a #VisuOpenGLView object.
 * @xyz: a location to store the result.
 * @winx: position on X axis of screen.
 * @winy: position on Y axis of screen.
 * @winz: height before projection on screen.
 *
 * Use this routine to get the cartesian coordinates in real space of
 * a point located at @winx and @winy on screen.
 */
void openGLViewGet_realCoordinates(VisuOpenGLView *view, float xyz[3],
				   float winx, float winy, float winz)
{
  GLdouble model[16], project[16];
  GLint viewport[4];
  GLdouble xyzGL[3], winGL[3];

  g_return_if_fail(view);

  glGetDoublev(GL_MODELVIEW_MATRIX, model);
  glGetDoublev(GL_PROJECTION_MATRIX, project);
  glGetIntegerv(GL_VIEWPORT, viewport);
  
  winGL[0] = (GLdouble)winx;
  winGL[1] = (GLdouble)(view->window->height - winy);
  winGL[2] = (GLdouble)winz;
  gluUnProject(winGL[0], winGL[1], winGL[2], model, project, viewport,
	       xyzGL, xyzGL + 1, xyzGL + 2);
  xyz[0] = (float)xyzGL[0];
  xyz[1] = (float)xyzGL[1];
  xyz[2] = (float)xyzGL[2];
  DBG_fprintf(stderr, "OpenGL View: get real coordinates from %gx%gx%g: %gx%gx%g.\n",
	      winx, winy, winz, xyz[0], xyz[1], xyz[2]);
}

void openGLViewRotate_box(VisuOpenGLView *view, float dTheta, float dPhi, float angles[2])
{
  g_return_if_fail(view && angles);

  angles[0] = view->camera->theta + dTheta;
  angles[1] = view->camera->phi + dPhi;
}
void openGLViewRotate_camera(VisuOpenGLView *view, float dTheta, float dPhi, float angles[3])
{
  double cth, sth, cph, sph, com, som;
  double cdth, sdth, cdph, sdph;
  double Theta, Phi, Omega;
  #define RADTODEG 57.29577951
  float MinRprime[3], MinR[3];
  float Mspherical[3];
  float matPhi[3][3], matTheta[3][3], matOmega[3][3], matdPhi[3][3], matdTheta[3][3];
  float matPhiPrime[3][3], matThetaPrime[3][3];
  float matRprime2R[3][3];
  float matRes[3][3], matRes2[3][3];

  g_return_if_fail(view && angles);

  cth = cos(view->camera->theta * PI180);
  sth = sin(view->camera->theta * PI180);
  cph = cos(view->camera->phi * PI180);
  sph = sin(view->camera->phi * PI180);
  com = cos(view->camera->omega * PI180);
  som = sin(view->camera->omega * PI180);

  cdth = cos(dTheta * PI180);
  sdth = sin(dTheta * PI180);
  cdph = cos(dPhi * PI180);
  sdph = sin(dPhi * PI180);


  matPhi[0][0] = cph;
  matPhi[1][0] = sph;
  matPhi[2][0] = 0.;
  matPhi[0][1] = -sph;
  matPhi[1][1] = cph;
  matPhi[2][1] = 0.;
  matPhi[0][2] = 0.;
  matPhi[1][2] = 0.;
  matPhi[2][2] = 1.;

  matTheta[0][0] = cth;
  matTheta[1][0] = 0.;
  matTheta[2][0] = -sth;
  matTheta[0][1] = 0.;
  matTheta[1][1] = 1.;
  matTheta[2][1] = 0.;
  matTheta[0][2] = sth;
  matTheta[1][2] = 0.;
  matTheta[2][2] = cth;

  matOmega[0][0] = com;
  matOmega[1][0] = som;
  matOmega[2][0] = 0.;
  matOmega[0][1] = -som;
  matOmega[1][1] = com;
  matOmega[2][1] = 0.;
  matOmega[0][2] = 0.;
  matOmega[1][2] = 0.;
  matOmega[2][2] = 1.;

  matdPhi[0][0] = 1.;
  matdPhi[1][0] = 0.;
  matdPhi[2][0] = 0.;
  matdPhi[0][1] = 0.;
  matdPhi[1][1] = cdph;
  matdPhi[2][1] = -sdph;
  matdPhi[0][2] = 0.;
  matdPhi[1][2] = sdph;
  matdPhi[2][2] = cdph;

  matdTheta[0][0] = cdth;
  matdTheta[1][0] = 0.;
  matdTheta[2][0] = -sdth;
  matdTheta[0][1] = 0.;
  matdTheta[1][1] = 1.;
  matdTheta[2][1] = 0.;
  matdTheta[0][2] = sdth;
  matdTheta[1][2] = 0.;
  matdTheta[2][2] = cdth;

  tool_matrix_productMatrix(matRes, matdPhi, matdTheta);
  tool_matrix_productMatrix(matRes2, matOmega, matRes);
  tool_matrix_productMatrix(matRes, matTheta, matRes2);
  tool_matrix_productMatrix(matRprime2R, matPhi, matRes);

  MinRprime[0] = 0.;
  MinRprime[1] = 0.;
  MinRprime[2] = 1.;
  tool_matrix_productVector(MinR, matRprime2R, MinRprime);
/*   fprintf(stderr, "M : %f %f %f -> %f\n", MinR[0], MinR[1], MinR[2], */
/* 	  MinR[0]*MinR[0] + MinR[1]*MinR[1] + MinR[2]*MinR[2]); */

/*   cartesian_to_spherical(Mspherical, MinR); */
  Mspherical[0] = sqrt(MinR[0]*MinR[0] + MinR[1]*MinR[1] + MinR[2]*MinR[2]);
  if (MinR[1] == 0 && MinR[0] == 0)
    {
      Mspherical[1] = (MinR[2] > 0.)?0.:180.;
      Mspherical[2] = view->camera->phi;
    }
  else
    {
      Mspherical[1] = acos(MinR[2] / Mspherical[0]) * RADTODEG;
      if (MinR[0] == 0.)
	Mspherical[2] = (MinR[1] > 0.)?90.:-90.;
      else
	{
	  Mspherical[2] = atan(MinR[1] / MinR[0]) * RADTODEG;
	  if (MinR[0] < 0.)
	    Mspherical[2] += 180.;
	}
    }
/*   fprintf(stderr, "avant %f %f\n", Mspherical[1], Mspherical[2]); */
  while (Mspherical[1] - view->camera->theta < -90.)
    Mspherical[1] += 360.;
  while (Mspherical[1] - view->camera->theta > 90.)
    Mspherical[1] -= 360.;
  while (Mspherical[2] - view->camera->phi < -90.)
    Mspherical[2] += 360.;
  while (Mspherical[2] - view->camera->phi > 90.)
    Mspherical[2] -= 360.;
/*   fprintf(stderr, "aprs %f %f\n", Mspherical[1], Mspherical[2]); */

  Theta = Mspherical[1];
  Phi = Mspherical[2];

/*   fprintf(stderr, "%f %f, %f %f\n", view->camera->theta, view->camera->phi, Theta, Phi); */
/*   fprintf(stderr, "%f %f, %f %f\n", dTheta, dPhi, Theta - view->camera->theta, Phi -  view->camera->phi); */

  cth = cos(Theta * PI180);
  sth = sin(Theta * PI180);
  cph = cos(Phi * PI180);
  sph = sin(Phi * PI180);

  matPhiPrime[0][0] = cph;
  matPhiPrime[1][0] = -sph;
  matPhiPrime[2][0] = 0.;
  matPhiPrime[0][1] = sph;
  matPhiPrime[1][1] = cph;
  matPhiPrime[2][1] = 0.;
  matPhiPrime[0][2] = 0.;
  matPhiPrime[1][2] = 0.;
  matPhiPrime[2][2] = 1.;

  matThetaPrime[0][0] = cth;
  matThetaPrime[1][0] = 0.;
  matThetaPrime[2][0] = sth;
  matThetaPrime[0][1] = 0.;
  matThetaPrime[1][1] = 1.;
  matThetaPrime[2][1] = 0.;
  matThetaPrime[0][2] = -sth;
  matThetaPrime[1][2] = 0.;
  matThetaPrime[2][2] = cth;

  tool_matrix_productMatrix(matRes2, matPhiPrime, matRprime2R);
  tool_matrix_productMatrix(matRes, matThetaPrime, matRes2);

  MinRprime[0] = 0.;
  MinRprime[1] = 1.;
  MinRprime[2] = 0.;
  tool_matrix_productVector(MinR, matRes, MinRprime);
/*   fprintf(stderr, "vect u : %f %f %f -> %f\n", MinR[0], MinR[1], MinR[2], */
/* 	  MinR[0]*MinR[0] + MinR[1]*MinR[1] + MinR[2]*MinR[2]); */
  Omega = acos(CLAMP(MinR[1], -1.f, 1.f)) * RADTODEG;
  if (MinR[0] > 0.)
    Omega = -Omega;
  while (Omega - view->camera->omega < -90.)
    Omega += 360.;
  while (Omega - view->camera->omega > 90.)
    Omega -= 360.;

  /*   fprintf(stderr, "Theta phi omega : %f %f %f\n", Theta, Phi, Omega); */
  angles[0] = Theta;
  angles[1] = Phi;
  angles[2] = Omega;
}



/* This is a positive float that enable to increase or
   decrease the rendering load by modifying the
   number of facettes. */
#define FLAG_PARAMETER_OPENGL_DETAILS   "opengl_details"
#define DESC_PARAMETER_OPENGL_DETAILS   "Give a value to the quality of rendering (100 is normal) ; positive integer"
#define PARAMETER_OPENGL_DETAILS_DEFAULT 100
static gboolean readOpenGLPrecision(gchar **lines, int nbLines, int position,
				    VisuData *dataObj, GError **error);
static void exportParametersVisuOpenGLView(GString *data, VisuData *dataObj);

#define FLAG_RESOURCE_OPENGL_ANGLES "opengl_theta_phi_omega"
#define DESC_RESOURCE_OPENGL_ANGLES "2 real values (degrees) for user orientation with respect to sample"
static gboolean readOpenGLThetaPhi(gchar **lines, int nbLines, int position,
				   VisuData *dataObj, GError **error);
#define FLAG_RESOURCE_OPENGL_TRANSLAT "opengl_xs_ys"
#define DESC_RESOURCE_OPENGL_TRANSLAT "2 real values for image position with respect to [0.0, 1.0]x[0.0, 1.0] window"
static gboolean readOpenGLXsYs(gchar **lines, int nbLines, int position,
			       VisuData *dataObj, GError **error);
#define FLAG_RESOURCE_OPENGL_GROSS "opengl_gross"
#define DESC_RESOURCE_OPENGL_GROSS "gross factor (must be real > 0.0)"
static gboolean readOpenGLGross(gchar **lines, int nbLines, int position,
				VisuData *dataObj, GError **error);
#define FLAG_RESOURCE_OPENGL_PERSP "opengl_d_red"
#define DESC_RESOURCE_OPENGL_PERSP "reduced perspective distance (must be real > 1.0)"
static gboolean readOpenGLPersp(gchar **lines, int nbLines, int position,
				VisuData *dataObj, GError **error);
static void exportResourcesVisuOpenGLView(GString *data, VisuData *dataObj);


void VisuOpenGLViewInit(void)
{
  VisuConfigFileEntry *resourceEntry;

  DBG_fprintf(stderr, "OpenGl View : initialization.\n");
  /* Parameters */
  resourceEntry = visu_configFile_addEntry(VISU_CONFIGFILE_PARAMETER,
					  FLAG_PARAMETER_OPENGL_DETAILS,
					  DESC_PARAMETER_OPENGL_DETAILS,
					  1, readOpenGLPrecision);
  visu_configFile_addExportFunction(VISU_CONFIGFILE_PARAMETER,
				   exportParametersVisuOpenGLView);

  VisuOpenGLViewSet_precision((float)PARAMETER_OPENGL_DETAILS_DEFAULT / 100.);

  /* Resources */
  resourceEntry = visu_configFile_addEntry(VISU_CONFIGFILE_RESOURCE,
					  FLAG_RESOURCE_OPENGL_ANGLES,
					  DESC_RESOURCE_OPENGL_ANGLES,
					  1, readOpenGLThetaPhi);
  visu_configFile_entry_setVersion(resourceEntry, 3.1f);
  resourceEntry = visu_configFile_addEntry(VISU_CONFIGFILE_RESOURCE,
					  FLAG_RESOURCE_OPENGL_TRANSLAT,
					  DESC_RESOURCE_OPENGL_TRANSLAT,
					  1, readOpenGLXsYs);
  resourceEntry = visu_configFile_addEntry(VISU_CONFIGFILE_RESOURCE,
					  FLAG_RESOURCE_OPENGL_GROSS,
					  DESC_RESOURCE_OPENGL_GROSS,
					  1, readOpenGLGross);
  resourceEntry = visu_configFile_addEntry(VISU_CONFIGFILE_RESOURCE,
					  FLAG_RESOURCE_OPENGL_PERSP,
					  DESC_RESOURCE_OPENGL_PERSP,
					  1, readOpenGLPersp);
  visu_configFile_addExportFunction(VISU_CONFIGFILE_RESOURCE,
				   exportResourcesVisuOpenGLView);

}

static gboolean readOpenGLPrecision(gchar **lines, int nbLines, int position,
				    VisuData *dataObj _U_, GError **error)
{
  int val;

  g_return_val_if_fail(nbLines == 1, FALSE);

  if (!tool_configFile_readInteger(lines[0], position, &val, 1, error))
    return FALSE;
  if (val <= 0 || val > 500)
    {
      *error = g_error_new(TOOL_CONFIGFILE_ERROR, TOOL_CONFIGFILE_ERROR_VALUE,
			   _("Parse error at line %d: width must be in %d-%d.\n"),
			   position, 0, 500);
      return FALSE;
    }
  VisuOpenGLViewSet_precision((float)val / 100.);
  return TRUE;
}
static void exportParametersVisuOpenGLView(GString *data, VisuData *dataObj _U_)
{
  g_string_append_printf(data, "# %s\n", DESC_PARAMETER_OPENGL_DETAILS);
  g_string_append_printf(data, "%s: %d\n\n", FLAG_PARAMETER_OPENGL_DETAILS,
	  (int)(precisionOfRendering * 100.));
}
static gboolean readOpenGLThetaPhi(gchar **lines, int nbLines, int position,
				   VisuData *dataObj, GError **error)
{
  float val[3];
  
  g_return_val_if_fail(nbLines == 1, FALSE);

  if (!tool_configFile_readFloat(lines[0], position, val, 3, error))
    return FALSE;
  anglesDefault[0] = val[0];
  anglesDefault[1] = val[1];
  anglesDefault[2] = val[2];
  if (dataObj)
    visu_data_setAngleOfView(dataObj, val[0], val[1], val[2],
			    VISU_CAMERA_THETA | VISU_CAMERA_PHI | VISU_CAMERA_OMEGA);
  
  return TRUE;
}
static gboolean readOpenGLXsYs(gchar **lines, int nbLines, int position,
			       VisuData *dataObj, GError **error)
{
  float val[2];
  
  g_return_val_if_fail(nbLines == 1, FALSE);

  if (!tool_configFile_readFloat(lines[0], position, val, 2, error))
    return FALSE;
  translatDefault[0] = val[0];
  translatDefault[1] = val[1];
  if (dataObj)
    visu_data_setPositionOfView(dataObj, val[0], val[1],
			       MASK_XS | MASK_YS);
  
  return TRUE;
}
static gboolean readOpenGLGross(gchar **lines, int nbLines, int position,
				VisuData *dataObj, GError **error)
{
  float val;
  
  g_return_val_if_fail(nbLines == 1, FALSE);

  if (!tool_configFile_readFloat(lines[0], position, &val, 1, error))
    return FALSE;
  grossDefault = val;
  if (dataObj)
    visu_data_setZoomOfView(dataObj, val);

  return TRUE;
}
static gboolean readOpenGLPersp(gchar **lines, int nbLines, int position,
				VisuData *dataObj, GError **error)
{
  float val;
  
  g_return_val_if_fail(nbLines == 1, FALSE);

  if (!tool_configFile_readFloat(lines[0], position, &val, 1, error))
    return FALSE;
  perspDefault = CLAMP(val, 1.1f, 100.f);
  if (dataObj)
    visu_data_setPerspectiveOfView(dataObj, val);

  return TRUE;
}
static void exportResourcesVisuOpenGLView(GString *data, VisuData *dataObj _U_)
{
  GList *tmpLst;
  VisuOpenGLView *view;

  tmpLst = visu_data_getAllObjects();
  if (tmpLst)
    view = visu_data_getOpenGLView(VISU_DATA(tmpLst->data));
  else
    view = (VisuOpenGLView*)0;

  g_string_append_printf(data, "# %s\n", DESC_RESOURCE_OPENGL_ANGLES);
  g_string_append_printf(data, "%s:\n", FLAG_RESOURCE_OPENGL_ANGLES);
  if (view)
    g_string_append_printf(data, "    %9.3f %9.3f %9.3f\n",
			   view->camera->theta, view->camera->phi, view->camera->omega);
  else
    g_string_append_printf(data, "    %9.3f %9.3f %9.3f\n",
			   anglesDefault[0], anglesDefault[1], anglesDefault[2]);
  g_string_append_printf(data, "# %s\n", DESC_RESOURCE_OPENGL_TRANSLAT);
  g_string_append_printf(data, "%s:\n", FLAG_RESOURCE_OPENGL_TRANSLAT);
  if (view)
    g_string_append_printf(data, "    %9.3f %9.3f\n",
			   view->camera->xs, view->camera->ys);
  else
    g_string_append_printf(data, "    %9.3f %9.3f\n",
			   translatDefault[0], translatDefault[1]);
  g_string_append_printf(data, "# %s\n", DESC_RESOURCE_OPENGL_GROSS);
  g_string_append_printf(data, "%s:\n", FLAG_RESOURCE_OPENGL_GROSS);
  if (view)
    g_string_append_printf(data, "    %9.3f\n", view->camera->gross);
  else
    g_string_append_printf(data, "    %9.3f\n", grossDefault);
  g_string_append_printf(data, "# %s\n", DESC_RESOURCE_OPENGL_PERSP);
  g_string_append_printf(data, "%s:\n", FLAG_RESOURCE_OPENGL_PERSP);
  if (view)
    g_string_append_printf(data, "    %9.3f\n", view->camera->d_red);
  else
    g_string_append_printf(data, "    %9.3f\n", perspDefault);
  g_string_append_printf(data, "\n");
}

