/*   EXTRAITS DE LA LICENCE
	Copyright CEA, contributeurs : Luc BILLARD et Damien
	CALISTE, laboratoire L_Sim, (2001-2005)
  
	Adresse ml :
	BILLARD, non joignable par ml ;
	CALISTE, damien P caliste AT cea P fr.

	Ce logiciel est un programme informatique servant  visualiser des
	structures atomiques dans un rendu pseudo-3D. 

	Ce logiciel est rgi par la licence CeCILL soumise au droit franais et
	respectant les principes de diffusion des logiciels libres. Vous pouvez
	utiliser, modifier et/ou redistribuer ce programme sous les conditions
	de la licence CeCILL telle que diffuse par le CEA, le CNRS et l'INRIA 
	sur le site "http://www.cecill.info".

	Le fait que vous puissiez accder  cet en-tte signifie que vous avez 
	pris connaissance de la licence CeCILL, et que vous en avez accept les
	termes (cf. le fichier Documentation/licence.fr.txt fourni avec ce logiciel).
*/

/*   LICENCE SUM UP
	Copyright CEA, contributors : Luc BILLARD et Damien
	CALISTE, laboratoire L_Sim, (2001-2005)

	E-mail address:
	BILLARD, not reachable any more ;
	CALISTE, damien P caliste AT cea P fr.

	This software is a computer program whose purpose is to visualize atomic
	configurations in 3D.

	This software is governed by the CeCILL  license under French law and
	abiding by the rules of distribution of free software.  You can  use, 
	modify and/ or redistribute the software under the terms of the CeCILL
	license as circulated by CEA, CNRS and INRIA at the following URL
	"http://www.cecill.info". 

	The fact that you are presently reading this means that you have had
	knowledge of the CeCILL license and that you accept its terms. You can
	find a copy of this licence shipped with this software at Documentation/licence.en.txt.
*/

#include "toolConfigFile.h"
#include <visu_tools.h>

/**
 * SECTION:toolConfigFile
 * @short_description: Generic read methods of the configuration files
 * are defined here.
 *
 * <para>These methods give generic tools to read common data in the
 * configuration files, such as arrays of float values or one
 * #VisuElement... It also defines a enumeration of detailed errors
 * (see #toolConfigFileError) when reading a file. All read elements
 * are either from tokens (given g_strsplit()) or strings.</para>
 */

gboolean configFileRead_floatFromTokens(gchar **tokens, int *position, float *values,
					guint size, int lineId, GError **error)
{
  int res;
  guint i, nb;

  g_return_val_if_fail(error && (*error == (GError*)0), FALSE);
  g_return_val_if_fail(values, FALSE);
  g_return_val_if_fail(tokens && position, FALSE);
  
  /* Read @size floating point values from tokens. */
  nb = 0;
  for (i = *position; tokens[i] && nb < size; i++)
    {
      if (tokens[i][0] != '\0')
	{
	  res = sscanf(tokens[i], "%f", values + nb);
	  if (res != 1)
	    {
	      *error = g_error_new(CONFIG_FILE_ERROR, CONFIG_FILE_ERROR_READ,
				   _("Parse error at line %d, %d floating point"
				     " values should appear here.\n"), lineId, size);
	      *position = i;
	      return FALSE;
	    }
	  nb += 1;
	}
    }
  *position = i;

  if (nb != size)
    {
      *error = g_error_new(CONFIG_FILE_ERROR, CONFIG_FILE_ERROR_MISSING,
			   _("Parse error at line %d, %d floating point value(s)"
			     " should appear here but %d has been found.\n"),
			   lineId, size, nb);
      return FALSE;
    }
  return TRUE;
}


gboolean configFileRead_float(gchar *line, int position, float *values,
			      guint size, GError **error)
{
  int id;
  gboolean res;
  gchar **tokens;

  /* Read @size floating point values from @line. */
  tokens = g_strsplit_set(line, " \n", MAX_LINE_LENGTH);
  id = 0;
  res = configFileRead_floatFromTokens(tokens, &id, values, size, position, error);
  g_strfreev(tokens);

  return res;
}

gboolean configFileRead_booleanFromTokens(gchar **tokens, int *position, gboolean *values,
					  guint size, int lineId, GError **error)
{
  int res;
  guint i, nb;

  g_return_val_if_fail(error && (*error == (GError*)0), FALSE);
  g_return_val_if_fail(values, FALSE);
  g_return_val_if_fail(tokens && position, FALSE);
  
  /* Read @size floating point values from @line. */
  nb = 0;
  for (i = *position; tokens[i] && nb < size; i++)
    {
      if (tokens[i][0] != '\0')
	{
	  res = sscanf(tokens[i], "%d", values + nb);
	  if (res != 1)
	    {
	      *error = g_error_new(CONFIG_FILE_ERROR, CONFIG_FILE_ERROR_READ,
				   _("Parse error at line %d, %d boolean"
				     " values should appear here.\n"), lineId, size);
	      *position = i;
	      return FALSE;
	    }
	  nb += 1;
	}
    }
  *position = i;

  if (nb != size)
    {
      *error = g_error_new(CONFIG_FILE_ERROR, CONFIG_FILE_ERROR_MISSING,
			   _("Parse error at line %d, %d boolean(s)"
			     " values should appear here but %d has been found.\n"),
			   lineId, size, nb);
      return FALSE;
    }
  return TRUE;
}

gboolean configFileRead_boolean(gchar *line, int position, gboolean *values,
				guint size, GError **error)
{
  int id;
  gboolean res;
  gchar **tokens;

  /* Read @size boolean values from @line. */
  tokens = g_strsplit_set(line, " \n", MAX_LINE_LENGTH);
  id = 0;
  res = configFileRead_booleanFromTokens(tokens, &id, values, size, position, error);
  g_strfreev(tokens);

  return res;
}

gboolean configFileRead_stringFromTokens(gchar **tokens, int *position, gchar ***values,
					  guint size, int lineId, GError **error)
{
  guint i, nb;

  g_return_val_if_fail(error && (*error == (GError*)0), FALSE);
  g_return_val_if_fail(values, FALSE);
  g_return_val_if_fail(tokens && position, FALSE);
  
  /* Read @size floating point values from @line. */
  *values = g_malloc(sizeof(gchar**) * (size + 1));
  nb = 0;
  for (i = *position; tokens[i] && nb < size; i++)
    if (tokens[i][0] != '\0')
      (*values)[nb++] = g_strdup(tokens[i]);
  *position = i;
  (*values)[nb] = (gchar*)0;

  if (nb != size)
    {
      *error = g_error_new(CONFIG_FILE_ERROR, CONFIG_FILE_ERROR_MISSING,
			   _("Parse error at line %d, %d string(s)"
			     " should appear here but %d has been found.\n"),
			   lineId, size, nb);
      g_strfreev(*values);
      *values = (gchar**)0;
      return FALSE;
    }
  return TRUE;
}

gboolean configFileRead_string(gchar *line, int position, gchar ***values,
			       guint size, gboolean join, GError **error)
{
  guint i, nb;
  gchar *tmp;

  g_return_val_if_fail(error && (*error == (GError*)0), FALSE);
  g_return_val_if_fail(values, FALSE);
  
  /* Read @size floating point values from @line. */
  *values = g_strsplit_set(line, " \n", MAX_LINE_LENGTH);
  nb = 0;
  for (i = 0; (*values)[i]; i++)
    if ((*values)[i][0] != '\0')
      {
	tmp = (*values)[i];
	(*values)[i] = (*values)[nb];
	(*values)[nb] = tmp;
	nb += 1;
      }

  if (!join)
    {
      if (nb != size)
	{
	  *error = g_error_new(CONFIG_FILE_ERROR, CONFIG_FILE_ERROR_MISSING,
			       _("Parse error at line %d, %d string(s)"
				 " should appear here but %d has been found.\n"),
			       position, size, nb);
	  g_strfreev(*values);
	  return FALSE;
	}
    }
  else
    {
      tmp = g_strjoinv(" ", (*values) + (size - 1));
      for (i = size - 1; (*values)[i]; i++)
	g_free((*values)[i]);
      (*values)[size - 1] = tmp;
      (*values)[size] = (gchar*)0;
    }
  return TRUE;
}

gboolean configFileRead_floatWithElement(gchar *line, int position, float *values,
					 guint size, VisuElement **ele, GError **error)
{
  int id;
  gboolean res;
  gchar **tokens;

  /* Read @size boolean values from @line. */
  tokens = g_strsplit_set(line, " \n", MAX_LINE_LENGTH);
  id = 0;
  res = configFileRead_elementFromTokens(tokens, &id, ele, 1, position, error);
  if (!res)
    {
      g_strfreev(tokens);
      return FALSE;
    }
  res = configFileRead_floatFromTokens(tokens, &id, values, size, position, error);
  g_strfreev(tokens);

  return res;
}
gboolean configFileRead_booleanWithElement(gchar *line, int position, gboolean *values,
					   guint size, VisuElement **ele, GError **error)
{
  int id;
  gboolean res;
  gchar **tokens;

  /* Read @size boolean values from @line. */
  tokens = g_strsplit_set(line, " \n", MAX_LINE_LENGTH);
  id = 0;
  res = configFileRead_elementFromTokens(tokens, &id, ele, 1, position, error);
  if (!res)
    {
      g_strfreev(tokens);
      return FALSE;
    }
  res = configFileRead_booleanFromTokens(tokens, &id, values, size, position, error);
  g_strfreev(tokens);

  return res;
}
gboolean configFileRead_stringWithElement(gchar *line, int position, gchar ***values,
					  guint size, VisuElement **ele, GError **error)
{
  int id;
  gboolean res;
  gchar **tokens;

  /* Read @size boolean values from @line. */
  tokens = g_strsplit_set(line, " \n", MAX_LINE_LENGTH);
  id = 0;
  res = configFileRead_elementFromTokens(tokens, &id, ele, 1, position, error);
  if (!res)
    {
      g_strfreev(tokens);
      return FALSE;
    }
  res = configFileRead_stringFromTokens(tokens, &id, values, size, position, error);
  g_strfreev(tokens);

  return res;
}

gboolean configFileRead_elementFromTokens(gchar **tokens, int *position, VisuElement **values,
					  guint size, int lineId, GError **error)
{
  guint i, nb;

  g_return_val_if_fail(error && (*error == (GError*)0), FALSE);
  g_return_val_if_fail(values, FALSE);
  g_return_val_if_fail(tokens && position, FALSE);
  
  /* Read @size VisuElement from @tokens. */
  nb = 0;
  for (i = *position; tokens[i] && nb < size; i++)
    {
      if (tokens[i][0] != '\0')
	{
	  values[nb] = visuElementGet_fromName(tokens[i]);
	  if (!values[nb])
	    {
	      values[nb] = visuElementNew_withName(tokens[i]);
	      if (!values[nb])
		{
		  *error = g_error_new(CONFIG_FILE_ERROR, CONFIG_FILE_ERROR_BAD_ELEMENT_NAME,
				       _("Parse error at line %d, '%s' wrong"
					 " element name.\n"), lineId, tokens[i]);
		  *position = i;
		  return FALSE;
		}
	      visuElementAdd(values[nb]);
	    }
	  nb += 1;
	}
    }
  *position = i;

  if (nb != size)
    {
      *error = g_error_new(CONFIG_FILE_ERROR, CONFIG_FILE_ERROR_MISSING,
			   _("Parse error at line %d, %d elements"
			     " should appear here but %d has been found.\n"),
			   lineId, size, nb);
      return FALSE;
    }
  return TRUE;
}

gboolean configFileRead_integerFromTokens(gchar **tokens, int *position, int *values,
					  guint size, int lineId, GError **error)
{
  int res;
  guint i, nb;

  g_return_val_if_fail(error && (*error == (GError*)0), FALSE);
  g_return_val_if_fail(values, FALSE);
  g_return_val_if_fail(tokens && position, FALSE);
  
  /* Read @size floating point values from @line. */
  nb = 0;
  for (i = *position; tokens[i] && nb < size; i++)
    {
      if (tokens[i][0] != '\0')
	{
	  res = sscanf(tokens[i], "%d", values + nb);
	  if (res != 1)
	    {
	      *error = g_error_new(CONFIG_FILE_ERROR, CONFIG_FILE_ERROR_READ,
				   _("Parse error at line %d, %d integer"
				     " values should appear here.\n"), lineId, size);
	      *position = i;
	      return FALSE;
	    }
	  nb += 1;
	}
    }
  *position = i;

  if (nb != size)
    {
      *error = g_error_new(CONFIG_FILE_ERROR, CONFIG_FILE_ERROR_MISSING,
			   _("Parse error at line %d, %d integer(s)"
			     " values should appear here but %d has been found.\n"),
			   lineId, size, nb);
      return FALSE;
    }
  return TRUE;
}
gboolean configFileRead_integer(gchar *line, int position, int *values,
				guint size, GError **error)
{
  int id;
  gboolean res;
  gchar **tokens;

  /* Read @size floating point values from @line. */
  tokens = g_strsplit_set(line, " \n", MAX_LINE_LENGTH);
  id = 0;
  res = configFileRead_integerFromTokens(tokens, &id, values, size, position, error);
  g_strfreev(tokens);

  return res;
}

gboolean configFileClamp_float(float *variable, float value, float min, float max)
{
  g_return_val_if_fail(variable, FALSE);

  if (max > min && value > max)
    {
      *variable = max;
      return TRUE;
    }
  if (min < max && value < min)
    {
      *variable = min;
      return TRUE;
    }
  
  *variable = value;
  return FALSE;
}

