/*
 * Telepathy Inspector - A Telepathy client which exposes Telepathy interfaces.
 *                       Meant to inspect and/or test connection managers.
 *
 * ti-wnd-connection.c:
 * Connection Window - UI to expose org.freedesktop.Telepathy.Connection functionality
 *
 * Copyright (C) 2008 Collabora Ltd. <http://www.collabora.co.uk/>
 * Copyright (C) 2008 Nokia Corporation
 * Copyright (C) 2006-2007 INdT - Instituto Nokia de Tecnologia
 * Originally by Daniel d'Andrada T. de Carvalho <daniel.carvalho@indt.org.br>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public License
 * along with this library; if not, write to the Free Software Foundation, Inc.,
 * 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA
 */

#include "wnd-connection.h"

#include <string.h>

#include <glade/glade.h>
#include <telepathy-glib/connection.h>
#include <telepathy-glib/dbus.h>
#include <telepathy-glib/enums.h>
#include <telepathy-glib/interfaces.h>
#include <telepathy-glib/util.h>

#include "constants.h"
#include "dlg-hold-handles.h"
#include "dlg-get-handles.h"
#include "dlg-request-channel.h"
#include "handle-mapper.h"
#include "page-aliasing.h"
#include "page-contacts.h"
#include "page-capabilities.h"
#include "page-simple-presence.h"
#include "page-presence.h"
#include "page-properties.h"
#include "util.h"
#include "wnd-channel.h"

G_DEFINE_TYPE (TIWndConnection, ti_wnd_connection, G_TYPE_OBJECT);

/* Function prototypes */
static gboolean on_delete_window (TIWndConnection *self, GdkEvent  *event);
static void _ti_wnd_connection_setup_first_page (TIWndConnection *self);
static void _ti_wnd_connection_setup_acquired_handles_list (TIWndConnection *self);
static void _ti_wnd_connection_remove_interface_pages (TIWndConnection *self);
static gchar *_get_status_string (guint status, guint reason);
static void _ti_wnd_connection_build_interfaces_treeview (TIWndConnection *self);
static void _ti_wnd_connection_build_channels_treeview (TIWndConnection *self);
static void _ti_wnd_connection_connect (TIWndConnection *self);
static void _ti_wnd_connection_disconnect (TIWndConnection *self);
static void _ti_wnd_connection_on_new_channel (TpConnection *connection,
    const gchar *object_path, const gchar *channel_type, guint handle_type,
    guint handle, gboolean suppress_handler, gpointer unused, GObject *object);
static void _ti_wnd_connection_refresh_channels_list (TIWndConnection *self);
static void _ti_wnd_connection_request_channel (TIWndConnection *self);
static void _ti_wnd_connection_request_handles (TIWndConnection *self);
static void _ti_wnd_connection_hold_handles (TIWndConnection *self);
static void _ti_wnd_connection_closed_channel_window (TIWndConnection *self, TIWndChannel *wnd_channel);
static void _ti_wnd_connection_add_channel_to_list (TIWndConnection *self,
                                                    const gchar *obj_path,
                                                    const gchar *type,
                                                    guint handle_type,
                                                    guint handle_number);
static gchar *_get_small_channel_type (const gchar *type);
static void _ti_wnd_connection_show_selected_channel (TIWndConnection *self);
static void _ti_wnd_connection_inspect_selected_handles (TIWndConnection *self);
static void _ti_wnd_connection_release_selected_handles (TIWndConnection *self);
static void _ti_wnd_connection_handles_selection_changed (TIWndConnection *self);
static void _ti_wnd_connection_get_selected_handles (TIWndConnection *self,
                                                     GArray *contact_handles, GArray *room_handles, GArray *list_handles);
static void _ti_wnd_connection_create_interface_page (TIWndConnection *self, gchar *interface_name);
static void _ti_wnd_connection_on_connected (TIWndConnection *self);
static void _ti_wnd_connection_on_notify_status (TIWndConnection *self);
static void _ti_wnd_connection_on_invalidated (TpConnection *connection,
    guint domain, gint code, gchar *message, TIWndConnection *self);

struct _TIWndConnectionPrivate {
    TIHandleMapper *handle_mapper;

    TIDlgHoldHandles *dlg_hold_handles;
    TIDlgGetHandles *dlg_get_handles;
    TIDlgRequestChannel *dlg_request_channel;

    GtkWindow *parent;

    GladeXML *glade_xml;

    GtkWidget *window;
    GtkNotebook *notebook;

    GtkListStore *interfaces_list;
    GtkListStore *channels_list;

    GtkTreeSelection *channel_selection;

    TpConnection *connection;

    GtkWidget *button_connect;
    GtkWidget *button_disconnect;

    GtkWidget *button_request_handles;

    GtkWidget *button_request_channel;
    GtkWidget *button_refresh_channels_list;

    GtkTreeSelection *acquired_handles_selection;
    GtkWidget *button_inspect_handles;
    GtkWidget *button_release_handles;
    GtkWidget *button_hold_handles;

    /* An array of TIWndChannel * containing all channel windows currently
     * open */
    GPtrArray *channel_wnds;

    TIPageAliasing *page_aliasing;
    TIPageContacts *page_contacts;
    TIPageCapabilities *page_capabilities;
    TIPageSimplePresence *page_simple_presence;
    TIPagePresence *page_presence;
    TIPageProperties *page_properties;
};

enum {
    PROP_CONNECTION = 1,
    PROP_TRANSIENT_FOR,
};

static void
ti_wnd_connection_dispose (GObject *object)
{
  TIWndConnection *self = TI_WND_CONNECTION (object);
  guint i;
  GObject *obj;

  TI_OBJ_UNREF (self->priv->handle_mapper);
  TI_OBJ_UNREF (self->priv->dlg_hold_handles);
  TI_OBJ_UNREF (self->priv->dlg_get_handles);
  TI_OBJ_UNREF (self->priv->dlg_request_channel);
  TI_OBJ_UNREF (self->priv->glade_xml);

  if (self->priv->connection != NULL)
    {
      g_signal_handlers_disconnect_by_func (self->priv->connection,
          G_CALLBACK (_ti_wnd_connection_on_connected), self);
      g_signal_handlers_disconnect_by_func (self->priv->connection,
          G_CALLBACK (_ti_wnd_connection_on_notify_status), self);
      g_signal_handlers_disconnect_by_func (self->priv->connection,
          G_CALLBACK (_ti_wnd_connection_on_invalidated), self);

      g_object_unref (self->priv->connection);
      self->priv->connection = NULL;
    }

  if (self->priv->channel_wnds != NULL)
    {
      for (i = 0; i < self->priv->channel_wnds->len; i++)
        {
          obj = G_OBJECT (g_ptr_array_index (self->priv->channel_wnds, i));
          g_object_unref (obj);
        }

      g_ptr_array_free (self->priv->channel_wnds, TRUE);
      self->priv->channel_wnds = NULL;
    }

  TI_OBJ_UNREF (self->priv->page_aliasing);
  TI_OBJ_UNREF (self->priv->page_contacts);
  TI_OBJ_UNREF (self->priv->page_capabilities);
  TI_OBJ_UNREF (self->priv->page_simple_presence);
  TI_OBJ_UNREF (self->priv->page_presence);
  TI_OBJ_UNREF (self->priv->page_properties);

  G_OBJECT_CLASS (ti_wnd_connection_parent_class)->dispose (object);
}


static void
ti_wnd_connection_get_property (GObject *object,
                                guint property_id,
                                GValue *value,
                                GParamSpec *pspec)
{
  TIWndConnection *self = TI_WND_CONNECTION (object);

  switch (property_id)
    {
    case PROP_CONNECTION:
      g_value_set_object (value, self->priv->connection);
      break;
    case PROP_TRANSIENT_FOR:
      g_value_set_object (value, self->priv->parent);
      break;
    default:
      G_OBJECT_WARN_INVALID_PROPERTY_ID (object, property_id, pspec);
    }
}


static void
ti_wnd_connection_set_property (GObject *object,
                                guint property_id,
                                const GValue *value,
                                GParamSpec *pspec)
{
  TIWndConnection *self = TI_WND_CONNECTION (object);

  switch (property_id)
    {
    case PROP_CONNECTION:
      g_return_if_fail (self->priv->connection == NULL);
      self->priv->connection = TP_CONNECTION (g_value_dup_object (value));
      break;
    case PROP_TRANSIENT_FOR:
      g_return_if_fail (self->priv->parent == NULL);
      self->priv->parent = GTK_WINDOW (g_value_dup_object (value));
      break;
    default:
      G_OBJECT_WARN_INVALID_PROPERTY_ID (object, property_id, pspec);
    }
}


static void
ti_wnd_connection_constructed (GObject *object)
{
  void (*chain_up) (GObject *) =
    G_OBJECT_CLASS (ti_wnd_connection_parent_class)->constructed;
  TIWndConnection *self = TI_WND_CONNECTION (object);
  gchar *glade_file_path = NULL;
  gboolean connection_ready;
  guint status, status_reason;
  GtkLabel *label;
  gchar *str;

  if (chain_up != NULL)
    chain_up (object);

  self->priv->handle_mapper = ti_handle_mapper_new (self->priv->connection);

  glade_file_path = ti_get_glade_path ("wnd-connection.xml");
  self->priv->glade_xml = glade_xml_new (glade_file_path, NULL, NULL);

  if (self->priv->glade_xml == NULL)
    {
      g_critical ("Error loading glade file \"%s\".", glade_file_path);
      g_object_unref (self);
      self = NULL;
      goto CLEAN_UP;
    }

  self->priv->window = glade_xml_get_widget (self->priv->glade_xml,
      "window_connection");
  g_assert (GTK_IS_WINDOW (self->priv->window));

  self->priv->dlg_hold_handles = ti_dlg_hold_handles_new (
      GTK_WINDOW (self->priv->window));
  self->priv->dlg_get_handles = ti_dlg_get_handles_new (
      GTK_WINDOW (self->priv->window));
  self->priv->dlg_request_channel = ti_dlg_request_channel_new (
      GTK_WINDOW (self->priv->window), self->priv->handle_mapper);

  _ti_wnd_connection_setup_first_page (self);
  _ti_wnd_connection_setup_acquired_handles_list (self);

  g_signal_connect_swapped (self->priv->window, "delete-event",
      G_CALLBACK (on_delete_window), self);

  gtk_window_set_transient_for (GTK_WINDOW (self->priv->window),
      GTK_WINDOW (self->priv->parent));

    /* Connect to the Telepathy object */

  g_object_get (self->priv->connection,
      "connection-ready", &connection_ready,
      "status", &status,
      "status-reason", &status_reason,
      NULL);

  label = GTK_LABEL (glade_xml_get_widget (self->priv->glade_xml,
        "label_status_value"));
  str = _get_status_string (status, status_reason);
  gtk_label_set_text (label, str);
  g_free (str);

  g_signal_connect_swapped (self->priv->connection, "notify::status",
      G_CALLBACK (_ti_wnd_connection_on_notify_status), self);

  /* FIXME: get protocol from the TpConnection (we can just parse the
   * object path) */
  label = GTK_LABEL (glade_xml_get_widget (self->priv->glade_xml,
        "label_protocol_value"));
  gtk_label_set_text (label, "???");

  if (connection_ready)
    _ti_wnd_connection_on_connected (self);
  else
    g_signal_connect_swapped (self->priv->connection,
        "notify::connection-ready",
        G_CALLBACK (_ti_wnd_connection_on_connected), self);

  g_signal_connect (self->priv->connection, "invalidated",
      G_CALLBACK (_ti_wnd_connection_on_invalidated), self);

  tp_cli_connection_connect_to_new_channel (self->priv->connection,
      _ti_wnd_connection_on_new_channel, NULL, NULL,
      (GObject *) self, NULL);

CLEAN_UP:
  g_free (glade_file_path);
}


static void
ti_wnd_connection_class_init (TIWndConnectionClass *klass)
{
  GObjectClass *gobject_class = G_OBJECT_CLASS (klass);
  GParamSpec *param_spec;

  gobject_class->constructed = ti_wnd_connection_constructed;
  gobject_class->dispose = ti_wnd_connection_dispose;
  gobject_class->get_property = ti_wnd_connection_get_property;
  gobject_class->set_property = ti_wnd_connection_set_property;

  g_type_class_add_private (klass, sizeof (TIWndConnectionPrivate));

  /* Emitted when the connection is closed. */
  klass->close_signal_id = g_signal_newv ("closed", G_TYPE_FROM_CLASS (klass),
           G_SIGNAL_RUN_LAST | G_SIGNAL_NO_RECURSE | G_SIGNAL_NO_HOOKS,
           NULL /* class closure */,
           NULL /* accumulator */,
           NULL /* accu_data */,
           g_cclosure_marshal_VOID__VOID,
           G_TYPE_NONE /* return_type */,
           0     /* n_params */,
           NULL  /* param_types */);

  param_spec = g_param_spec_object ("connection", "TpConnection",
      "The TpConnection represented by this window", TP_TYPE_CONNECTION,
      G_PARAM_CONSTRUCT_ONLY | G_PARAM_READWRITE |
      G_PARAM_STATIC_NAME | G_PARAM_STATIC_BLURB | G_PARAM_STATIC_NICK);
  g_object_class_install_property (gobject_class, PROP_CONNECTION, param_spec);

  param_spec = g_param_spec_object ("transient-for", "Transient for",
      "The parent window for this window", GTK_TYPE_WINDOW,
      G_PARAM_CONSTRUCT_ONLY | G_PARAM_READWRITE |
      G_PARAM_STATIC_NAME | G_PARAM_STATIC_BLURB | G_PARAM_STATIC_NICK);
  g_object_class_install_property (gobject_class, PROP_TRANSIENT_FOR,
      param_spec);
}

static void
ti_wnd_connection_init (TIWndConnection *self)
{
  self->priv = G_TYPE_INSTANCE_GET_PRIVATE (self, TI_TYPE_WND_CONNECTION,
      TIWndConnectionPrivate);

  self->priv->handle_mapper = NULL;
  self->priv->dlg_get_handles = NULL;
  self->priv->dlg_request_channel = NULL;
  self->priv->glade_xml = NULL;
  self->priv->window = NULL;
  self->priv->notebook = NULL;
  self->priv->channel_wnds = g_ptr_array_new ();
  self->priv->page_aliasing = NULL;
  self->priv->page_contacts = NULL;
  self->priv->page_capabilities = NULL;
  self->priv->page_simple_presence = NULL;
  self->priv->page_presence = NULL;
  self->priv->page_properties = NULL;
}

static void
_ti_wnd_connection_on_invalidated (TpConnection *connection,
                                   guint domain,
                                   gint code,
                                   gchar *message,
                                   TIWndConnection *self)
{
  GtkLabel *label;

  label = GTK_LABEL (glade_xml_get_widget (self->priv->glade_xml,
        "label_status_value"));
  gtk_label_set_text (label, message);

  gtk_widget_set_sensitive (self->priv->button_connect, FALSE);
  gtk_widget_set_sensitive (self->priv->button_disconnect, FALSE);
  gtk_widget_set_sensitive (self->priv->button_request_handles, FALSE);
  gtk_widget_set_sensitive (self->priv->button_request_channel, FALSE);
  gtk_widget_set_sensitive (self->priv->button_refresh_channels_list, FALSE);
  gtk_widget_set_sensitive (self->priv->button_hold_handles, FALSE);

  gtk_list_store_clear (self->priv->interfaces_list);
  _ti_wnd_connection_remove_interface_pages (self);

  gtk_list_store_clear (self->priv->channels_list);
}

static void
_ti_wnd_connection_on_notify_status (TIWndConnection *self)
{
  GtkLabel *label;
  guint status, status_reason;
  gchar *str;

  g_object_get (self->priv->connection,
      "status", &status,
      "status-reason", &status_reason,
      NULL);

  label = GTK_LABEL (glade_xml_get_widget (self->priv->glade_xml,
        "label_status_value"));
  str = _get_status_string (status, status_reason);
  gtk_label_set_text (label, str);
  g_free (str);
}

TIWndConnection *
ti_wnd_connection_new (GtkWindow *parent,
                       TpConnection *connection)
{
  TIWndConnection *self = NULL;

  g_assert (connection != NULL);
  g_assert (parent != NULL);

  self = g_object_new (TI_TYPE_WND_CONNECTION,
      "connection", connection,
      "transient-for", parent,
      NULL);

  g_assert (self->priv->connection == connection);
  g_assert (self->priv->parent == parent);

  return self;
}

/**
 * Setup First Page - Helper Function
 */
static void
_ti_wnd_connection_setup_first_page (TIWndConnection *self)
{
  TIWndConnectionPrivate *priv = self->priv;
  GtkWidget *first_page;

  priv->notebook = GTK_NOTEBOOK (glade_xml_get_widget (priv->glade_xml,
        "notebook"));
  g_assert (GTK_IS_NOTEBOOK (priv->notebook));

  first_page = gtk_notebook_get_nth_page (priv->notebook, 0);
  gtk_notebook_set_tab_label_text (priv->notebook, first_page, "Connection");

  priv->interfaces_list = gtk_list_store_new (1, G_TYPE_STRING);
  _ti_wnd_connection_build_interfaces_treeview (self);

  priv->channels_list = gtk_list_store_new (4,
      G_TYPE_STRING, G_TYPE_STRING, G_TYPE_STRING, G_TYPE_STRING);
  _ti_wnd_connection_build_channels_treeview (self);

  priv->button_connect = glade_xml_get_widget (priv->glade_xml,
      "button_connect");
  g_assert (GTK_IS_BUTTON (priv->button_connect));
  g_signal_connect_swapped (priv->button_connect, "clicked",
      G_CALLBACK (_ti_wnd_connection_connect), self);

  priv->button_disconnect = glade_xml_get_widget (priv->glade_xml,
      "button_disconnect");
  g_assert (GTK_IS_BUTTON (priv->button_disconnect));
  g_signal_connect_swapped (priv->button_disconnect, "clicked",
      G_CALLBACK (_ti_wnd_connection_disconnect), self);

  priv->button_refresh_channels_list = glade_xml_get_widget (priv->glade_xml,
      "button_refresh_channels_list");
  g_assert (GTK_IS_BUTTON (priv->button_refresh_channels_list));
  g_signal_connect_swapped (priv->button_refresh_channels_list, "clicked",
      G_CALLBACK (_ti_wnd_connection_refresh_channels_list), self);

  priv->button_request_channel = glade_xml_get_widget (priv->glade_xml,
      "button_request_channel");
  g_assert (GTK_IS_BUTTON (priv->button_request_channel));
  g_signal_connect_swapped (priv->button_request_channel, "clicked",
      G_CALLBACK (_ti_wnd_connection_request_channel), self);

  priv->button_request_handles = glade_xml_get_widget (priv->glade_xml,
      "button_request_handles");
  g_assert (GTK_IS_BUTTON (priv->button_request_handles));
  g_signal_connect_swapped (GTK_BUTTON (priv->button_request_handles),
      "clicked", G_CALLBACK (_ti_wnd_connection_request_handles), self);

  priv->button_hold_handles = glade_xml_get_widget (priv->glade_xml,
      "button_hold_handles");
  g_assert (GTK_IS_BUTTON (priv->button_hold_handles));
  g_signal_connect_swapped (GTK_BUTTON (priv->button_hold_handles),
      "clicked", G_CALLBACK (_ti_wnd_connection_hold_handles), self);

  priv->button_release_handles = glade_xml_get_widget (priv->glade_xml,
      "button_release_handles");
  g_assert (GTK_IS_BUTTON (priv->button_release_handles));
  g_signal_connect_swapped (GTK_BUTTON (priv->button_release_handles),
      "clicked", G_CALLBACK (_ti_wnd_connection_release_selected_handles),
      self);

  priv->button_inspect_handles = glade_xml_get_widget (priv->glade_xml,
      "button_inspect_handles");
  g_assert (GTK_IS_BUTTON (priv->button_inspect_handles));
  g_signal_connect_swapped (GTK_BUTTON (priv->button_inspect_handles),
      "clicked", G_CALLBACK (_ti_wnd_connection_inspect_selected_handles),
      self);
}

static void
_ti_wnd_connection_setup_acquired_handles_list (TIWndConnection *self)
{
  TIWndConnectionPrivate *priv = self->priv;
  GtkWidget *treeview_handles;
  GtkCellRenderer *renderer;
  GtkTreeViewColumn *column;

  treeview_handles = glade_xml_get_widget (priv->glade_xml, "treeview_handles");
  gtk_tree_view_set_model (GTK_TREE_VIEW(treeview_handles),
      GTK_TREE_MODEL (ti_handle_mapper_get_handles_list_store (
          priv->handle_mapper)));

  renderer = gtk_cell_renderer_text_new ();

  column = gtk_tree_view_column_new_with_attributes ("Handle", renderer,
      "text", TI_HANDLES_LIST_COLUMN_NUMBER,
      NULL);
  gtk_tree_view_append_column (GTK_TREE_VIEW (treeview_handles), column);

  column = gtk_tree_view_column_new_with_attributes ("Type", renderer,
      "text", TI_HANDLES_LIST_COLUMN_TYPE,
      NULL);
  gtk_tree_view_append_column (GTK_TREE_VIEW (treeview_handles), column);

  column = gtk_tree_view_column_new_with_attributes ("Name", renderer,
      "text", TI_HANDLES_LIST_COLUMN_NAME,
      NULL);
  gtk_tree_view_append_column (GTK_TREE_VIEW (treeview_handles), column);

  priv->acquired_handles_selection = gtk_tree_view_get_selection (
      GTK_TREE_VIEW (treeview_handles));
  gtk_tree_selection_set_mode (priv->acquired_handles_selection,
      GTK_SELECTION_MULTIPLE);

  g_signal_connect_swapped (priv->acquired_handles_selection, "changed",
      G_CALLBACK (_ti_wnd_connection_handles_selection_changed), self);
}

static void
_ti_wnd_connection_build_interfaces_treeview (TIWndConnection *self)
{
  TIWndConnectionPrivate *priv = self->priv;
  GtkWidget *treeview_interfaces;
  GtkCellRenderer *renderer;
  GtkTreeViewColumn *column;
  GtkTreeSelection *tree_selection;

  treeview_interfaces = glade_xml_get_widget (priv->glade_xml,
      "treeview_interfaces");
  gtk_tree_view_set_model (GTK_TREE_VIEW (treeview_interfaces),
      GTK_TREE_MODEL (priv->interfaces_list));

  renderer = gtk_cell_renderer_text_new ();
  column = gtk_tree_view_column_new_with_attributes ("Interface", renderer,
      "text", 0,
      NULL);

  gtk_tree_view_append_column (GTK_TREE_VIEW (treeview_interfaces), column);

  tree_selection = gtk_tree_view_get_selection (GTK_TREE_VIEW (
        treeview_interfaces));
  gtk_tree_selection_set_mode (tree_selection, GTK_SELECTION_NONE);
}

/**
 * Build the GtkTreeView for the channels list.
 */
static void
_ti_wnd_connection_build_channels_treeview (TIWndConnection *self)
{
  GtkWidget *treeview_channels;
  GtkCellRenderer *renderer;
  GtkTreeViewColumn *column;

  treeview_channels = glade_xml_get_widget (self->priv->glade_xml,
      "treeview_channels");
  g_signal_connect_swapped (treeview_channels, "row-activated",
      G_CALLBACK (_ti_wnd_connection_show_selected_channel), self);
  gtk_tree_view_set_model (GTK_TREE_VIEW (treeview_channels),
      GTK_TREE_MODEL (self->priv->channels_list));

  renderer = gtk_cell_renderer_text_new ();

  column = gtk_tree_view_column_new_with_attributes ("Obj. Path", renderer,
      "text", 0,
      NULL);
  gtk_tree_view_append_column (GTK_TREE_VIEW (treeview_channels), column);

  column = gtk_tree_view_column_new_with_attributes ("Type", renderer,
      "text", 1,
      NULL);
  gtk_tree_view_append_column (GTK_TREE_VIEW (treeview_channels), column);

  column = gtk_tree_view_column_new_with_attributes ("H. Type", renderer,
      "text", 2,
      NULL);
  gtk_tree_view_append_column (GTK_TREE_VIEW (treeview_channels), column);

  column = gtk_tree_view_column_new_with_attributes ("H. Number", renderer,
      "text", 3,
      NULL);
  gtk_tree_view_append_column (GTK_TREE_VIEW (treeview_channels), column);

  self->priv->channel_selection = gtk_tree_view_get_selection (
      GTK_TREE_VIEW (treeview_channels));
  gtk_tree_selection_set_mode (self->priv->channel_selection,
      GTK_SELECTION_SINGLE);
}

/**
 * Get Status String - Helper Function
 */
static gchar *
_get_status_string (guint status,
                    guint reason)
{
  switch (status)
    {
    case TP_CONNECTION_STATUS_CONNECTED:
      return g_strdup_printf ("%u - CONNECTED (reason %u)",
          status, reason);

    case TP_CONNECTION_STATUS_CONNECTING:
      return g_strdup_printf ("%u - CONNECTING (reason %u)",
          status, reason);

    case TP_CONNECTION_STATUS_DISCONNECTED:
      return g_strdup_printf ("%u - DISCONNECTED (reason %u)",
          status, reason);

    default:
      return g_strdup_printf ("%u - ??? (reason %u)",
          status, reason);
    }
}

void
ti_wnd_connection_show (TIWndConnection *self)
{
  gtk_widget_show_all (self->priv->window);
}

/**
 * On Delete Window
 * Called when the top window receives a "delete-event" signal.
 */
static gboolean
on_delete_window (TIWndConnection *self, GdkEvent  *event)
{
  gtk_widget_hide_all (self->priv->window);

  g_signal_emit (self, TI_WND_CONNECTION_GET_CLASS(self)->close_signal_id, 0,
      NULL);

  return TRUE;
}

static void
_ti_wnd_connection_connect (TIWndConnection *self)
{
  GError *error = NULL;

  if (!tp_cli_connection_run_connect (self->priv->connection, -1, &error,
        NULL))
    {
      g_printerr ("Connect(): %s\n", error->message);
      g_error_free (error);
      return;
    }

  gtk_widget_set_sensitive (self->priv->button_connect, FALSE);
  gtk_widget_set_sensitive (self->priv->button_disconnect, FALSE);

  gtk_widget_set_sensitive (self->priv->button_request_handles, FALSE);
  gtk_widget_set_sensitive (self->priv->button_request_channel, FALSE);
  gtk_widget_set_sensitive (self->priv->button_refresh_channels_list, FALSE);
}

static void
_ti_wnd_connection_disconnect (TIWndConnection *self)
{
  GError *error = NULL;

  if (!tp_cli_connection_run_disconnect (self->priv->connection, -1, &error,
        NULL))
    {
      g_printerr ("Connect(): %s\n", error->message);
      g_error_free (error);
      return;
    }

  gtk_widget_set_sensitive (self->priv->button_connect, FALSE);
  gtk_widget_set_sensitive (self->priv->button_disconnect, FALSE);

  gtk_widget_set_sensitive (self->priv->button_request_handles, FALSE);
  gtk_widget_set_sensitive (self->priv->button_request_channel, FALSE);
  gtk_widget_set_sensitive (self->priv->button_refresh_channels_list, FALSE);
}

static void
_ti_wnd_connection_on_connected (TIWndConnection *self)
{
  char **str_list;
  char **str_list_ptr;
  GtkTreeIter iter;

  gtk_widget_set_sensitive (self->priv->button_connect, FALSE);
  gtk_widget_set_sensitive (self->priv->button_disconnect, TRUE);
  gtk_widget_set_sensitive (self->priv->button_request_handles, TRUE);
  gtk_widget_set_sensitive (self->priv->button_request_channel, TRUE);
  gtk_widget_set_sensitive (self->priv->button_refresh_channels_list, TRUE);
  gtk_widget_set_sensitive (self->priv->button_hold_handles, TRUE);

  gtk_list_store_clear (self->priv->interfaces_list);

  g_object_get (self->priv->connection,
      "interfaces", &str_list,
      NULL);

  /* Fill the list. */
  for (str_list_ptr = str_list;
       str_list_ptr != NULL && *str_list_ptr != NULL;
       str_list_ptr++)
    {
      gtk_list_store_append (self->priv->interfaces_list, &iter);
      gtk_list_store_set (self->priv->interfaces_list, &iter, 0,
          *str_list_ptr, -1);

      _ti_wnd_connection_create_interface_page (self, *str_list_ptr);
    }

  g_strfreev (str_list);

  _ti_wnd_connection_refresh_channels_list (self);
}

static void
_ti_wnd_connection_on_new_channel (TpConnection *connection,
                                   const gchar *object_path,
                                   const gchar *channel_type,
                                   guint handle_type,
                                   guint handle,
                                   gboolean suppress_handler,
                                   gpointer unused,
                                   GObject *object)
{
  TIWndConnection *self = TI_WND_CONNECTION (object);
  GError *error = NULL;

  /* We don't popup channel windows for suppressed and/or ContactList
   * channels */
  if (!suppress_handler &&
      tp_strdiff (channel_type, TP_IFACE_CHANNEL_TYPE_CONTACT_LIST))
    {
      TIWndChannel *channel_window = NULL;
      TpChannel *channel;

      channel = tp_channel_new (connection, object_path, channel_type,
          handle_type, handle, &error);

      if (error != NULL)
        {
          g_printerr ("Unable to create channel in response to NewChannel: "
              "%s\n", error->message);
          g_error_free (error);
          return;
        }

      channel_window = ti_wnd_channel_new (GTK_WINDOW (self->priv->window),
          channel, self->priv->handle_mapper);

      g_object_unref (channel);

      if (channel_window != NULL)
        {
          g_ptr_array_add (self->priv->channel_wnds, channel_window);
          g_signal_connect_swapped (channel_window, "close-wnd",
              G_CALLBACK (_ti_wnd_connection_closed_channel_window), self);
          ti_wnd_channel_show (channel_window);
        }
    }

  _ti_wnd_connection_refresh_channels_list (self);
}

static void
_got_channels (TpConnection *conn,
               const GPtrArray *channels,
               const GError *error,
               gpointer unused,
               GObject *object)
{
  TIWndConnection *self = TI_WND_CONNECTION (object);
  guint i;

  gtk_list_store_clear (self->priv->channels_list);

  if (error != NULL)
    {
      g_printerr ("ListChannels(): %s\n", error->message);
      return;
    }

  for (i = 0; i < channels->len; i++)
    {
      GValueArray *channel = g_ptr_array_index (channels, i);
      const gchar *object_path = g_value_get_boxed (channel->values + 0);
      const gchar *channel_type = g_value_get_string (channel->values + 1);
      guint handle_type = g_value_get_uint (channel->values + 2);
      guint handle = g_value_get_uint (channel->values + 3);

      _ti_wnd_connection_add_channel_to_list (self, object_path, channel_type,
          handle_type, handle);
    }
}

static void
_ti_wnd_connection_refresh_channels_list (TIWndConnection *self)
{
  tp_cli_connection_call_list_channels (self->priv->connection, -1,
      _got_channels, NULL, NULL, (GObject *) self);
}

static gchar *
_get_small_channel_type (const gchar *type)
{
  if (g_str_equal (type, "org.freedesktop.Telepathy.Channel.Type.ContactList"))
    {
      return g_strdup ("ContactList");
    }
  else if (g_str_equal (type,
        "org.freedesktop.Telepathy.Channel.Type.ContactSearch"))
    {
      return g_strdup ("ContactSearch");
    }
  else if (g_str_equal (type,
        "org.freedesktop.Telepathy.Channel.Type.RoomList"))
    {
      return g_strdup ("RoomList");
    }
  else if (g_str_equal (type, "org.freedesktop.Telepathy.Channel.Type.Text"))
    {
      return g_strdup ("Text");
    }
  else
    {
      return g_strdup ("UNKNOWN");
    }
}

static void
_ti_wnd_connection_add_channel_to_list (TIWndConnection *self,
                                        const gchar *obj_path,
                                        const gchar *type,
                                        guint handle_type,
                                        guint handle_number)
{
  GtkTreeIter iter;
  gchar *small_chan_type;
  gchar *htype_str;
  gchar *hnumber_str;

  small_chan_type = _get_small_channel_type (type);
  htype_str = g_strdup_printf ("%u", handle_type);
  hnumber_str = g_strdup_printf ("%u", handle_number);

  gtk_list_store_append (self->priv->channels_list, &iter);
  gtk_list_store_set (self->priv->channels_list, &iter,
      0, obj_path,
      1, small_chan_type,
      2, htype_str,
      3, hnumber_str,
      -1);

  g_free (small_chan_type);
  g_free (htype_str);
  g_free (hnumber_str);
}

static void
_ti_wnd_connection_request_channel (TIWndConnection *self)
{
  gchar *type;
  guint handle_type;
  guint handle;
  gboolean suppress_handler;
  gchar *object_path;
  GError *error = NULL;
  TpChannel *channel;
  TIWndChannel *channel_window;

  if (!ti_dlg_request_channel_run (self->priv->dlg_request_channel, &type,
      &handle_type, &handle, &suppress_handler))
    goto finally;

  if (!tp_cli_connection_run_request_channel (self->priv->connection, -1,
        type, handle_type, handle, suppress_handler, &object_path, &error,
        NULL))
    goto finally;

  g_free (type);

  channel = tp_channel_new (self->priv->connection, object_path,
      NULL, TP_UNKNOWN_HANDLE_TYPE, 0, &error);

  g_free (object_path);

  if (channel == NULL)
    goto finally;

  if (!tp_channel_run_until_ready (channel, &error, NULL))
    {
      g_object_unref (channel);
      goto finally;
    }

  channel_window = ti_wnd_channel_new (GTK_WINDOW (self->priv->window),
      channel, self->priv->handle_mapper);

  g_object_unref (channel);

  if (channel_window == NULL)
    {
      g_printerr ("ti_wnd_channel_new failed\n");
      goto finally;
    }

  g_ptr_array_add (self->priv->channel_wnds, channel_window);
  g_signal_connect_swapped (channel_window, "close-wnd",
      G_CALLBACK (_ti_wnd_connection_closed_channel_window),
      self);
  ti_wnd_channel_show (channel_window);

finally:
  if (error != NULL)
    {
      g_printerr ("While requesting channel: %s\n", error->message);
      g_error_free (error);
    }
}

static void
_ti_wnd_connection_request_handles (TIWndConnection *self)
{
  guint type;
  gchar **names;
  GArray *handles;
  GError *error = NULL;
  guint i;
  guint handle;

  if (!ti_dlg_get_handles_run (self->priv->dlg_get_handles, "Request Handles",
        &type, &names))
    return;

  if (!tp_cli_connection_run_request_handles (self->priv->connection, -1, type,
        (const gchar **) names, &handles, &error, NULL))
    {
      g_strfreev (names);
      g_printerr ("RequestHandles(): %s\n", error->message);
      g_error_free (error);
      return;
    }

  for (i = 0; i < handles->len; i++)
    {
      handle = g_array_index (handles, guint, i);
      ti_handle_mapper_set_handle_name (self->priv->handle_mapper,
          type, handle, names[i]);
    }

  g_strfreev (names);
  g_array_free (handles, TRUE);
}

static void
_ti_wnd_connection_hold_handles (TIWndConnection *self)
{
  GArray *handles;
  guint handle_type;
  GError *error = NULL;

  if (!ti_dlg_hold_handles_run (self->priv->dlg_hold_handles, &handle_type,
        &handles))
    return;

  if (!tp_cli_connection_run_hold_handles (self->priv->connection, -1,
        handle_type, handles, &error, NULL))
    {
      g_printerr ("HoldHandles(): %s\n", error->message);
      g_error_free (error);
    }
  else
    {
      guint i;

      for (i = 0; i < handles->len; i++)
        {
          guint handle = g_array_index (handles, guint, i);

          ti_handle_mapper_set_handle (self->priv->handle_mapper, handle_type,
              handle);
        }

      g_array_free (handles, TRUE);
    }
}

static void
_ti_wnd_connection_closed_channel_window (TIWndConnection *self, TIWndChannel *wnd_channel)
{
  gboolean ok;

  ok = g_ptr_array_remove (self->priv->channel_wnds, wnd_channel);
  g_assert (ok);

  g_object_unref (wnd_channel);

  _ti_wnd_connection_refresh_channels_list (self);
}

/**
 * Show Selected Channel
 * Opens a window for the selected channel.
 */
static void
_ti_wnd_connection_show_selected_channel (TIWndConnection *self)
{
  gboolean ok;
  GtkTreeModel *model = NULL;
  GtkTreeIter iter;
  gchar *obj_path = NULL;
  TpChannel *channel = NULL;
  TIWndChannel *channel_window = NULL;
  GError *error = NULL;

  ok = gtk_tree_selection_get_selected (self->priv->channel_selection,
      &model, &iter);
  g_assert (ok);

  gtk_tree_model_get (model, &iter,
      0, &obj_path,
      -1);

  channel = tp_channel_new (self->priv->connection, obj_path,
      NULL, TP_UNKNOWN_HANDLE_TYPE, 0, &error);

  if (channel == NULL ||
      !tp_channel_run_until_ready (channel, &error, NULL))
    {
      g_printerr ("tp_channel_run_until_ready: %s\n", error->message);
      g_error_free (error);
      goto CLEAN_UP;
    }

  channel_window = ti_wnd_channel_new (GTK_WINDOW (self->priv->window),
      channel, self->priv->handle_mapper);

  g_object_unref (channel);

  if (channel_window == NULL)
      goto CLEAN_UP;

  g_ptr_array_add (self->priv->channel_wnds, channel_window);
  g_signal_connect_swapped (channel_window, "close-wnd",
      G_CALLBACK (_ti_wnd_connection_closed_channel_window), self);
  ti_wnd_channel_show (channel_window);

CLEAN_UP:
  g_free (obj_path);
}

static void
inspect_helper (TpConnection *connection,
                TpHandleType handle_type,
                GArray *handles,
                TIHandleMapper *handle_mapper)
{
  gchar **names, **iter;
  GError *error = NULL;

  if (handles->len == 0)
    return;

  if (!tp_cli_connection_run_inspect_handles (connection, -1, handle_type,
        handles, &names, &error, NULL))
    {
      g_printerr ("InspectHandles(): %s\n", error->message);
      g_error_free (error);
    }
  else
    {
      guint i;

      for (i = 0, iter = names; iter != NULL && *iter != NULL; i++, iter++)
        {
          TpHandle handle;

          if (i >= handles->len)
            {
              g_printerr ("InspectHandles() returned too many handles!\n");
            }

          handle = g_array_index (handles, guint, i);
          ti_handle_mapper_set_handle_name (handle_mapper, handle_type,
              handle, *iter);
        }

      g_strfreev (names);
    }
}

static void
_ti_wnd_connection_inspect_selected_handles (TIWndConnection *self)
{
  GArray *contact_handles = NULL;
  GArray *room_handles;
  GArray *list_handles;

  contact_handles = g_array_new (FALSE, FALSE, sizeof (guint));
  room_handles = g_array_new (FALSE, FALSE, sizeof (guint));
  list_handles = g_array_new (FALSE, FALSE, sizeof (guint));

  _ti_wnd_connection_get_selected_handles (self,
      contact_handles, room_handles, list_handles);

  inspect_helper (self->priv->connection, TP_HANDLE_TYPE_CONTACT,
      contact_handles, self->priv->handle_mapper);
  inspect_helper (self->priv->connection, TP_HANDLE_TYPE_ROOM,
      room_handles, self->priv->handle_mapper);
  inspect_helper (self->priv->connection, TP_HANDLE_TYPE_LIST,
      list_handles, self->priv->handle_mapper);

  g_array_free (contact_handles, TRUE);
  g_array_free (room_handles, TRUE);
  g_array_free (list_handles, TRUE);
}

static void
release_helper (TpConnection *connection,
                TpHandleType handle_type,
                GArray *handles,
                TIHandleMapper *handle_mapper)
{
  GError *error = NULL;
  guint i;

  if (handles->len == 0)
    return;

  for (i = 0; i < handles->len; i++)
    ti_handle_mapper_invalidate_handle (handle_mapper, handle_type,
        g_array_index (handles, guint, i));

  if (!tp_cli_connection_run_release_handles (connection, -1, handle_type,
        handles, &error, NULL))
    {
      g_printerr ("ReleaseHandles(): %s\n", error->message);
      g_error_free (error);
    }
}

static void
_ti_wnd_connection_release_selected_handles (TIWndConnection *self)
{
  GArray *contact_handles;
  GArray *room_handles;
  GArray *list_handles;

  contact_handles = g_array_new (FALSE, FALSE, sizeof (guint));
  room_handles = g_array_new (FALSE, FALSE, sizeof (guint));
  list_handles = g_array_new (FALSE, FALSE, sizeof (guint));

  _ti_wnd_connection_get_selected_handles (self,
      contact_handles, room_handles, list_handles);

  release_helper (self->priv->connection, TP_HANDLE_TYPE_CONTACT,
      contact_handles, self->priv->handle_mapper);
  release_helper (self->priv->connection, TP_HANDLE_TYPE_ROOM,
      room_handles, self->priv->handle_mapper);
  release_helper (self->priv->connection, TP_HANDLE_TYPE_LIST,
      list_handles, self->priv->handle_mapper);

  ti_remove_selected_elements (self->priv->acquired_handles_selection);

  g_array_free (contact_handles, TRUE);
  g_array_free (room_handles, TRUE);
  g_array_free (list_handles, TRUE);
}

static void
_ti_wnd_connection_handles_selection_changed (TIWndConnection *self)
{
  gint selection_count;

  selection_count = gtk_tree_selection_count_selected_rows (
      self->priv->acquired_handles_selection);

  gtk_widget_set_sensitive (self->priv->button_release_handles,
      (selection_count > 0));
  gtk_widget_set_sensitive (self->priv->button_inspect_handles,
      (selection_count > 0));
}

static void
_ti_free_gtk_tree_path (gpointer data, gpointer user_data)
{
  gtk_tree_path_free ((GtkTreePath *) data);
}

static void
_ti_wnd_connection_get_selected_handles (TIWndConnection *self,
                                         GArray *contact_handles,
                                         GArray *room_handles,
                                         GArray *list_handles)
{
  GList *rows = NULL;
  GList *curr_row = NULL;
  GtkTreeIter iter;
  GtkTreePath *path = NULL;
  guint handle_number;
  guint handle_type;
  GtkTreeModel *tree_model = NULL;

  rows = gtk_tree_selection_get_selected_rows (
      self->priv->acquired_handles_selection, &tree_model);

  for (curr_row = rows; curr_row != NULL; curr_row = curr_row->next)
    {
      path = (GtkTreePath *) curr_row->data;

      gtk_tree_model_get_iter (tree_model, &iter, path);

      gtk_tree_model_get (tree_model, &iter,
                          TI_HANDLES_LIST_COLUMN_NUMBER, &handle_number,
                          TI_HANDLES_LIST_COLUMN_TYPE, &handle_type,
                          -1);

      if (handle_type == TP_HANDLE_TYPE_CONTACT)
        {
          g_array_append_val (contact_handles, handle_number);
        }
      else if (handle_type == TP_HANDLE_TYPE_ROOM)
        {
          g_array_append_val (room_handles, handle_number);
        }
      else if (handle_type == TP_HANDLE_TYPE_LIST)
        {
          g_array_append_val (list_handles, handle_number);
        }
      else
        {
          g_assert_not_reached ();
        }
    }

  if (rows != NULL)
    {
      g_list_foreach (rows, _ti_free_gtk_tree_path, NULL);
      g_list_free (rows);
    }
}

static void
_ti_wnd_connection_remove_interface_pages (TIWndConnection *self)
{
  if (self->priv->page_aliasing != NULL)
    {
      ti_page_detach_from_notebook (TI_PAGE (self->priv->page_aliasing));
    }

  if (self->priv->page_capabilities != NULL)
    {
      ti_page_detach_from_notebook (TI_PAGE (self->priv->page_capabilities));
    }

  if (self->priv->page_presence != NULL)
    {
      ti_page_detach_from_notebook (TI_PAGE (self->priv->page_presence));
    }

  if (self->priv->page_properties != NULL)
    {
      ti_page_detach_from_notebook (TI_PAGE (self->priv->page_properties));
    }
}

static void
_ti_wnd_connection_create_interface_page (TIWndConnection *self, gchar *interface_name)
{
  if (g_str_equal (interface_name, "org.freedesktop.Telepathy.Connection.Interface.Presence"))
    {
      if (self->priv->page_presence == NULL)
        {
          self->priv->page_presence = ti_page_presence_new (GTK_WINDOW (self->priv->window), self->priv->notebook, self->priv->connection, self->priv->handle_mapper);
        }
        else
        {
          ti_page_attach_to_notebook (TI_PAGE (self->priv->page_presence), self->priv->notebook);
        }
    }
  else if (g_str_equal (interface_name, "org.freedesktop.Telepathy.Connection.Interface.SimplePresence"))
    {
      if (self->priv->page_simple_presence == NULL)
        {
          self->priv->page_simple_presence = ti_page_simple_presence_new (GTK_WINDOW (self->priv->window), self->priv->notebook, self->priv->connection, self->priv->handle_mapper);
        }
        else
        {
          ti_page_attach_to_notebook (TI_PAGE (self->priv->page_simple_presence), self->priv->notebook);
        }
    }
  else if (g_str_equal (interface_name, "org.freedesktop.Telepathy.Connection.Interface.Aliasing"))
    {
      if (self->priv->page_aliasing == NULL)
        {
          self->priv->page_aliasing = ti_page_aliasing_new (GTK_WINDOW (self->priv->window), self->priv->notebook, self->priv->connection, self->priv->handle_mapper);
        }
      else
        {
          ti_page_attach_to_notebook (TI_PAGE (self->priv->page_aliasing), self->priv->notebook);
        }
    }
  else if (g_str_equal (interface_name, "org.freedesktop.Telepathy.Connection.Interface.Contacts"))
    {
      if (self->priv->page_contacts == NULL)
        {
          self->priv->page_contacts = ti_page_contacts_new (GTK_WINDOW (self->priv->window), self->priv->notebook, self->priv->connection, self->priv->handle_mapper);
        }
      else
        {
          ti_page_attach_to_notebook (TI_PAGE (self->priv->page_contacts), self->priv->notebook);
        }
    }
  else if (g_str_equal (interface_name, "org.freedesktop.Telepathy.Connection.Interface.Capabilities"))
    {
      if (self->priv->page_capabilities == NULL)
        {
          self->priv->page_capabilities = ti_page_capabilities_new (GTK_WINDOW (self->priv->window), self->priv->notebook, self->priv->connection, self->priv->handle_mapper);
        }
      else
        {
          ti_page_attach_to_notebook (TI_PAGE (self->priv->page_capabilities), self->priv->notebook);
        }
    }
  else if (g_str_equal (interface_name, "org.freedesktop.Telepathy.Properties"))
    {
      if (self->priv->page_properties == NULL)
        {
          self->priv->page_properties = ti_page_properties_new (self->priv->notebook,
                TP_PROXY (self->priv->connection));
        }
      else
        {
          ti_page_attach_to_notebook (TI_PAGE (self->priv->page_properties), self->priv->notebook);
        }
    }
  else
    {
      /* TODO: Add more interface pages if that's the case */
      g_debug ("Unrecognized interface: \"%s\"", interface_name);
    }
}
