/*
 * MediaSignalling page for channel notebook.
 *
 * Copyright (C) 2008 Collabora Ltd. <http://www.collabora.co.uk/>
 * Copyright (C) 2008 Nokia Corporation
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public License
 * along with this library; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA
 */

#include "config.h"

#include "page-media-signalling.h"

#include <glade/glade.h>

#include "extensions/stream-engine.h"
#include "page-priv.h"


struct _TIPageMediaSignallingClass {
    TIPageClass parent;
    gpointer priv;
};


struct _TIPageMediaSignallingPrivate {
    TpChannel *channel;
    GtkButton *se_button;
    TIStreamEngine *stream_engine;
};


G_DEFINE_TYPE (TIPageMediaSignalling, ti_page_media_signalling, TI_TYPE_PAGE);

enum {
    PROP_CHANNEL = 1
};


static void
ti_page_media_signalling_init (TIPageMediaSignalling *self)
{
  self->priv = G_TYPE_INSTANCE_GET_PRIVATE (self,
      TI_TYPE_PAGE_MEDIA_SIGNALLING, TIPageMediaSignallingPrivate);
}


static void
channel_handled (TIStreamEngine *se,
                 const GError *error,
                 gpointer unused G_GNUC_UNUSED,
                 GObject *object)
{
  TIPageMediaSignalling *self = TI_PAGE_MEDIA_SIGNALLING (object);

  if (error != NULL)
    {
      g_debug ("%p: channel not handled: %s", self, error->message);
      return;
    }

  g_debug ("%p: channel handled", self);

  if (self->priv->stream_engine != NULL)
    g_object_unref (self->priv->stream_engine);

  self->priv->stream_engine = g_object_ref (se);
}


static void
se_button_clicked (TIPageMediaSignalling *self)
{
  TpDBusDaemon *dbus;
  TpProxy *connection;
  gchar *channel_path;
  gchar *channel_type;
  guint handle_type;
  guint handle;
  TIStreamEngine *se;

  g_object_get (self->priv->channel,
      "dbus-daemon", &dbus,
      "connection", &connection,
      "object-path", &channel_path,
      "channel-type", &channel_type,
      "handle-type", &handle_type,
      "handle", &handle,
      NULL);

  se = ti_stream_engine_new (dbus);

  ti_cli_channel_handler_call_handle_channel (se, -1,
      connection->bus_name, connection->object_path,
      channel_type, channel_path, handle_type, handle,
      channel_handled, NULL, NULL, (GObject *) self);

  g_free (channel_path);
  g_free (channel_type);
  g_object_unref (connection);
  g_object_unref (se);
  g_object_unref (dbus);
}


static void
ti_page_media_signalling_dispose (GObject *object)
{
  TIPageMediaSignalling *self = TI_PAGE_MEDIA_SIGNALLING (object);

  if (self->priv->channel != NULL)
    {
      g_object_unref (self->priv->channel);
      self->priv->channel = NULL;
    }

  if (self->priv->stream_engine != NULL)
    {
      g_object_unref (self->priv->stream_engine);
      self->priv->stream_engine = NULL;
    }

  if (self->priv->se_button != NULL)
    {
      g_signal_handlers_disconnect_by_func (self->priv->se_button,
          G_CALLBACK (se_button_clicked), self);
    }

  G_OBJECT_CLASS (ti_page_media_signalling_parent_class)->dispose (object);
}


static void
ti_page_media_signalling_get_property (GObject *object,
                                     guint property_id,
                                     GValue *value,
                                     GParamSpec *pspec)
{
  TIPageMediaSignalling *self = TI_PAGE_MEDIA_SIGNALLING (object);

  switch (property_id)
    {
    case PROP_CHANNEL:
      g_value_set_object (value, self->priv->channel);
      break;
    default:
      G_OBJECT_WARN_INVALID_PROPERTY_ID (object, property_id, pspec);
    }
}


static void
ti_page_media_signalling_set_property (GObject *object,
                                     guint property_id,
                                     const GValue *value,
                                     GParamSpec *pspec)
{
  TIPageMediaSignalling *self = TI_PAGE_MEDIA_SIGNALLING (object);

  switch (property_id)
    {
    case PROP_CHANNEL:
      g_return_if_fail (self->priv->channel == NULL);
      self->priv->channel = TP_CHANNEL (g_value_dup_object (value));
      break;
    default:
      G_OBJECT_WARN_INVALID_PROPERTY_ID (object, property_id, pspec);
    }
}


static void
ti_page_media_signalling_setup_page (TIPage *page,
                                   GladeXML *glade_xml)
{
  TIPageMediaSignalling *self = TI_PAGE_MEDIA_SIGNALLING (page);

  self->priv->se_button = GTK_BUTTON (glade_xml_get_widget (glade_xml,
        "se_button"));
  g_assert (self->priv->se_button != NULL);
  g_signal_connect_swapped (self->priv->se_button, "clicked",
      G_CALLBACK (se_button_clicked), self);
}


static void
ti_page_media_signalling_class_init (TIPageMediaSignallingClass *klass)
{
  GObjectClass *object_class = G_OBJECT_CLASS (klass);
  TIPageClass *page_class = TI_PAGE_CLASS (klass);
  GParamSpec *param_spec;

  object_class->dispose = ti_page_media_signalling_dispose;
  object_class->get_property = ti_page_media_signalling_get_property;
  object_class->set_property = ti_page_media_signalling_set_property;

  page_class->setup_page = ti_page_media_signalling_setup_page;

  g_type_class_add_private (klass, sizeof (TIPageMediaSignallingPrivate));

  param_spec = g_param_spec_object ("channel", "TpChannel",
      "This page's associated channel", TP_TYPE_CHANNEL,
      G_PARAM_CONSTRUCT_ONLY | G_PARAM_READWRITE |
      G_PARAM_STATIC_NAME | G_PARAM_STATIC_BLURB | G_PARAM_STATIC_NICK);
  g_object_class_install_property (object_class, PROP_CHANNEL, param_spec);
}


TIPageMediaSignalling *
ti_page_media_signalling_new (GtkNotebook *parent_notebook,
                            TpChannel *channel)
{
  TIPage *page = NULL;

  g_return_val_if_fail (channel != NULL, NULL);

  page = g_object_new (TI_TYPE_PAGE_MEDIA_SIGNALLING,
      "channel", channel,
      NULL);

  _ti_page_new (&page, parent_notebook, "page-media-signalling.xml");

  return TI_PAGE_MEDIA_SIGNALLING (page);
}
