/* Ruby Video4Linux Extension Library
 * Copyright (C) 2004 Paulo Matias
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include "ruby.h"
#include <stdio.h>
#include <fcntl.h>
#include <unistd.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <sys/ioctl.h>
#include <sys/mman.h>
#include <linux/videodev.h>

typedef enum {
  IO_METHOD_READ,
  IO_METHOD_MMAP,
} io_method;

typedef struct _v4ldata
{
	
  struct video_capability vid_cap;
  struct video_mbuf vid_buf;
  struct video_mmap vid_mmap;
  struct video_window vid_win;
  struct video_channel vid_chan;
  struct video_picture vid_pic;
  struct video_tuner vid_tuner;

  unsigned char *map;
  
  int fd;
  io_method io;
  
} V4ldata;

VALUE cV4l;

static void v4l_free(V4ldata *ptr)
{
  if(!ptr) return;
  if(ptr->io == IO_METHOD_MMAP)
   munmap(ptr->map, ptr->vid_buf.size);
  close(ptr->fd);
  free(ptr);
}

static void v4l_alloc_map(V4ldata *ptr)
{
 unsigned long imgsz = ptr->vid_win.width * ptr->vid_win.height * 3;
 if(ptr->io != IO_METHOD_READ)
   return;
 if((unsigned char *)-1 != (unsigned char *)ptr->map)
   free(ptr->map);
 if(!(ptr->map = (unsigned char *)malloc(imgsz)))
   rb_raise(rb_eException, "Out of memory!");
}

VALUE v4l_new(VALUE class, VALUE device)
{
	
  VALUE argv[1], tdata;
  V4ldata *ptr = ALLOC(V4ldata);
  char *sdevice = STR2CSTR(device);
  
  ptr->fd = open(sdevice, O_RDWR /* required */ | O_NONBLOCK, 0);
  if(ptr->fd == -1)
    rb_raise(rb_eException, "Couldn't open %s.", sdevice);
  
  if(ioctl(ptr->fd, VIDIOCGCAP, &ptr->vid_cap) == -1)
  {
    perror("VIDIOCGCAP");
    rb_raise(rb_eException, "VIDIOCGCAP");
  }
  
  if(!(ptr->vid_cap.type & VID_TYPE_CAPTURE))
    rb_raise(rb_eException, "%s is no video capture device.", sdevice);

  ptr->map = (unsigned char *)-1;
  
  if(ioctl(ptr->fd, VIDIOCGMBUF, &ptr->vid_buf) == -1)
    ptr->io = IO_METHOD_READ;
  else
  {
    ptr->io = IO_METHOD_MMAP;
    ptr->map = (unsigned char *)mmap(0, ptr->vid_buf.size, PROT_READ|PROT_WRITE, MAP_SHARED, ptr->fd, 0);
    if((unsigned char *)-1 == (unsigned char *)ptr->map)
    {
      perror("mmap()");
      rb_warn("mmap() failed: falling back to read() method");
      ptr->io = IO_METHOD_READ;
    }
  }

  if(ioctl(ptr->fd, VIDIOCGWIN, &ptr->vid_win) == -1)
  {
   perror("VIDIOCGWIN");
   rb_raise(rb_eException, "VIDIOCGWIN");
  }
  
  v4l_alloc_map(ptr);
  
  if(ioctl(ptr->fd, VIDIOCGCHAN, &(ptr->vid_chan)) == -1)
  {
    perror("VIDIOCGCHAN");
    rb_raise(rb_eException, "VIDIOCGCHAN");
  }
  ptr->vid_chan.channel = 0;
  ptr->vid_chan.norm = VIDEO_MODE_NTSC;
  if(ioctl(ptr->fd, VIDIOCSCHAN, &(ptr->vid_chan)) == -1)
  {
    perror("VIDIOCSCHAN");
    rb_raise(rb_eException, "VIDIOCSCHAN");
  }
  
  ptr->vid_mmap.format = VIDEO_PALETTE_RGB24;
  
  if(ioctl(ptr->fd, VIDIOCGPICT, &(ptr->vid_pic)) == -1)
  {
    perror("VIDIOCGPICT");
    rb_raise(rb_eException, "VIDIOCGPICT");
  }
  ptr->vid_pic.palette = VIDEO_PALETTE_RGB24;
  if(ioctl(ptr->fd, VIDIOCSPICT, &(ptr->vid_pic)) == -1)
  {
    perror("VIDIOCSPICT");
    rb_raise(rb_eException, "VIDIOCSPICT");
  }

  if(ioctl(ptr->fd, VIDIOCGTUNER, &(ptr->vid_tuner)) == -1)
    perror("VIDIOCGTUNER");
  ptr->vid_tuner.mode = VIDEO_MODE_NTSC;
  if(ioctl(ptr->fd, VIDIOCSTUNER, &(ptr->vid_tuner)) == -1)
    perror("VIDIOCSTUNER");
  
  tdata = Data_Wrap_Struct(class, 0, v4l_free, ptr);
  rb_obj_call_init(tdata, 0, argv);
  
  return tdata;
  
}

VALUE v4l_get_name(VALUE self)
{
  V4ldata *ptr;
  Data_Get_Struct(self, V4ldata, ptr);
  if(!ptr) return Qnil;
  return rb_str_new2(ptr->vid_cap.name);
}

VALUE v4l_get_width(VALUE self)
{
  V4ldata *ptr;
  Data_Get_Struct(self, V4ldata, ptr);
  if(!ptr) return Qnil;
  if(ioctl(ptr->fd, VIDIOCGWIN, &ptr->vid_win) == -1)
  {
   perror("VIDIOCGWIN");
   rb_raise(rb_eException, "VIDIOCGWIN");
  }
  return INT2NUM(ptr->vid_win.width);
}

VALUE v4l_set_width(VALUE self, VALUE width)
{
  V4ldata *ptr;
  int iwidth = NUM2INT(width);
  Data_Get_Struct(self, V4ldata, ptr);
  if(!ptr) return Qnil;
  if(iwidth > ptr->vid_cap.maxwidth)
    iwidth = ptr->vid_cap.maxwidth;
  if(iwidth < ptr->vid_cap.minwidth)
    iwidth = ptr->vid_cap.minwidth;
  if(ioctl(ptr->fd, VIDIOCGWIN, &ptr->vid_win) == -1)
  {
   perror("VIDIOCGWIN");
   rb_raise(rb_eException, "VIDIOCGWIN");
  }
  ptr->vid_win.width = iwidth;
  if(ioctl(ptr->fd, VIDIOCSWIN, &ptr->vid_win) == -1) {
   perror("VIDIOCSWIN");
   rb_raise(rb_eException, "VIDIOCSWIN");
  }
  v4l_alloc_map(ptr);
  return self;
}

VALUE v4l_get_height(VALUE self)
{
  V4ldata *ptr;
  Data_Get_Struct(self, V4ldata, ptr);
  if(!ptr) return Qnil;
  if(ioctl(ptr->fd, VIDIOCGWIN, &ptr->vid_win) == -1)
  {
   perror("VIDIOCGWIN");
   rb_raise(rb_eException, "VIDIOCGWIN");
  }
  return INT2NUM(ptr->vid_win.height);
}

VALUE v4l_set_height(VALUE self, VALUE height)
{
  V4ldata *ptr;
  int iheight = NUM2INT(height);
  Data_Get_Struct(self, V4ldata, ptr);
  if(!ptr) return Qnil;
  if(iheight > ptr->vid_cap.maxheight)
    iheight = ptr->vid_cap.maxheight;
  if(iheight < ptr->vid_cap.minheight)
    iheight = ptr->vid_cap.minheight;
  if(ioctl(ptr->fd, VIDIOCGWIN, &ptr->vid_win) == -1)
  {
   perror("VIDIOCGWIN");
   rb_raise(rb_eException, "VIDIOCGWIN");
  }
  ptr->vid_win.height = iheight;
  if(ioctl(ptr->fd, VIDIOCSWIN, &ptr->vid_win) == -1) {
   perror("VIDIOCSWIN");
   rb_raise(rb_eException, "VIDIOCSWIN");
  }
  v4l_alloc_map(ptr);
  return self;
}

VALUE v4l_get_channels(VALUE self)
{
  V4ldata *ptr;
  Data_Get_Struct(self, V4ldata, ptr);
  if(!ptr) return Qnil;
  int i;
  VALUE ary = rb_ary_new();
  for(i = 0; i < ptr->vid_cap.channels; i++)
  {
    ptr->vid_chan.channel = i;
    if(ioctl(ptr->fd, VIDIOCGCHAN, &(ptr->vid_chan)) == -1)
    {
      perror("VIDIOCGCHAN");
      rb_raise(rb_eException, "VIDIOCGCHAN");
    }
    rb_ary_push(ary, rb_str_new2(ptr->vid_chan.name));
  }
  return ary;
}

VALUE v4l_get_channel(VALUE self)
{
  V4ldata *ptr;
  Data_Get_Struct(self, V4ldata, ptr);
  if(!ptr) return Qnil;
  return INT2NUM(ptr->vid_chan.channel);
}

VALUE v4l_set_channel(VALUE self, VALUE channel)
{
  V4ldata *ptr;
  Data_Get_Struct(self, V4ldata, ptr);
  if(!ptr) return Qnil;
  ptr->vid_chan.channel = NUM2INT(channel);
  ptr->vid_chan.norm = VIDEO_MODE_NTSC;
  if(ioctl(ptr->fd, VIDIOCGCHAN, &(ptr->vid_chan)) == -1)
  {
    perror("VIDIOCGCHAN");
    rb_raise(rb_eException, "VIDIOCGCHAN");
  }
  if(ioctl(ptr->fd, VIDIOCSCHAN, &(ptr->vid_chan)) == -1)
  {
    perror("VIDIOCSCHAN");
    rb_raise(rb_eException, "VIDIOCSCHAN");
  }
  return self;
}

VALUE v4l_get_brightness(VALUE self)
{
  V4ldata *ptr;
  Data_Get_Struct(self, V4ldata, ptr);
  if(!ptr) return Qnil;
  if(ioctl(ptr->fd, VIDIOCGPICT, &(ptr->vid_pic)) == -1)
  {
    perror("VIDIOCGPICT");
    rb_raise(rb_eException, "VIDIOCGPICT");
  }
  return INT2NUM(ptr->vid_pic.brightness);
}

VALUE v4l_set_brightness(VALUE self, VALUE brightness)
{
  V4ldata *ptr;
  Data_Get_Struct(self, V4ldata, ptr);
  if(!ptr) return Qnil;
  if(ioctl(ptr->fd, VIDIOCGPICT, &(ptr->vid_pic)) == -1)
  {
    perror("VIDIOCGPICT");
    rb_raise(rb_eException, "VIDIOCGPICT");
  }
  ptr->vid_pic.brightness = NUM2INT(brightness);
  if(ioctl(ptr->fd, VIDIOCSPICT, &(ptr->vid_pic)) == -1)
  {
    perror("VIDIOCSPICT");
    rb_raise(rb_eException, "VIDIOCSPICT");
  }
  return self;
}

VALUE v4l_get_contrast(VALUE self)
{
  V4ldata *ptr;
  Data_Get_Struct(self, V4ldata, ptr);
  if(!ptr) return Qnil;
  if(ioctl(ptr->fd, VIDIOCGPICT, &(ptr->vid_pic)) == -1)
  {
    perror("VIDIOCGPICT");
    rb_raise(rb_eException, "VIDIOCGPICT");
  }
  return INT2NUM(ptr->vid_pic.contrast);
}

VALUE v4l_set_contrast(VALUE self, VALUE contrast)
{
  V4ldata *ptr;
  Data_Get_Struct(self, V4ldata, ptr);
  if(!ptr) return Qnil;
  if(ioctl(ptr->fd, VIDIOCGPICT, &(ptr->vid_pic)) == -1)
  {
    perror("VIDIOCGPICT");
    rb_raise(rb_eException, "VIDIOCGPICT");
  }
  ptr->vid_pic.contrast = NUM2INT(contrast);
  if(ioctl(ptr->fd, VIDIOCSPICT, &(ptr->vid_pic)) == -1)
  {
    perror("VIDIOCSPICT");
    rb_raise(rb_eException, "VIDIOCSPICT");
  }
  return self;
}

VALUE v4l_get_hue(VALUE self)
{
  V4ldata *ptr;
  Data_Get_Struct(self, V4ldata, ptr);
  if(!ptr) return Qnil;
  if(ioctl(ptr->fd, VIDIOCGPICT, &(ptr->vid_pic)) == -1)
  {
    perror("VIDIOCGPICT");
    rb_raise(rb_eException, "VIDIOCGPICT");
  }
  return INT2NUM(ptr->vid_pic.hue);
}

VALUE v4l_set_hue(VALUE self, VALUE hue)
{
  V4ldata *ptr;
  Data_Get_Struct(self, V4ldata, ptr);
  if(!ptr) return Qnil;
  if(ioctl(ptr->fd, VIDIOCGPICT, &(ptr->vid_pic)) == -1)
  {
    perror("VIDIOCGPICT");
    rb_raise(rb_eException, "VIDIOCGPICT");
  }
  ptr->vid_pic.hue = NUM2INT(hue);
  if(ioctl(ptr->fd, VIDIOCSPICT, &(ptr->vid_pic)) == -1)
  {
    perror("VIDIOCSPICT");
    rb_raise(rb_eException, "VIDIOCSPICT");
  }
  return self;
}

VALUE v4l_get_colour(VALUE self)
{
  V4ldata *ptr;
  Data_Get_Struct(self, V4ldata, ptr);
  if(!ptr) return Qnil;
  if(ioctl(ptr->fd, VIDIOCGPICT, &(ptr->vid_pic)) == -1)
  {
    perror("VIDIOCGPICT");
    rb_raise(rb_eException, "VIDIOCGPICT");
  }
  return INT2NUM(ptr->vid_pic.colour);
}

VALUE v4l_set_colour(VALUE self, VALUE colour)
{
  V4ldata *ptr;
  Data_Get_Struct(self, V4ldata, ptr);
  if(!ptr) return Qnil;
  if(ioctl(ptr->fd, VIDIOCGPICT, &(ptr->vid_pic)) == -1)
  {
    perror("VIDIOCGPICT");
    rb_raise(rb_eException, "VIDIOCGPICT");
  }
  ptr->vid_pic.colour = NUM2INT(colour);
  if(ioctl(ptr->fd, VIDIOCSPICT, &(ptr->vid_pic)) == -1)
  {
    perror("VIDIOCSPICT");
    rb_raise(rb_eException, "VIDIOCSPICT");
  }
  return self;
}

VALUE v4l_get_frame(VALUE self)
{
  
  V4ldata *ptr;
  Data_Get_Struct(self, V4ldata, ptr);
	
  if(!ptr) return Qnil;
  
  unsigned long i, imgsz = ptr->vid_win.width * ptr->vid_win.height * 3;
  unsigned char buf, *p = ptr->map;
  
  switch(ptr->io)
  {
    case IO_METHOD_MMAP:
      ptr->vid_mmap.frame = 0;
      ptr->vid_mmap.width = ptr->vid_win.width;
      ptr->vid_mmap.height = ptr->vid_win.height;
      if(ioctl(ptr->fd, VIDIOCMCAPTURE, &ptr->vid_mmap) == -1)
      {
        perror("VIDIOCMCAPTURE");
	rb_warn("VIDIOCMCAPTURE failed");
      }
      if(ioctl(ptr->fd, VIDIOCSYNC, &ptr->vid_mmap.frame) == -1) {
        perror("VIDIOCSYNC");
	rb_warn("VIDIOCSYNC failed");
      }
      break;
    case IO_METHOD_READ:
      while(read(ptr->fd, ptr->map, imgsz) <= 0);
      break;
  }
  
  /* Converts from BGR to RGB */
  i = ptr->vid_win.width * ptr->vid_win.height;
  while(0 < i)
  {
    buf = p[2]; p[2] = p[0]; p[0] = buf;
    p += 3;
    i--;
  }
  
  return rb_str_new(ptr->map, imgsz);
  
}

VALUE v4l_capture(VALUE self)
{
  if(!rb_block_given_p())
    rb_raise(rb_eArgError, "must take a block");
  while(1)
    rb_yield(v4l_get_frame(self));
  return self;
}

VALUE v4l_write_file(int argc, VALUE *argv, VALUE self)
{

  FILE *fp;
  V4ldata *ptr;
  unsigned char *map;
  Data_Get_Struct(self, V4ldata, ptr);

  if(!ptr) return Qnil;
	
  if(argc == 1)
    map = ptr->map;
  else if(argc == 2)
    map = STR2CSTR(argv[1]);
  else
    rb_raise(rb_eArgError, "usage: write_file(path, [data])");
  
  fp = fopen(STR2CSTR(argv[0]), "wb");
  fprintf(fp, "P6\n %d %d\n 255\n", ptr->vid_win.width, ptr->vid_win.height);
  fflush(fp);
  fwrite(map, ptr->vid_win.width * ptr->vid_win.height, 3, fp);
  fclose(fp);
  
  return self;
  
}

VALUE v4l_close(VALUE self)
{
  V4ldata *ptr;
  Data_Get_Struct(self, V4ldata, ptr);
  v4l_free(ptr);
  ptr = 0;
  return self;
}

VALUE v4l_is_closed(VALUE self)
{
  V4ldata *ptr;
  Data_Get_Struct(self, V4ldata, ptr);
  if(ptr)
    return Qfalse;
  return Qtrue;
}

void Init_V4l()
{
  cV4l = rb_define_class("V4l", rb_cObject);
  rb_define_singleton_method(cV4l, "new", v4l_new, 1);
  rb_define_method(cV4l, "name",  v4l_get_name, 0);
  rb_define_method(cV4l, "width",  v4l_get_width, 0);
  rb_define_method(cV4l, "width=", v4l_set_width, 1);
  rb_define_method(cV4l, "height",  v4l_get_height, 0);
  rb_define_method(cV4l, "height=", v4l_set_height, 1);
  rb_define_method(cV4l, "channels",  v4l_get_channels, 0);
  rb_define_method(cV4l, "channel",  v4l_get_channel, 0);
  rb_define_method(cV4l, "channel=", v4l_set_channel, 1);
  rb_define_method(cV4l, "brightness",  v4l_get_brightness, 0);
  rb_define_method(cV4l, "brightness=", v4l_set_brightness, 1);
  rb_define_method(cV4l, "contrast",  v4l_get_contrast, 0);
  rb_define_method(cV4l, "contrast=", v4l_set_contrast, 1);
  rb_define_method(cV4l, "hue",  v4l_get_hue, 0);
  rb_define_method(cV4l, "hue=", v4l_set_hue, 1);
  rb_define_method(cV4l, "colour",  v4l_get_colour, 0);
  rb_define_method(cV4l, "colour=", v4l_set_colour, 1);
  rb_define_method(cV4l, "get_frame", v4l_get_frame, 0);
  rb_define_method(cV4l, "capture", v4l_capture, 0);
  rb_define_method(cV4l, "write_file", v4l_write_file, -1);
  rb_define_method(cV4l, "close", v4l_close, 0);
  rb_define_method(cV4l, "closed?", v4l_is_closed, 0);
}
