/*
 * Copyright (C) 2006 Sjoerd Simons <sjoerd@luon.net>
 * Copyright (C) 2003, 2004 Laurent Sansonetti <lrz@gnome.org>
 *
 * This file is part of Ruby/GStreamer.
 *
 * Ruby/GStreamer is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * Ruby/GStreamer is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with Ruby/GStreamer; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
 */

#include "rbgst.h"

/* Class: Gst::StaticPadTemplate
 * Describe the media type of a Gst::Pad.  
 */

/*
 * Method: name
 *
 * Gets the name of the pad template.
 * This overrides Gst::Object#name.
 *
 * Returns: the name of the pad template.
 */
static VALUE
rb_gst_staticpadtemplate_get_name (VALUE self)
{
	/* 
	 *  We can't call Gst::Object#name since the real name
	 *  of the pad template is in the "name_template" field of 
	 *  GstStaticPadTemplate.
	 */
	GstStaticPadTemplate *pad = RGST_STATIC_PAD_TEMPLATE (self);
	return CSTR2RVAL (pad->name_template);
}

/* Method: presence
 * Returns: the pad's presence (see Gst::Pad::Presence).
 */
static VALUE
rb_gst_staticpadtemplate_get_presence (VALUE self)
{
	GstStaticPadTemplate *pad = RGST_STATIC_PAD_TEMPLATE (self);
	return GENUM2RVAL (pad->presence, GST_TYPE_PAD_PRESENCE);
}

/* Method: direction
 * Returns: the pad's direction (see Gst::Pad::Direction).
 */
static VALUE
rb_gst_staticpadtemplate_get_direction (VALUE self)
{
	GstStaticPadTemplate *pad = RGST_STATIC_PAD_TEMPLATE (self);
	return GENUM2RVAL (pad->direction, GST_TYPE_PAD_DIRECTION);
}

/*
 * Method: caps
 *
 * Gets the capabilities of the pad template element.
 *
 * Returns: a Gst::Caps object.
 */
static VALUE
rb_gst_staticpadtemplate_get_caps (VALUE self)
{
	return RGST_CAPS_NEW (gst_static_pad_template_get_caps (RGST_STATIC_PAD_TEMPLATE (self)));
}

/*
 * Method: each_caps { |caps| ... }
 *
 * Calls the block for each capability of the pad template, 
 * passing a reference to the Gst::Caps object as parameter.
 *
 * Returns: always nil.
 */
static VALUE
rb_gst_staticpadtemplate_each_caps (VALUE self)
{
	return rb_ary_yield (rb_gst_staticpadtemplate_get_caps (self));
}


void 
Init_gst_staticpadtemplate (void)
{
	VALUE c = G_DEF_CLASS (GST_TYPE_STATIC_PAD_TEMPLATE, "StaticPadTemplate", mGst);

	rb_define_method (c, "name", rb_gst_staticpadtemplate_get_name, 0);
	rb_define_method (c, "presence", rb_gst_staticpadtemplate_get_presence, 0);
	rb_define_method (c, "direction", rb_gst_staticpadtemplate_get_direction, 0);
	rb_define_method (c, "caps", rb_gst_staticpadtemplate_get_caps, 0);
	rb_define_method (c, "each_caps", rb_gst_staticpadtemplate_each_caps, 0);
}
