/*
 *   file: form_phone.c
 *  
 *  Copyright (C) Nicola Fragale <nicolafragale@libero.it>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include <gtk/gtk.h>
#include <glade/glade.h>

#include "libral.h"

#include "app.h"
#include "form.h"
#include "form_phone.h"
#include "app.h"
#include "groups_cb.h"
#include "models.h"
#include "types.h"
#include "utils.h"


struct _RubricaFormPhonePrivate {
  FormPhoneType type;

  GtkWidget* tel;
  GtkWidget* tree;
  GtkWidget* add;
  GtkWidget* del;
  GtkWidget* mod;

  GtkWidget* tel1;     //  home    or operator
  GtkWidget* tel2;     //  work    or green
  GtkWidget* tel3;     //  fax     
  GtkWidget* tel4;     //  mobile  or customer care
  GtkWidget* tel5;     //  other   
  
  gboolean dispose_has_run;
};


static void rubrica_form_phone_class_init (RubricaFormPhoneClass* klass);
static void rubrica_form_phone_init       (RubricaFormPhone* obj);

static void rubrica_form_phone_dispose    (RubricaFormPhone* obj);
static void rubrica_form_phone_finalize   (RubricaFormPhone* obj);


static gboolean extract_telephones  (GtkTreeModel *model, GtkTreePath *path,
				     GtkTreeIter *iter, gpointer data);


static void form_phone_extract_data (RubricaFormPhone* form, RCard *card);
static void form_phone_display_data (RubricaFormPhone* form, RCard *card);
static void form_phone_clean_form   (RubricaFormPhone* form);


static RubricaFormClass* parent_class;


GType
rubrica_form_phone_get_type()
{
  static GType form_phone_type = 0;
  
  if (!form_phone_type)
    {
      static const GTypeInfo form_phone_info =
	{
	  sizeof(RubricaFormPhoneClass),
	  NULL,
	  NULL,
	  (GClassInitFunc) rubrica_form_phone_class_init,
	  NULL,
	  NULL,
	  sizeof(RubricaFormPhone),
	  0,
	  (GInstanceInitFunc) rubrica_form_phone_init
	};

      form_phone_type = g_type_register_static (RUBRICA_FORM_TYPE, 
					       "RubricaFormPhone",
					       &form_phone_info, 0);
    }
  
  return form_phone_type;
}


static void
rubrica_form_phone_class_init(RubricaFormPhoneClass* klass)
{
  GObjectClass *class;
 
  class  = G_OBJECT_CLASS (klass);
  class->dispose  = (GObjectFinalizeFunc) rubrica_form_phone_dispose;
  class->finalize = (GObjectFinalizeFunc) rubrica_form_phone_finalize;

  parent_class = RUBRICA_FORM_CLASS(klass);
  parent_class->extract_data = (gpointer) form_phone_extract_data;
  parent_class->display_data = (gpointer) form_phone_display_data;
  parent_class->clean_form   = (gpointer) form_phone_clean_form;
}


static void
rubrica_form_phone_init(RubricaFormPhone* self)
{
  self->private = g_new(RubricaFormPhonePrivate, 1);
  
  self->private->type = FORM_PHONE_PERSONAL;
  self->private->tel  = NULL;
  self->private->tree = NULL;
  self->private->add  = NULL;
  self->private->mod  = NULL;
  self->private->del  = NULL;

  self->private->tel1 = NULL;
  self->private->tel2 = NULL;
  self->private->tel3 = NULL;
  self->private->tel4 = NULL;
  self->private->tel5 = NULL;

  self->private->dispose_has_run = FALSE;
}

 

static void 
rubrica_form_phone_dispose (RubricaFormPhone* self)
{
  g_return_if_fail(IS_RUBRICA_FORM(self));
  
  if (self->private->dispose_has_run)
    return;
  
  g_object_unref(self->private->tel);
  g_object_unref(self->private->tree);
  g_object_unref(self->private->add);
  g_object_unref(self->private->mod);
  g_object_unref(self->private->del);

  g_object_unref(self->private->tel1);
  g_object_unref(self->private->tel2);
  g_object_unref(self->private->tel3);
  g_object_unref(self->private->tel4);
  g_object_unref(self->private->tel5);
  
  self->private->dispose_has_run = TRUE;
}


static void 
rubrica_form_phone_finalize (RubricaFormPhone* self)
{
  g_return_if_fail(IS_RUBRICA_FORM(self));
  
  g_free(self->private->tel);
  g_free(self->private->tree);
  g_free(self->private->add);
  g_free(self->private->mod);
  g_free(self->private->del);

  g_free(self->private->tel1);
  g_free(self->private->tel2);
  g_free(self->private->tel3);
  g_free(self->private->tel4);
  g_free(self->private->tel5);

  self->private = NULL;
}



gboolean 
extract_telephones (GtkTreeModel *model, GtkTreePath *path,
		    GtkTreeIter *iter, gpointer data)
{
  RCard* card = (RCard*) data;
  RTelephone* tel = NULL;
  RTelephoneType ttype;
  gchar *number, *type;

  gtk_tree_model_get(model, iter, 
		     TELEPHONE_NUMBER_COLUMN, &number,
		     TELEPHONE_TYPE_COLUMN,   &type,
		     -1);
  ttype = r_telephone_lookup_str2enum(type);

  tel = r_telephone_new();
  g_object_set(tel, "telephone-number", number, "telephone-type", ttype, NULL);
  r_card_add_telephone(R_CARD(card), tel);

  return FALSE;
}


static void 
form_phone_extract_data (RubricaFormPhone* form, RCard *card)
{
  GtkTreeModel* model = NULL;

  g_return_if_fail(IS_R_CARD(card));

  model = gtk_tree_view_get_model(GTK_TREE_VIEW(form->private->tree));
  if (model)
    {
      gtk_tree_model_foreach(GTK_TREE_MODEL(model), extract_telephones, card);
      g_object_unref(model);
    }
}


static void 
form_phone_display_data (RubricaFormPhone* form, RCard *card)
{
  GtkTreeView* tree;
  GtkTreeModel *model;
  GtkTreeIter iter;
  gpointer data;

  /*   telephone model/view 
   */
  tree  = (GtkTreeView*) form->private->tree;
  model = gtk_tree_view_get_model(tree);

  data = r_card_get_telephone(R_CARD(card));
  for (; data; data = r_card_get_next_telephone(R_CARD(card)))
    {
      RTelephoneType ttype;
      gchar *number, *type;
      
      g_object_get(R_TELEPHONE(data), 
		   "telephone-number", &number,
		   "telephone-type",   &ttype,
		   NULL);
      type = r_telephone_lookup_enum2str(ttype);
      
      gtk_list_store_append(GTK_LIST_STORE(model), &iter);
      gtk_list_store_set(GTK_LIST_STORE(model),   &iter, 
			 TELEPHONE_NUMBER_COLUMN, number,
			 TELEPHONE_TYPE_COLUMN,   type,
			 -1);
    }
}


static void 
form_phone_clean_form   (RubricaFormPhone* form)
{
  GtkTreeModel *model;

  gtk_entry_set_text(GTK_ENTRY(form->private->tel), "");
  gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(form->private->tel1), TRUE);

  model = gtk_tree_view_get_model(GTK_TREE_VIEW(form->private->tree));
  gtk_list_store_clear(GTK_LIST_STORE(model));
}


/*  Callbacks
*/
void 
form_telephone_list_activated(GtkTreeSelection *selection, gpointer data)
{
  GtkTreeModel *model;
  GtkTreeIter iter;
  RubricaFormPhone *form = (RubricaFormPhone *) data;
  gchar *numb_str;
  gchar *type_str;
  gint type;

  model = gtk_tree_view_get_model(GTK_TREE_VIEW(form->private->tree));

  if (gtk_tree_selection_get_selected(selection, &model, &iter))
    {
      gtk_tree_model_get(model, &iter, 
			 TELEPHONE_NUMBER_COLUMN, &numb_str, 
			 TELEPHONE_TYPE_COLUMN,   &type_str,
			 -1);
      
      gtk_entry_set_text(GTK_ENTRY(form->private->tel), numb_str);
      type = r_telephone_lookup_str2enum(type_str);

      switch (type)
	{
	case R_TELEPHONE_HOME:
	case R_TELEPHONE_OPERATOR:
	  gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(form->private->tel1),
				       TRUE);
	  break;
	case R_TELEPHONE_WORK:
	case R_TELEPHONE_GREEN:
	  gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(form->private->tel2),
				       TRUE);
	  break;
	case R_TELEPHONE_FAX:
	  gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(form->private->tel3),
				       TRUE);
	  break;
	case R_TELEPHONE_CELLPHONE:
	case R_TELEPHONE_CUSTOMER_CARE:
	  gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(form->private->tel4),
				       TRUE);
	  break; 
	case R_TELEPHONE_OTHER:
	  gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(form->private->tel5),
				       TRUE);
	  break; 

	default:
	  break;
	}      
      
      gtk_widget_set_sensitive(form->private->mod, TRUE);
      gtk_widget_set_sensitive(form->private->del, TRUE);
      g_free(numb_str);
      g_free(type_str);
    }
}


void
form_tel_add_clicked (GtkButton *button, gpointer data)
{
  GtkTreeModel *model;
  GtkTreeIter iter;
  gchar *type_str = NULL;
  RubricaFormPhone *form = (RubricaFormPhone *) data;
  RTelephone* tel;
  RTelephoneType type = R_TELEPHONE_UNKNOWN;
  gchar* number;

  model = gtk_tree_view_get_model(GTK_TREE_VIEW(form->private->tree));

  tel    = r_telephone_new();
  number = g_strdup(gtk_entry_get_text(GTK_ENTRY(form->private->tel)));
  
  // home or operator telephone
  if (gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(form->private->tel1)))
    {
      if (form->private->type == FORM_PHONE_PERSONAL)
	{
	  type_str = "home";
	  type = R_TELEPHONE_HOME;
	}
      else
	{
	  type_str = "operator";
	  type = R_TELEPHONE_OPERATOR;
	}
    }
  
  // work or green telephone
  if (gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(form->private->tel2)))
    {
      if (form->private->type == FORM_PHONE_PERSONAL)
	{
	  type_str = "work";
	  type = R_TELEPHONE_WORK;
	}
      else
	{
	  type_str = "green";
	  type = R_TELEPHONE_GREEN;
	}
    }
  
  // fax
  if (gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(form->private->tel3)))
    {
      type_str = "fax";
      type = R_TELEPHONE_FAX;
    } 

  // mobile or customer care telephone
  if (gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(form->private->tel4)))
    {
      if (form->private->type == FORM_PHONE_PERSONAL)
	{
	  type_str = "cellphone";
	  type = R_TELEPHONE_CELLPHONE;
	}
      else
	{
	  type_str = "customer care";
	  type = R_TELEPHONE_CUSTOMER_CARE;
	}
    }
  
  // Other telephone
  if (gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(form->private->tel5)))
    {
      type = R_TELEPHONE_OTHER;
      type_str = "other";
    }

  g_object_set(tel, "telephone-number", number, "telephone-type", type, NULL);

  gtk_list_store_append(GTK_LIST_STORE(model), &iter);
  gtk_list_store_set(GTK_LIST_STORE(model), &iter, 
		     TELEPHONE_NUMBER_COLUMN, number,
		     TELEPHONE_TYPE_COLUMN, type_str,
		     -1);

  gtk_entry_set_text(GTK_ENTRY(form->private->tel), "");
  gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(form->private->tel1), TRUE);
    
  gtk_widget_set_sensitive(GTK_WIDGET(form->private->mod), FALSE);
  gtk_widget_set_sensitive(GTK_WIDGET(form->private->del), FALSE);  
}


void
form_tel_mod_clicked (GtkButton *button, gpointer data)
{
  GtkTreeModel *model;
  GtkTreeIter iter;
  GtkTreeSelection *selection;
  RubricaFormPhone *form = (RubricaFormPhone *) data;
  RTelephone* tel;
  RTelephoneType type = R_TELEPHONE_UNKNOWN;;
  gchar* number;
  gchar *type_str = NULL;

  model     = gtk_tree_view_get_model(GTK_TREE_VIEW(form->private->tree));
  selection = gtk_tree_view_get_selection(GTK_TREE_VIEW(form->private->tree));

  tel = r_telephone_new();
  number = g_strdup(gtk_entry_get_text(GTK_ENTRY(form->private->tel)));

  // home or operator telephone
  if (gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(form->private->tel1)))
    {
      if (form->private->type == FORM_PHONE_PERSONAL)
	{
	  type_str = "home";
	  type = R_TELEPHONE_HOME;
	}
      else
	{
	  type_str = "operator";
	  type = R_TELEPHONE_OPERATOR;
	}
    }
  
  // work or green telephone
  if (gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(form->private->tel2)))
    {
      if (form->private->type == FORM_PHONE_PERSONAL)
	{
	  type_str = "work";
	  type = R_TELEPHONE_WORK;
	}
      else
	{
	  type_str = "green";
	  type = R_TELEPHONE_GREEN;
	}
    }
  
  // fax
  if (gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(form->private->tel3)))
    {
      type_str = "fax";
      type = R_TELEPHONE_FAX;
    } 

  // mobile or customer care telephone
  if (gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(form->private->tel4)))
    {
      if (form->private->type == FORM_PHONE_PERSONAL)
	{
	  type_str = "cellphone";
	  type = R_TELEPHONE_CELLPHONE;
	}
      else
	{
	  type_str = "customer care";
	  type = R_TELEPHONE_CUSTOMER_CARE;
	}
    }
  
  // Other telephone
  if (gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(form->private->tel5)))
    {
      type = R_TELEPHONE_OTHER;
      type_str = "other";
    }

  g_object_set(tel, "telephone-number", number, "telephone-type", type, NULL);

  if (gtk_tree_selection_get_selected(selection, NULL, &iter))
  {
    gint i;
    GtkTreePath *path;
    
    path = gtk_tree_model_get_path(model, &iter);
    i    = gtk_tree_path_get_indices(path)[0];
    
    gtk_list_store_remove(GTK_LIST_STORE(model), &iter);
    gtk_tree_path_free(path);
    
    gtk_list_store_insert(GTK_LIST_STORE(model), &iter, i);
    gtk_list_store_set(GTK_LIST_STORE(model), &iter, 
		       TELEPHONE_NUMBER_COLUMN, number, 
		       TELEPHONE_TYPE_COLUMN, type_str,
		       -1);

    gtk_entry_set_text(GTK_ENTRY(form->private->tel),"");
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(form->private->tel1), TRUE);

    gtk_widget_set_sensitive(GTK_WIDGET(form->private->mod), FALSE);
    gtk_widget_set_sensitive(GTK_WIDGET(form->private->del), FALSE);   
  }  
}


void
form_tel_del_clicked (GtkButton *button, gpointer data)
{
  GtkTreeModel *model;
  GtkTreeIter iter;
  GtkTreeSelection *selection;
  RubricaFormPhone *form = (RubricaFormPhone *) data;

  model     = gtk_tree_view_get_model(GTK_TREE_VIEW(form->private->tree));
  selection = gtk_tree_view_get_selection(GTK_TREE_VIEW(form->private->tree));

  if (gtk_tree_selection_get_selected(selection, NULL, &iter))
  {
    gtk_list_store_remove(GTK_LIST_STORE(model), &iter);
    
    gtk_entry_set_text(GTK_ENTRY(form->private->tel),"");
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(form->private->tel1), TRUE);

    gtk_widget_set_sensitive(GTK_WIDGET(form->private->mod), FALSE);
    gtk_widget_set_sensitive(GTK_WIDGET(form->private->del), FALSE);   
  }  
}



/*   ***************************** Public *****************************
*/




/**
 * rubrica_form_phone_new
 *
 * create a #RubricaFormPhone
 *
 * Returns: a new #RubricaFormPhone
 */
RubricaFormPhone*
rubrica_form_phone_new (RubricaApp* app, GladeXML *gui, FormPhoneType type)
{
  RubricaFormPhone *form;
  GtkTreeModel     *model;
  GtkTreeSelection *selection;

  form = g_object_new(rubrica_form_phone_get_type(), NULL);
  
  form->private->type = type;
  form->private->tel  = glade_xml_get_widget(gui, "tel");
  form->private->tree = glade_xml_get_widget(gui, "tel_tree");
  form->private->add  = glade_xml_get_widget(gui, "tel_add");
  form->private->del  = glade_xml_get_widget(gui, "tel_del");
  form->private->mod  = glade_xml_get_widget(gui, "tel_mod");

  form->private->tel3 = glade_xml_get_widget(gui, "fax");
  form->private->tel5 = glade_xml_get_widget(gui, "other");

  if (type == FORM_PHONE_PERSONAL)
    {
      form->private->tel1 = glade_xml_get_widget(gui, "home");
      form->private->tel2 = glade_xml_get_widget(gui, "work");
      form->private->tel4 = glade_xml_get_widget(gui, "cell");
    }
  else
    {
      form->private->tel1 = glade_xml_get_widget(gui, "operator");
      form->private->tel2 = glade_xml_get_widget(gui, "green");
      form->private->tel4 = glade_xml_get_widget(gui, "ccare");
    }
  
  gtk_widget_set_sensitive(GTK_WIDGET(form->private->mod), FALSE);
  gtk_widget_set_sensitive(GTK_WIDGET(form->private->del), FALSE);

  /*   telephone model/view 
   */
  model = rubrica_model_new(TELEPHONE_MODEL);
  gtk_tree_view_set_model(GTK_TREE_VIEW(form->private->tree), 
			  GTK_TREE_MODEL(model));
  selection = gtk_tree_view_get_selection(GTK_TREE_VIEW(form->private->tree));
  gtk_tree_selection_set_mode(selection, GTK_SELECTION_SINGLE); 

  rubrica_model_add_columns(GTK_TREE_VIEW(form->private->tree),
			    TELEPHONE_COLUMNS);
  
  g_signal_connect(G_OBJECT(selection), "changed", 
		   G_CALLBACK(form_telephone_list_activated), form);
  g_object_unref(model);


  /*    telephone buttons 
  */
  g_signal_connect (G_OBJECT (form->private->add), "clicked", 
		    G_CALLBACK (form_tel_add_clicked), form); 
  g_signal_connect (G_OBJECT (form->private->mod), "clicked", 
		    G_CALLBACK (form_tel_mod_clicked), form); 
  g_signal_connect (G_OBJECT (form->private->del), "clicked", 
		    G_CALLBACK (form_tel_del_clicked), form); 

  
  return form;
}


/**
 * rubrica_form_phone_free
 * @form: a #RubricaFormPhone
 * 
 * free the #RubricaFormPhone
 */
void
rubrica_form_phone_free(RubricaFormPhone* form)
{
  g_return_if_fail(IS_RUBRICA_FORM(form));

  g_object_unref(form);   
}
