//plugincontainer.c:

/*
 *      Copyright (C) Philipp 'ph3-der-loewe' Schafft - 2011-2012
 *
 *  This file is part of libroar a part of RoarAudio,
 *  a cross-platform sound system for both, home and professional use.
 *  See README for details.
 *
 *  This file is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License version 3
 *  as published by the Free Software Foundation.
 *
 *  libroar is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this software; see the file COPYING.  If not, write to
 *  the Free Software Foundation, 51 Franklin Street, Fifth Floor,
 *  Boston, MA 02110-1301, USA.
 *
 *  NOTE for everyone want's to change something and send patches:
 *  read README and HACKING! There a addition information on
 *  the license of this document you need to read before you send
 *  any patches.
 *
 *  NOTE for uses of non-GPL (LGPL,...) software using libesd, libartsc
 *  or libpulse*:
 *  The libs libroaresd, libroararts and libroarpulse link this lib
 *  and are therefore GPL. Because of this it may be illigal to use
 *  them with any software that uses libesd, libartsc or libpulse*.
 */

#include "libroar.h"

#define MAX_PLUGINS 64

struct roar_plugincontainer {
 size_t refc;
 struct roar_dl_librarypara * default_para;
 struct roar_dl_lhandle * handle[MAX_PLUGINS];
 size_t deprefc[MAX_PLUGINS];
 size_t numhandles;
};

static struct roar_plugincontainer * _new_init(void) {
 struct roar_plugincontainer * ret = roar_mm_malloc(sizeof(struct roar_plugincontainer));
 if ( ret == NULL )
  return NULL;

 memset(ret, 0, sizeof(struct roar_plugincontainer));

 ret->refc = 1;

 return ret;
}

static struct roar_plugincontainer * roar_plugincontainer_new(struct roar_dl_librarypara * default_para) {
 struct roar_plugincontainer * ret = _new_init();
 int err;

 if ( ret == NULL )
  return NULL;

 if ( default_para != NULL ) {
  if ( roar_dl_para_ref(default_para) == -1 ) {
   err = roar_error;
   roar_plugincontainer_unref(ret);
   roar_err_set(err);
   return NULL;
  }
  ret->default_para = default_para;
 }

 return ret;
}

struct roar_plugincontainer * roar_plugincontainer_new_simple(const char * appname, const char * abiversion) {
 struct roar_plugincontainer * ret;
 struct roar_dl_librarypara * para = roar_dl_para_new(NULL, NULL, appname, abiversion);
 int err;

 ret = roar_plugincontainer_new(para);
 err = roar_error;

 roar_dl_para_unref(para);

 roar_err_set(err);
 return ret;
}

int roar_plugincontainer_ref(struct roar_plugincontainer * cont) {
 if ( cont == NULL ) {
  roar_err_set(ROAR_ERROR_FAULT);
  return -1;
 }

 cont->refc++;

 return 0;
}

int roar_plugincontainer_unref(struct roar_plugincontainer * cont) {
 size_t i;

 if ( cont == NULL ) {
  roar_err_set(ROAR_ERROR_FAULT);
  return -1;
 }

 cont->refc--;

 if ( cont->refc )
  return 0;

 while (cont->numhandles) {
  for (i = 0; i < MAX_PLUGINS; i++) {
   if ( cont->handle[i] == NULL )
    continue;

   // skip plugins in use by others. We will unload them in a later loop.
   if ( cont->deprefc[i] )
    continue;

   roar_dl_close(cont->handle[i]);
   cont->handle[i] = NULL;
   cont->numhandles--;
  }
 }

 if ( cont->default_para != NULL )
  roar_dl_para_unref(cont->default_para);

 roar_mm_free(cont);

 return 0;
}

// plugin loading and unloading:
int roar_plugincontainer_load(struct roar_plugincontainer * cont, const char * name, struct roar_dl_librarypara * para) {
 ssize_t idx = -1;
 size_t i;

 if ( cont == NULL || name == NULL ) {
  roar_err_set(ROAR_ERROR_FAULT);
  return -1;
 }

 if ( para == NULL )
  para = cont->default_para;
 // we do not need to _ref(para) here. This will be done by the roardl open function.


 // search for a free index.
 if ( cont->numhandles == MAX_PLUGINS ) {
  // return early if we are full.
  roar_err_set(ROAR_ERROR_NOSPC);
  return -1;
 }
 for (i = 0; i < MAX_PLUGINS; i++) {
  if ( cont->handle[i] == NULL ) {
   idx = i;
   break;
  }
 }

 if ( idx == -1 ) {
  roar_err_set(ROAR_ERROR_NOSPC);
  return -1;
 }

 if ( strstr(name, "/") != NULL ) {
  cont->handle[idx] = roar_dl_open(name, ROAR_DL_FLAG_DEFAULTS, 1, para);
  if ( cont->handle[idx] == NULL )
   return -1;

  cont->deprefc[idx] = 0;
  cont->numhandles++;

  return 0;
 } else {
  roar_err_set(ROAR_ERROR_NOSYS);
  return -1;
 }

 roar_err_set(ROAR_ERROR_NOSYS);
 return -1;
}

#if 0
int roar_plugincontainer_unload(struct roar_plugincontainer * cont, const char * name) {
 // get b name, then call roar_plugincontainer_unload_lhandle().
 roar_err_set(ROAR_ERROR_NOSYS);
 return -1;
}

int roar_plugincontainer_unload_lhandle(struct roar_plugincontainer * cont, struct roar_dl_lhandle * lhandle) {
 size_t i;

 if ( cont == NULL ) {
  roar_err_set(ROAR_ERROR_FAULT);
  return -1;
 }

 if ( lhandle == NULL ) {
  roar_err_set(ROAR_ERROR_INVAL);
  return -1;
 }

 for (i = 0; i < MAX_PLUGINS; i++) {
  if ( cont->handle[i] == lhandle ) {
   if ( cont->deprefc[i] ) {
    // still in use.
    roar_err_set(ROAR_ERROR_BUSY);
    return -1;
   }
   roar_dl_close(cont->handle[i]);
   cont->handle[i] = NULL;
   cont->numhandles--;
   return 0;
  }
 }

 roar_err_set(ROAR_ERROR_NOENT);
 return -1;
}
#endif

// appsched:
int roar_plugincontainer_appsched_trigger(struct roar_plugincontainer * cont, enum roar_dl_appsched_trigger trigger) {
 size_t i;

 if ( cont == NULL ) {
  roar_err_set(ROAR_ERROR_FAULT);
  return -1;
 }

 for (i = 0; i < MAX_PLUGINS; i++) {
  if ( cont->handle[i] == NULL )
   continue;

  roar_dl_appsched_trigger(cont->handle[i], trigger);
 }

 return 0;
}

//ll
