//memmgr.c:

/*
 *      Copyright (C) Philipp 'ph3-der-loewe' Schafft - 2010-2011
 *
 *  This file is part of libroar a part of RoarAudio,
 *  a cross-platform sound system for both, home and professional use.
 *  See README for details.
 *
 *  This file is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License version 3
 *  as published by the Free Software Foundation.
 *
 *  libroar is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this software; see the file COPYING.  If not, write to
 *  the Free Software Foundation, 51 Franklin Street, Fifth Floor,
 *  Boston, MA 02110-1301, USA.
 *
 *  NOTE for everyone want's to change something and send patches:
 *  read README and HACKING! There a addition information on
 *  the license of this document you need to read before you send
 *  any patches.
 *
 *  NOTE for uses of non-GPL (LGPL,...) software using libesd, libartsc
 *  or libpulse*:
 *  The libs libroaresd, libroararts and libroarpulse link this lib
 *  and are therefore GPL. Because of this it may be illigal to use
 *  them with any software that uses libesd, libartsc or libpulse*.
 */

#include "libroar.h"

#ifdef ROAR_USE_MEMMGR
#define NULL_BUFFER_CONST "RoarAudio"
static char _libroar_null_buffer[] = NULL_BUFFER_CONST; // A byte sequense >= 8 byte.

static inline void _libroar_null_buffer_check(void) {
 if ( memcmp(_libroar_null_buffer, NULL_BUFFER_CONST, sizeof(_libroar_null_buffer)) != 0 )
  roar_panic(ROAR_FATAL_ERROR_MEMORY_CORRUPTION_GUARD, NULL);
 memcpy(_libroar_null_buffer, NULL_BUFFER_CONST, sizeof(_libroar_null_buffer));
}
#endif

#define _err(x) do { roar_err_set((x)); return NULL; } while (0)

#ifdef ROAR_USE_MEMMGR

int     roar_mm_reset(void) {
 // currently this does nothing.
 // we need to free pools and such here later.
 return 0;
}

ssize_t roar_mm_sizeof(void * buf) {
 if ( buf == NULL ) {
  roar_err_set(ROAR_ERROR_FAULT);
  return -1;
 }

 if ( buf == _libroar_null_buffer_check )
  return 0;

 roar_err_set(ROAR_ERROR_NOTSUP);
 return -1;
}

void * roar_mm_calloc(size_t nmemb, size_t size) {
 void * ret;

 if ( nmemb == 0 || size == 0 ) {
  _libroar_null_buffer_check();
  return _libroar_null_buffer;
 }

#ifdef ROAR_HAVE_CALLOC
 roar_err_clear_all();
 ret = calloc(nmemb, size);

 if ( ret == NULL )
  roar_err_update();

 return ret;
#else
 ret = roar_mm_malloc(nmemb*size);
 if ( ret == NULL )
  return NULL;

 memset(ret, 0, nmemb*size);

 return ret;
#endif
}

void * roar_mm_malloc(size_t size) {
 void * ret;

 if ( size == 0 ) {
  _libroar_null_buffer_check();
  return _libroar_null_buffer;
 }

#ifdef ROAR_HAVE_MALLOC
 roar_err_clear_all();
 ret = malloc(size);

 if ( ret == NULL )
  roar_err_update();

 return ret;
#elif defined(ROAR_HAVE_CALLOC)
 return roar_mm_calloc(1, size);
#elif defined(ROAR_HAVE_REALLOC)
 return roar_mm_realloc(NULL, size);
#else
 _err(ROAR_ERROR_NOSYS);
#endif
}

int    roar_mm_free(void *ptr) {

 if ( ptr == _libroar_null_buffer ) {
  _libroar_null_buffer_check();
  return 0;
 }

#ifdef ROAR_HAVE_FREE
 free(ptr);
 return 0;
#elif defined(ROAR_HAVE_REALLOC)
 if ( roar_mm_realloc(ptr, 0) != _libroar_null_buffer )
  return -1;
 return 0;
#else
 roar_err_set(ROAR_ERROR_NOSYS);
 return -1;
#endif
}

void * roar_mm_realloc(void *ptr, size_t size) {
 void * ret;
#ifndef ROAR_HAVE_REALLOC
 ssize_t len;
#endif

 if ( ptr == NULL )
  return roar_mm_malloc(size);

 if ( size == 0 ) {
  roar_mm_free(ptr);
  _libroar_null_buffer_check();
  return _libroar_null_buffer;
 }

#ifdef ROAR_HAVE_REALLOC
 roar_err_clear_all();
 ret = realloc(ptr, size);

 if ( ret == NULL )
  roar_err_update();

 return ret;
#else
 len = roar_mm_sizeof(ptr);
 if ( len == -1 )
  return NULL;

 ret = roar_mm_malloc(size); 
 if ( ret == NULL )
  return NULL;

 memcpy(ret, ptr, len > (ssize_t)size ? size : len);

 return ret;
#endif
}
#endif

void * roar_mm_memdup(const void * s, size_t len) {
 void * ret;

 if ( s == NULL )
  _err(ROAR_ERROR_FAULT);

#ifdef ROAR_USE_MEMMGR
 if ( len == 0 ) {
  _libroar_null_buffer_check();
  return _libroar_null_buffer;
 }
#endif

 ret = roar_mm_malloc(len);

 if ( ret == NULL )
  return NULL;

 memcpy(ret, s, len);

 return ret;
}

void roar_mm_free_retvoid(void *ptr) {
 roar_mm_free(ptr);
}

#if defined(ROAR_USE_MEMMGR) || !defined(ROAR_HAVE_STRDUP)
char * roar_mm_strdup(const char *s) {
 void * ret;
#ifndef ROAR_HAVE_STRDUP
 ssize_t len;
#endif

 if ( s == NULL )
  _err(ROAR_ERROR_FAULT);

#ifdef ROAR_HAVE_STRDUP
 roar_err_clear_all();
 ret = strdup(s);

 if ( ret == NULL )
  roar_err_update();
#else
 len = roar_mm_strlen(s);
 if ( len == -1 )
  return NULL;

 ret = roar_mm_memdup(s, len+1);
#endif

 return ret;
}
#endif

#if defined(ROAR_USE_MEMMGR) || !defined(ROAR_HAVE_STRNDUP)
char *roar_mm_strndup(const char *s, size_t n) {
 void * ret;
#ifndef ROAR_HAVE_STRNDUP
 ssize_t len;
#endif

 if ( s == NULL )
  _err(ROAR_ERROR_FAULT);

#ifdef ROAR_USE_MEMMGR
 if ( n == 0 ) {
  _libroar_null_buffer_check();
  return _libroar_null_buffer;
 }
#endif

#ifdef ROAR_HAVE_STRNDUP
 roar_err_clear_all();
 ret = strndup(s, n);

 if ( ret == NULL )
  roar_err_update();
#else
 len = roar_mm_strnlen(s, n);
 if ( len == -1 )
  return NULL;

 ret = roar_mm_memdup(s, len+1);
#endif

 return ret;
}
#endif

#ifndef ROAR_HAVE_STRLEN
ssize_t roar_mm_strlen(const char *s) {
 size_t ret = 0;

 if ( s == NULL ) {
  roar_err_set(ROAR_ERROR_FAULT);
  return -1;
 }

 for (; *s != 0; s++, ret++);

 return ret;
}
#endif

ssize_t roar_mm_strnlen(const char *s, size_t maxlen) {
 size_t ret = 0;

 if ( s == NULL ) {
  roar_err_set(ROAR_ERROR_FAULT);
  return -1;
 }

#ifdef ROAR_HAVE_STRNLEN
 ret = strnlen(s, maxlen);
#else
 for (; ret < maxlen && *s != 0; s++, ret++);
#endif

 if ( ret == maxlen ) {
  roar_err_set(ROAR_ERROR_MSGSIZE);
  return -1;
 }

 return ret;
}


#ifndef ROAR_HAVE_STRLCPY
ssize_t roar_mm_strlcpy(char *dst, const char *src, size_t size) {
 ssize_t len;

 if ( dst == NULL || src == NULL ) {
  roar_err_set(ROAR_ERROR_FAULT);
  return -1;
 }

 if ( size == 0 )
  return 0;

 len = roar_mm_strlen(src);
 if ( len == -1 )
  return -1;

 if ( len < (ssize_t)size ) {
  memcpy(dst, src, len);
  dst[len] = 0;
 } else {
  memcpy(dst, src, size - 1);
  dst[size-1] = 0;
 }

 return len;
}
#endif

#ifndef ROAR_HAVE_STRLCAT
ssize_t roar_mm_strlcat(char *dst, const char *src, size_t size) {
 ssize_t slen;
 ssize_t dlen;
 int err;

 if ( dst == NULL || src == NULL ) {
  roar_err_set(ROAR_ERROR_FAULT);
  return -1;
 }

 if ( size == 0 )
  return 0;

 slen = roar_mm_strlen(src);
 if ( slen == -1 )
  return -1;

 dlen = roar_mm_strnlen(dst, size);
 err  = roar_error;
 if ( dlen == -1 && err == ROAR_ERROR_MSGSIZE )
  dlen = size;

 if ( dlen == -1 )
  return -1;

 if ( (dlen + slen) < (ssize_t)size ) {
  memcpy(dst+dlen, src, slen);
  dst[dlen+slen] = 0;
 } else if ( dlen != size ) {
  memcpy(dst+dlen, src, size-dlen-1);
  dst[size-1] = 0;
 }

 return dlen+slen;
}
#endif

#ifndef ROAR_HAVE_STRTOK_R
char * roar_mm_strtok_r(char *str, const char *delim, char **saveptr) {
 char * next;

 if ( delim == NULL || saveptr == NULL ) {
  roar_err_set(ROAR_ERROR_FAULT);
  return NULL;
 }

 if ( delim[0] == 0 ) {
  roar_err_set(ROAR_ERROR_NOTSUP);
  return NULL;
 }

 if ( delim[1] != 0 ) {
  roar_err_set(ROAR_ERROR_NOTSUP);
  return NULL;
 }

 if ( str != NULL ) {
  next = str;
 } else {
  next = *saveptr;
 }

 if ( next == NULL ) {
  roar_err_set(ROAR_ERROR_NODATA);
  return NULL;
 }

 *saveptr = strstr(next, delim);

 if ( *saveptr != NULL ) {
  **saveptr = 0;
  (*saveptr)++;
 }

 return next;
}
#endif

//ll
