#' Save or restore RKWard workplace
#' 
#' \code{rk.save.workplace} can be used to save a representation of the RKWard
#' workplace (i.e. which scripts, data edtiors and other windows are shown) to
#' a file. \code{rk.restore.workplace} restores an RKWard workplace as saved by
#' \code{rk.save.workplace}.
#' 
#' If the \code{file} parameter is omitted (or \code{NULL}), a suitable
#' filename is selected automatically. If a workspace has been loaded, this is
#' the URL of the workspace with an appended \code{.rkworkplace}. Otherwise a
#' filename in the RKWard directory, as generated by
#' \link{rk.get.tempfile.name}.
#' 
#' NOTE: Not all types of windows can be saved and restored. Esp. graphics
#' device windows will not be restored (but WILL be closed by
#' \code{rk.restore.workplace()}, if \code{close.windows} is TRUE).
#' 
#' @aliases rk.save.workplace rk.restore.workplace
#' @param file a character string giving the url of the file to save to, or
#'   NULL for automatic selection of a suitable file (see Details).
#' @param description For internal use, only. A character string describing the
#'   workplace status to save. Generally, you should leave this as the default
#'   value (\code{NULL}).
#' @param close.windows a logical; whether current windows should be closed
#'   before restoring.
#' @return Both functions return \code{NULL}.
#' @author Thomas Friedrichsmeier \email{rkward-devel@@lists.sourceforge.net}
#' @seealso \url{rkward://page/rkward_for_r_users}, \link{rk.get.workspace.url}
#' @keywords utilities
#' @rdname rk.workplace
#' @examples
#' 
#' ## Not run
#' rk.save.workplace ()
#' rk.restore.workplace ()
#' ## End not run
#' 
"rk.save.workplace" <- function (file=NULL, description=NULL) {
	if (is.null (file)) {
		file <- rk.get.workspace.url ()
		if (is.null (file)) file <- rk.get.tempfile.name (prefix="unsaved", extension=".RData")
		file <- paste (file, "rkworkplace", sep=".")
	}
	if (is.null (description)) lines <- .rk.do.plain.call ("workplace.layout", "get")
	else lines <- description
	writeLines (lines, file)
}

"rk.restore.workplace" <- function (file=NULL, close.windows=TRUE) {
	if (is.null (file)) {
		if (exists (".rk.workplace.save", envir=globalenv (), inherits=FALSE)) {
			# For backwards compatibility with workspaces saved by RKWard 0.5.4 and earlier.
			# TODO: remove in time.
			lines <- as.character (.GlobalEnv$.rk.workplace.save)
			rm (list = c (".rk.workplace.save"), envir=globalenv ())
		} else {
			file <- rk.get.workspace.url ()
			if (is.null (file)) file <- rk.get.tempfile.name (prefix="unsaved", extension=".RData")
			file <- paste (file, "rkworkplace", sep=".")
		}
	}

	close <- "close"
	if (!isTRUE (close.windows)) close <- "noclose"
	if (!exists ("lines", inherits=FALSE)) lines <- readLines (file)
	.rk.do.plain.call ("workplace.layout", c ("set", close, lines), synchronous=FALSE)
	invisible (NULL)
}
