/*
 * refdbg_priv.h - Private refdbg header file
 *
 * refdbg - GObject refcount debugger
 * Copyright (C) 2004-2005 Josh Green <jgreen@users.sourceforge.net>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA
 * 02111-1307, USA or point your web browser to http://www.gnu.org.
 */
#ifndef __REFDBG_PRIV_H__
#define __REFDBG_PRIV_H__

/**************************************************************************
 * !! NOTICE !! The following declarations are private and should not
 * be modified or used by the user directly. Use the functions defined
 * in refdbg.h to interface to refdbg.
 **************************************************************************/

/* just to shorten things */
#define casecmp g_ascii_strcasecmp

#include <stdio.h>
#include <glib.h>
#include <glib-object.h>

/* event types */
typedef enum
{
  EVENT_TYPE_PRE_NEW,          /* new object request (pre actual creation) */
  EVENT_TYPE_NEW,              /* new object finished being constructed */
  EVENT_TYPE_REF,              /* object ref */
  EVENT_TYPE_UNREF,            /* object unref */
  EVENT_TYPE_PRE_FINALIZE,     /* pre object finalize (last unref) */
  EVENT_TYPE_FINALIZE          /* object finalize */
} EventType;

typedef enum
{
  EVENT_ERR_NONE,              /* no error */
  EVENT_ERR_UNKNOWN_OBJECT,    /* ref/unref on an unknown object */
  EVENT_ERR_DESTROYED_OBJECT,  /* ref/unref of a destroyed object */
  EVENT_ERR_NOT_OBJECT,        /* ref/unref on non object */
  EVENT_ERR_INIT_REFCOUNT,     /* unexpected initial refcount (new object) */
  EVENT_ERR_BAD_REFCOUNT,      /* unexpected refcount on ref/unref */
} EventErrType;

/* event rule flags */
typedef enum
{
  EVENT_RULE_FLAG_PRE_NEW                = 1 << 0,
  EVENT_RULE_FLAG_NEW                    = 1 << 1,
  EVENT_RULE_FLAG_REF                    = 1 << 2,
  EVENT_RULE_FLAG_UNREF                  = 1 << 3,
  EVENT_RULE_FLAG_PRE_FINALIZE           = 1 << 4,
  EVENT_RULE_FLAG_FINALIZE               = 1 << 5,

  /* update EVENT_RULE_FLAG_ERROR_SHIFT below if the first error shift
     offset is changed */

  EVENT_RULE_FLAG_ERR_UNKNOWN_OBJECT     = 1 << 6,
  EVENT_RULE_FLAG_ERR_DESTROYED_OBJECT   = 1 << 7,
  EVENT_RULE_FLAG_ERR_NOT_OBJECT         = 1 << 8,
  EVENT_RULE_FLAG_ERR_INIT_REFCOUNT      = 1 << 9,
  EVENT_RULE_FLAG_ERR_BAD_REFCOUNT       = 1 << 10,

  /* set this to immediately display/break on unknown object errors
     during object construction */
  EVENT_RULE_FLAG_PARANOID               = 1 << 11,

  /* only used for display masks, enables refcount timer for object's
     matching the rule criteria, also entails NEW, REF and UNREF
     log events (required for timer operation) */
  EVENT_RULE_FLAG_TIMER                  = 1 << 12
} EventRuleFlags;

#define EVENT_RULE_FLAG_ERROR_SHIFT 5 /* -1 since EVENT_ERR_NONE not used */
#define EVENT_RULE_FLAG_COUNT  13     /* flag count */

#define EVENT_RULE_MASK_ALL    0x7FF /* all events */
#define EVENT_RULE_MASK_NONE   0x000 /* no events */
#define EVENT_RULE_MASK_EVENT  0x03F /* normal events (non errors) */
#define EVENT_RULE_MASK_ERROR  0x3C0 /* error events */

/* event rules define criteria that if met will cause action(s)
   (like display event, execute break point, and/or log it) */
typedef struct
{
  GType inc_type;          /* include GType or 0 for any */
  GType exc_type;          /* exclude GType or 0 for don't exclude */
  char *inc_type_name;     /* set to inc_type name when not registered yet */
  char *exc_type_name;     /* set to exc_type name when not registered yet */
  gpointer inc_object;     /* include object or NULL for any */
  gpointer exc_object;     /* exclude object or NULL for don't exclude */
  guint display_mask;      /* mask of events to display (RuleEventFlags) */
  guint break_mask;        /* mask of events to break on (RuleEventFlags) */
  guint log_mask;          /* mask of events to log (RuleEventFlags) */
} EventRule;

typedef struct
{
  guint32 type      :  3;  /* EventType */
  guint32 error     :  3;  /* EventErrType */
  guint32 reserved  : 26;
  guint32 timestamp;       /* time elapsed since last event (microseconds) */
  guint32 refcount;        /* current object refcount */
  GType obj_type;          /* object type or 0 if can't be resolved */
  gpointer object;         /* the object (!! should be last before backtrace !!) */

  /* followed by backtrace_count gpointers of back trace addresses */
} RefEvent;

/* pointer to first address in backtrace */
#define REF_EVENT_BACKTRACE(event)  (&(((RefEvent *)(event))->object) + 1)

/* total size of ref event including max backtrace size */
#define REF_EVENT_MAX_SIZE \
  (sizeof (RefEvent) + REFDBG_MAX_BACKTRACE_COUNT * sizeof (gpointer))

/* size of an reference event including backtrace */
#define REF_EVENT_SIZE \
  (sizeof (RefEvent) + backtrace_count * sizeof (gpointer))

#define REF_ARRAY_INDEX(index) \
  ((RefEvent *)(event_array->data + ((index) * REF_EVENT_SIZE)))

/* a structure used to store data for display and object command rules */
typedef struct
{
  GType *inc_types;      /* 0 terminated array of include GTypes or NULL if none */
  GType *exc_types;      /* 0 terminated array of exclude GTypes or NULL if none */
  char *inc_type_name;   /* single unresolved include GType string */
  char *exc_type_name;   /* single unresolved exclude GType string */
  gpointer *inc_objects; /* NULL terminated array of include objects or NULL */
  gpointer *exc_objects; /* NULL terminated array of exclude objects or NULL */
  guint display_mask;    /* display flag mask */
  guint break_mask;      /* break flag mask */
  guint log_mask;        /* log flag mask */
  int btnum;             /* max back trace or -1 */
  guint32 start_time;    /* start time in microseconds (0=start) */
  guint32 end_time;      /* end time in microseconds (0=end) */
  int limit_max;         /* max result limit */
  int limit_ofs;         /* result offset (negative = ofs from end) */
} DetailRule;


/* flags for 'object' command display mask */
typedef enum
{
  OBJECT_RULE_FLAG_ACTIVE    = 1 << 0,
  OBJECT_RULE_FLAG_DESTROYED = 1 << 1
} ObjectRuleFlags;

#define OBJECT_RULE_MASK_ALL 0x3


/* object info (values in obj_hash) */
typedef struct
{
  GType type;            /* object type */
  guint refcount;        /* refcount (0 = finalized - g_type_free_instance) */
} ObjInfo;

/* address info for backtrace address symbols (sym_hash) */
typedef struct
{
  const char *obj_fname;       /* name of the object file or NULL */
  gpointer obj_addr;           /* base address of object or NULL */

  const char *func;            /* function name */
  gpointer func_addr;          /* address of function */

  const char *src_fname;       /* name of source file */
  guint32 src_line;            /* source line number */
} AddrInfo;


/* REFDBG mutex locking macros */

#ifdef G_THREADS_ENABLED
#define REFDBG_LOCK    g_static_rec_mutex_lock (&refdbg_mutex)
#define REFDBG_UNLOCK  g_static_rec_mutex_unlock (&refdbg_mutex)
#define REFDBG_TRYLOCK g_static_rec_mutex_trylock (&refdbg_mutex)
#else
#warning "Glib has no thread support!"
#define REFDBG_LOCK    G_STMT_START{ (void)0; }G_STMT_END
#define REFDBG_UNLOCK  G_STMT_START{ (void)0; }G_STMT_END
#define REFDBG_TRYLOCK (TRUE)
#endif

/* -- refdbg.c -- */

extern gboolean refdbg_active;
extern int backtrace_count;
extern gboolean refdbg_enable_timer;
extern gboolean refdbg_enable_bt_paths;

extern GStaticRecMutex refdbg_mutex;
extern EventRule event_rules[];
extern int event_rule_count;
extern GArray *event_array;
extern GHashTable *obj_hash;
extern guint construct_count;
extern guint construct_index;
extern struct timeval start_time;
extern guint32 timer_expire;
extern gboolean save_event_log;
extern gboolean log_objects;
extern gboolean display_object_stats;

void refdbg_clear (void);
guint32 refdbg_get_timestamp (void);

/* So we can start using the glib 2.10 allocator now */
#ifndef g_slice_new
#define g_slice_new(type)  g_new (type, 1)
#define g_slice_new0(type) g_new0 (type, 1)
#define g_slice_free(type, mem)  g_free (mem)
#endif

/* shell.c */
extern gboolean refdbg_exit;

/* cmdparse.c */
gboolean refdbg_exec (const char *cmds, gboolean execute);
void init_detail_rule (DetailRule *rule);
void release_detail_rule (DetailRule *rule);

/* user.c */
extern guint refdbg_dispmax;
extern gchar *refdbg_logname;
void refdbg_real_stats (gboolean noshow_empty);
int refdbg_display_objects (DetailRule *rule, FILE *file);
void refdbg_real_save_log (gboolean ifempty);
const gchar *refdbg_get_event_type_name (RefEvent *event);
const gchar *refdbg_format_time_stamp (guint32 timestamp);
void refdbg_display_backtrace (void **trace, int bt_num, FILE * file);
void refdbg_display_events (DetailRule *rule, FILE *file);

/* symsnarf.c */
gboolean refdbg_sym_snarf_init (void);
AddrInfo *refdbg_sym_snarf (void *addr);
guint refdbg_sym_hash_size (void);

#endif
