 /*
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include <config.h>
#include <string.h>
#include <stdlib.h>

#include <gtk/gtk.h>
#include <gdk/gdkkeysyms.h>
#include <panel-applet.h>
#include <gconf/gconf-client.h>
#include <libgnomeui/gnome-window-icon.h>
#include "rcalc_applet.h"
#include "preferences.h"
#include "command_line.h"
#include "about.h"
#include "help.h"

#include <engine/engine.h>
#include <engine/debug.h>

RcEngine *RC_engine;

static const BonoboUIVerb mini_commander_menu_verbs [] = {
        BONOBO_UI_UNSAFE_VERB ("Props", rca_show_preferences),
        BONOBO_UI_UNSAFE_VERB ("Help",  show_help),
        BONOBO_UI_UNSAFE_VERB ("About", about_box),

        BONOBO_UI_VERB_END
};

void
set_atk_name_description (GtkWidget  *widget,
			  const char *name,
			  const char *description)
{	
    AtkObject *aobj;
	
    aobj = gtk_widget_get_accessible (widget);
    if (GTK_IS_ACCESSIBLE (aobj) == FALSE)
        return;

    atk_object_set_name (aobj, name);
    atk_object_set_description (aobj, description);
}


/* Send button presses on the applet to the entry. This makes Fitts' law work (ie click on the bottom
** most pixel and the key press will be sent to the entry */
static gboolean
send_button_to_entry_event (GtkWidget *widget, GdkEventButton *event, MCData *mc)
{

	if (event->button == 1) {
		gtk_widget_grab_focus (mc->entry);
		return TRUE;
	}
	return FALSE;

}

void
rca_applet_draw (MCData *mc)
{
    MCPreferences prefs = mc->preferences;
    int        size_frames = 0;
    gchar     *command_text = NULL;

    if (mc->entry != NULL)
	command_text = g_strdup (gtk_editable_get_chars (GTK_EDITABLE (mc->entry), 0, -1));

    mc->cmd_line_size_y = mc->preferences.normal_size_y - size_frames;   

    if (mc->applet_box) {
        gtk_widget_destroy (mc->applet_box);	
    }

    if ( ((mc->orient == PANEL_APPLET_ORIENT_LEFT) || (mc->orient == PANEL_APPLET_ORIENT_RIGHT)) && (prefs.panel_size_x < GNOME_Vertigo_PANEL_SMALL) )
      mc->applet_box = gtk_vbox_new (FALSE, 0);
    else
      mc->applet_box = gtk_hbox_new (FALSE, 0);

    gtk_container_set_border_width (GTK_CONTAINER (mc->applet_box), 0);

    rca_create_command_entry (mc);

    if (command_text != NULL) {
	gtk_entry_set_text (GTK_ENTRY (mc->entry), command_text);
	g_free (command_text);
    }
 
    gtk_box_pack_start (GTK_BOX (mc->applet_box), mc->comboboxentry, TRUE, TRUE, 0);

    gtk_container_add (GTK_CONTAINER (mc->applet), mc->applet_box);
    
    gtk_widget_show_all (mc->applet_box);
}

static void
rca_destroyed (GtkWidget *widget,
	      MCData    *mc)
{
    GConfClient *client;

    client = gconf_client_get_default ();

    g_object_unref (client);

    g_object_unref (mc->tooltips);	

    if (mc->about_dialog)
        gtk_widget_destroy (mc->about_dialog);

    g_free (mc);
}

static void
rca_orient_changed (PanelApplet *applet,
		   PanelAppletOrient orient,
		   MCData *mc)
{
  mc->orient = orient;
  rca_applet_draw (mc);
}

static void
rca_pixel_size_changed (PanelApplet *applet,
		       GtkAllocation *allocation,
		       MCData      *mc)
{
  if ((mc->orient == PANEL_APPLET_ORIENT_LEFT) || (mc->orient == PANEL_APPLET_ORIENT_RIGHT)) {
    if (mc->preferences.panel_size_x == allocation->width)
      return;
    mc->preferences.panel_size_x = allocation->width;
  } else {
    if (mc->preferences.normal_size_y == allocation->height)
      return;
    mc->preferences.normal_size_y = allocation->height;
  }

  rca_applet_draw (mc);
}

static gboolean
mini_commander_applet_fill (PanelApplet *applet)
{
    MCData *mc;
    GConfClient *client;
    GError *error;
    gchar *unit = NULL;

    client = gconf_client_get_default ();
    
    gnome_window_icon_set_default_from_file (GNOME_ICONDIR "/gnome-calc3.png");
    
    mc = g_new0 (MCData, 1);
    mc->applet = applet;

    panel_applet_add_preferences (applet, "/schemas/apps/rcalc-applet/prefs", NULL);
    panel_applet_set_flags (applet, PANEL_APPLET_EXPAND_MINOR);
    rca_load_preferences (mc);

    mc->about_dialog = NULL;

    mc->tooltips = gtk_tooltips_new ();
    g_object_ref (mc->tooltips);
    gtk_object_sink (GTK_OBJECT (mc->tooltips));
  
    g_signal_connect (mc->applet, "change_orient",
		      G_CALLBACK (rca_orient_changed), mc);
    g_signal_connect (mc->applet, "size_allocate",
		      G_CALLBACK (rca_pixel_size_changed), mc);
    
    mc->preferences.normal_size_y = panel_applet_get_size (applet);
    mc->orient = panel_applet_get_orient (applet);
    rca_applet_draw(mc);
    gtk_widget_show (GTK_WIDGET (mc->applet));
    
    g_signal_connect (mc->applet, "destroy", G_CALLBACK (rca_destroyed), mc); 
    g_signal_connect (mc->applet, "button_press_event",
		      G_CALLBACK (send_button_to_entry_event), mc);

    panel_applet_setup_menu_from_file (mc->applet,
				       NULL,
				       "GNOME_rcalcApplet.xml",
				       NULL,
				       mini_commander_menu_verbs,
				       mc);

    if (panel_applet_get_locked_down (mc->applet)) {
	    BonoboUIComponent *popup_component;

	    popup_component = panel_applet_get_popup_component (mc->applet);

	    bonobo_ui_component_set_prop (popup_component,
					  "/commands/Props",
					  "hidden", "1",
					  NULL);
    }

    set_atk_name_description (GTK_WIDGET (applet),
			      _("rcalc applet"),
			      _("This applet adds a rcalc command line to the panel"));
    
      
  unit = g_strdup_printf("%s", 
    panel_applet_gconf_get_string (PANEL_APPLET (applet),"angle_unit", &error) );
       
  if (!unit)
    unit = g_strdup("rad");
  else {
    if (g_ascii_strcasecmp(unit, "rad") && g_ascii_strcasecmp(unit, "deg")) {
      g_free(unit);
      unit = g_strdup("rad");
    }
  }
   
  RC_engine = rc_engine_new();
  rc_engine_set_streams( RC_engine, NULL, NULL );
	
  rc_engine_load_state( RC_engine );
  
  if (!g_ascii_strcasecmp(unit, "rad")) {
    rc_engine_set_angleunit( RC_engine, RC_ANGLE_RADIANS );
    mc->preferences.angle_unit = RC_ANGLE_RADIANS;
  }
  else {
    rc_engine_set_angleunit( RC_engine, RC_ANGLE_DEGREES );
    mc->preferences.angle_unit = RC_ANGLE_DEGREES;
  }
     

  return TRUE;
}

static gboolean
mini_commander_applet_factory (PanelApplet *applet,
			       const gchar *iid,
			       gpointer     data)
{
        gboolean retval = FALSE;

        if (!strcmp (iid, "OAFIID:GNOME_rcalcApplet"))
                retval = mini_commander_applet_fill(applet); 
    
        return retval;
}

PANEL_APPLET_BONOBO_FACTORY ("OAFIID:GNOME_rcalcApplet_Factory",
			     PANEL_TYPE_APPLET,
                             "rcalc-applet",
                             "0",
                             mini_commander_applet_factory,
                             NULL)
