/*
 * RageIRCd: an advanced Internet Relay Chat daemon (ircd).
 * (C) 2000-2005 the RageIRCd Development Team, all rights reserved.
 *
 * This software is free, licensed under the General Public License.
 * Please refer to doc/LICENSE and doc/README for further details.
 *
 * $Id: mkpasswd.c,v 1.15.2.2 2005/02/21 02:32:46 amcwilliam Exp $
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <time.h>
#include <unistd.h>
#ifndef SINGLE_BINARY
#include "setup.h"
#include "ssl.h"
#endif

extern char *crypt(const char *, const char *);
extern char *getpass(const char *);

#ifdef USE_OPENSSL
static int base64_encode(const char *, size_t, char *, size_t);

static const char base64_str[] = "ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz0123456789+/";
static const char base64_pad = '=';
#endif

static char enc_chars[] = "abcdefghijklmnopqrstuvwxyzABCDEFGHIJKLMNOPQRSTUVWXYZ0123456789./";

#define FLAG_RIPEMD_160		0x0001
#define FLAG_SHA1_160		0x0002
#define FLAG_MD5_128		0x0004
#define FLAG_MD5_56		0x0008
#define FLAG_DES_56		0x0010
#define FLAG_SALT		0x0020
#define FLAG_PASSWORD		0x0040

static char *usage[] = {
	" ",
	"Password Encryption Tool for RageIRCd v2.0.",
	" ",
	"RageIRCd: an advanced Internet Relay Chat daemon (ircd).",
	"(C) 2000-2005 the RageIRCd Development Team, all rights reserved.",
	" ",
	"Usage: ragemkpasswd [flags] [-s salt] [-p plaintext]",
	" ",
	"Where [flags] can be either of the following:",
	" ",
	"    -d     des/56: encrypt with DES hashing algorithm (56 bit)",
	"    -c     md5/56: encrypt with MD5-based hashing algorith (56 bit)",
	" ",
	"    The following are only available with --enable-openssl:",
	" ",
	"    -r     ripemd/160: encrypt with RIPEMD algorith (160 bit)",
	"    -a     sha1/160:   encrypt with SHA1 algorith (160 bit)",
	"    -m     md5/128:    encrypt with MD5 algirhtm (128 bit)",
	" ",
	"    Only the -c (MD5-based hashing) and -d (DES hashing) algorithms",
	"    use encryption salts, and therefore others will not be affected",
	"    by the -s option.",
	" ",
	"Other optional parameters:",
	"    -s <salt>          specifies salt to use instead of random bytes.",
	"                       (salt must be 8 bytes or less for -c, and 2",
	"                       bytes for -d)",
	"    -p <plaintext>     specifies plaintext to encrypt",
	" ",
	NULL
};

static void show_usage()
{
	char **text;
	for (text = usage; *text; text++) {
		fprintf(stderr, "%s\n", *text);
	}
	return;
}

static char *make_salt(int md5, char *para)
{
	static char salt[12];
	int i = 0;

	if (md5 == 1) {
		salt[0] = '$';
		salt[1] = '1';
		salt[2] = '$';
		if (para != NULL) {
			if (strlen(para) > 8) {
				fprintf(stderr, "WARNING: specified salt greater than 8 bytes, truncated.\n");
				para[8] = '\0';
			}
			for (i = 0; i < 8 && para[i] != '\0'; i++) {
				salt[i + 3] = para[i];
			}
		}
		else {
			for (i = 0; i < 8; i++) {
				salt[i + 3] = enc_chars[random() % 64];
			}
		}
		salt[i + 3] = '\0';
	}
	else {
		if (para != NULL) {
			if (strlen(para) != 2) {
				fprintf(stderr, "WARNING: specified salt does not equal 2 bytes, using random.\n");
			}
			else {
				for (i = 0; i < 2; i++) {
					salt[i] = para[i];
				}
			}
		}
		if (para == NULL || i == 0) {
			for (i = 0; i < 2; i++) {
				salt[i] = enc_chars[random() % 64];
			}
		}
		salt[i] = '\0';
	}
	return salt;
}

#ifdef USE_OPENSSL
static int base64_encode(const char *src, size_t srclen, char *target, size_t targetlen)
{
        u_char input[3], output[4];
        size_t datalen = 0;
        int i;

        while (srclen > 2) {
                for (i = 0; i < 3; i++) {
                        input[i] = *src++;
                        srclen--;
                }
                output[0] = input[0] >> 2;
                output[1] = ((input[0] & 0x03) << 4) + (input[1] >> 4);
                output[2] = ((input[1] & 0x0f) << 2) + (input[2] >> 6);
                output[3] = input[2] & 0x3f;

                if (datalen + 4 > targetlen) {
                        return -1;
                }
                for (i = 0; i < 4; i++) {
                        target[datalen++] = base64_str[output[i]];
                }
        }
        if (srclen != 0) {
                input[0] = input[1] = input[2] = '\0';
                for (i = 0; i < srclen; i++) {
                        input[i] = *src++;
                }
                output[0] = input[0] >> 2;
                output[1] = ((input[0] & 0x03) << 4) + (input[1] >> 4);
                output[2] = ((input[1] & 0x0f) << 2) + (input[2] >> 6);

                if (datalen + 4 > targetlen) {
                        return -1;
                }
                for (i = 0; i < 3; i++) {
                        target[datalen++] = (i == 2 && srclen == 1) ? base64_pad : base64_str[output[i]];
                }
                target[datalen++] = base64_pad;
        }
        if (datalen >= targetlen) {
                return -1;
        }
        target[datalen] = '\0';
        return datalen;
}
#endif

int main(int argc, char *argv[])
{
	extern char *optarg;
	char *salt = NULL, *plaintext = NULL;
	unsigned int flag = 0;
	int c = -1;
#ifdef USE_OPENSSL
	static char buf[512];
	int i = 0;
#endif

	srandom(time(NULL));
	while ((c = getopt(argc, argv, "cdarmh?s:p:")) != -1) {
		switch (c) {
#ifdef USE_OPENSSL
			case 'r':
				flag |= FLAG_RIPEMD_160;
				break;
			case 'a':
				flag |= FLAG_SHA1_160;
				break;
			case 'm':
				flag |= FLAG_MD5_128;
				break;
#endif
			case 'c':
				flag |= FLAG_MD5_56;
				break;
			case 'd':
				flag |= FLAG_DES_56;
				break;
			case 's':
				flag |= FLAG_SALT;
				salt = optarg;
				break;
			case 'p':
				flag |= FLAG_PASSWORD;
				plaintext = optarg;
				break;
			case 'h':
			case '?':
				show_usage();
				return 0;
			default:
				show_usage();
				printf("-%c is an invalid option\n", c);
				return 0;
		}
	}
	if (!(flag & FLAG_PASSWORD) || (flag & FLAG_PASSWORD && plaintext == NULL)) {
		plaintext = getpass("plaintext: ");
	}
#ifdef USE_OPENSSL
	if (flag & FLAG_RIPEMD_160) {
		if ((i = base64_encode(RIPEMD160(plaintext, strlen(plaintext), NULL),
		  RIPEMD160_DIGEST_LENGTH, buf, 512)) > 0) {
			fprintf(stderr, "%s\n", buf);
		}
		else {
			fprintf(stderr, "Error during base64 encode (i%d)\n", i);
		}
	}
	else if (flag & FLAG_SHA1_160) {
		if ((i = base64_encode(SHA1(plaintext, strlen(plaintext), NULL),
		  SHA_DIGEST_LENGTH, buf, 512)) > 0) {
			fprintf(stderr, "%s\n", buf);
		}
		else {
			fprintf(stderr, "Error during base64 encode (i%d)\n", i);
		}
	}
	else if (flag & FLAG_MD5_128) {
		if ((i = base64_encode(MD5(plaintext, strlen(plaintext), NULL),
		  MD5_DIGEST_LENGTH, buf, 512)) > 0) {
			fprintf(stderr, "%s\n", buf);
		}
		else {
			fprintf(stderr, "Error during base64 encode (i%d)\n", i);
		}
	}
	else if (flag & FLAG_MD5_56) {
#else
	if (flag & FLAG_MD5_56) {
#endif
		fprintf(stderr, "%s\n", crypt(plaintext, make_salt(1, salt)));
	}
	else {
		fprintf(stderr, "%s\n", crypt(plaintext, make_salt(0, salt)));
	}
	return 0;
}
